// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
//
// Implementation of the MalwareDetailsRedirectsCollector class.

#include "chrome/browser/safe_browsing/malware_details_history.h"

#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/safe_browsing/malware_details.h"
#include "content/browser/browser_thread.h"
#include "content/browser/renderer_host/render_view_host.h"
#include "content/browser/tab_contents/navigation_entry.h"
#include "content/browser/tab_contents/tab_contents.h"
#include "content/common/notification_details.h"
#include "content/common/notification_service.h"
#include "content/common/notification_source.h"
#include "content/common/notification_type.h"

MalwareDetailsRedirectsCollector::MalwareDetailsRedirectsCollector(
    Profile* profile)
    : profile_(profile),
      has_started_(false) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  if (profile) {
    registrar_.Add(this, NotificationType::PROFILE_DESTROYED,
                   Source<Profile>(profile));
  }
}

MalwareDetailsRedirectsCollector::~MalwareDetailsRedirectsCollector() {
}

void MalwareDetailsRedirectsCollector::StartHistoryCollection(
    const std::vector<GURL>& urls,
    Task* callback) {
  DVLOG(1) << "Num of urls to check in history service: " << urls.size();
  has_started_ = true;
  callback_ = callback;

  if (urls.size() == 0) {
    AllDone();
    return;
  }

  BrowserThread::PostTask(
      BrowserThread::UI, FROM_HERE,
      NewRunnableMethod(
          this, &MalwareDetailsRedirectsCollector::StartGetRedirects, urls));
}

void MalwareDetailsRedirectsCollector::StartGetRedirects(
        const std::vector<GURL>& urls) {
  // History access from profile needs to happen in UI thread
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  for (size_t i = 0; i < urls.size(); ++i) {
    urls_.push_back(urls[i]);
  }
  urls_it_ = urls_.begin();
  GetRedirects(*urls_it_);
}

void MalwareDetailsRedirectsCollector::GetRedirects(const GURL& url) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  if (!profile_) {
    AllDone();
    return;
  }

  HistoryService* history = profile_->GetHistoryService(
      Profile::EXPLICIT_ACCESS);
  if (!history) {
    AllDone();
    return;
  }

  history->QueryRedirectsTo(
      url,
      &request_consumer_,
      NewCallback(this,
                  &MalwareDetailsRedirectsCollector::OnGotQueryRedirectsTo));
}

void MalwareDetailsRedirectsCollector::OnGotQueryRedirectsTo(
    HistoryService::Handle handle,
    GURL url,
    bool success,
    history::RedirectList* redirect_list) {

  if (success && redirect_list->size() > 0) {
    std::vector<GURL> urllist;
    urllist.push_back(url);
    for (size_t i = 0; i < redirect_list->size(); i++) {
      urllist.push_back(redirect_list->at(i));
    }
    redirects_urls_.push_back(urllist);
  }

  // Proceed to next url
  ++urls_it_;

  if (urls_it_ == urls_.end()) {
    AllDone();
    return;
  }

  GetRedirects(*urls_it_);
}

bool MalwareDetailsRedirectsCollector::HasStarted() const {
  return has_started_;
}

void MalwareDetailsRedirectsCollector::AllDone() {
  DVLOG(1) << "AllDone";
  BrowserThread::PostTask(BrowserThread::IO, FROM_HERE, callback_);
}

const std::vector<safe_browsing::RedirectChain>&
MalwareDetailsRedirectsCollector::GetCollectedUrls() const {
  return redirects_urls_;
}

void MalwareDetailsRedirectsCollector::Observe(
    NotificationType type,
    const NotificationSource& source,
    const NotificationDetails& details) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK_EQ(type.value, NotificationType::PROFILE_DESTROYED);
  DVLOG(1) << "Profile gone.";
  profile_ = NULL;
}
