/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _BSD_SOURCE /* strdup() */

#include <string.h>
#include <stdlib.h>
#include <stdbool.h>

#include <ncursesw/ncurses.h>

#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_main_window.h"
#include "cdw_config.h"
#include "cdw_string.h"
#include "cdw_erase_wizard.h"
#include "cdw_debug.h"
#include "gettext.h"
#include "cdw_erase_disc.h"
#include "cdw_ext_tools.h"
#include "cdw_form.h"

/**
   \file blank_wizard.c

   Simple, one page wizard: a dialog window with dropdowns and buttons
   in which user can set up basic options for current operation of
   erasing and / or formatting optical disc.
*/


extern cdw_config_t global_config;


static void     cdw_erase_wizard_init(void);
static cdw_rv_t cdw_erase_wizard_build(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_erase_wizard_driver(void);
static void     cdw_erase_wizard_driver_handle_enter(cdw_form_t *cdw_form, int fi, __attribute__((unused)) void *data);
static void     cdw_erase_wizard_destroy(void);

static CDW_DROPDOWN *cdw_erase_wizard_make_erase_mode_dropdown(cdw_disc_type_t disc_type);
static CDW_DROPDOWN *cdw_erase_wizard_make_speed_dropdown(cdw_disc_t *disc);
static CDW_DROPDOWN *cdw_erase_wizard_make_format_mode_dropdown(cdw_disc_t *disc);

static cdw_rv_t cdw_erase_wizard_initial_warning(cdw_disc_type_t disc_type);
static void     cdw_erase_wizard_print_debug_info(cdw_task_t *task, bool format_mode_visible);



static struct {
	cdw_form_t *cdw_form;

	CDW_DROPDOWN *erase_speed_dropdown; /* fast / all (slow) */
	CDW_DROPDOWN *erase_mode_dropdown;
	/* dropdown showing possible formatting modes for DVD-RW:
	   'don't change format' (default),
	   'format to sequential',
	   'format to restricted' */
	CDW_DROPDOWN *format_mode_dropdown;
	bool format_mode_dropdown_visible;

	CDW_BUTTON *erase_button;
	CDW_BUTTON *cancel_button;
} wizard;



/* constants for layout of elements in wizard window */
#define first_col                  2 /* main message and some labels start in leftmost column */
#define top_label_row              2 /* main message is displayed on top of window */
#define erase_mode_row             2
#define erase_speed_row            5
#define format_mode_row            8 /* label above of button invoking configuration window */
#define buttons_row               13 /* Erase and Cancel buttons are at the bottom of window */

#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 4)
#define label_n_cols               (subwindow_n_cols - 2)
#define wide_dropdown_n_cols       (subwindow_n_cols - 6)
#define narrow_dropdown_n_cols     8




#define CDW_ERASE_WIZARD_N_FIELDS 8 /* does not include ending NULL element */
static FIELD *wizard_fields[CDW_ERASE_WIZARD_N_FIELDS + 1];

enum {
	f_erase_mode_l = 0, /* all/fast */
	f_erase_mode_i,
	f_erase_speed_l,
	f_erase_speed_i,
	f_format_mode_l,
	f_format_mode_i,

	f_erase_button_i,
	f_cancel_button_i
};




/**
   \brief Window displayed just before erasing is performed

   Wizard window displayed before actual erasing. User can select
   here some erasing / formatting options.

   In case of erasing a CD user can select erasing mode and erasing speed.
   In case of erasing a DVD user can select erasing mode, formatting mode and
   erasing / formatting speed.

   In case of erasing of DVD+/-RW user is also warned that multiple erasing
   of DVD can quickly render it unusable.

   \param task - variable describing current task
   \param current_disc - variable describing current disc

   \return CDW_OK if user pressed accepts erasing
   \return CDW_CANCEL if user cancels erasing
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_erase_wizard(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_ERASE_DISC, "ERROR: incorrect task id, should be %d, is %d\n", CDW_TASK_ERASE_DISC, task->id);
	cdw_assert (disc->type_erasable == CDW_TRUE, "ERROR: disc is not erasable\n");

	if (disc->type_erasable != CDW_TRUE) {
		cdw_vdm ("ERROR: disc is not erasable\n");
		return CDW_GEN_ERROR;
	}

	cdw_rv_t crv = cdw_erase_wizard_initial_warning(disc->type);
	if (crv == CDW_CANCEL) {
		return CDW_CANCEL;
	}
	cdw_main_ui_main_window_wrefresh();

	/* TODO: there is a problem with implicit initialization of
	   dropdowns in the wizard: dropdowns are initialized with
	   some deeply hidden values in functions creating dropdown,
	   and then (below) states of the dropdowns are explicitly
	   used to set values of fields in task variable;
	   this mix of implicit/explicit assignments is confusing */

	cdw_erase_wizard_init();

	crv = cdw_erase_wizard_build(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to create wizard\n");
		cdw_erase_wizard_destroy();
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("ERROR"),
				   /* 2TRANS: this is message in dialog window */
				   _("Failed to create erasing wizard. Please restart cdw and try again."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_GEN_ERROR;
	}

	crv = cdw_erase_wizard_driver();
	if (crv == CDW_OK) {
		/* save values set in wizard; this works both when uses
		   made any changes in wizard, or when only accepts current
		   settings */
		if (wizard.format_mode_dropdown_visible) {
			task->erase.format_mode = cdw_dropdown_get_current_item_id(wizard.format_mode_dropdown);
		}
		task->erase.erase_speed = cdw_dropdown_get_current_item_id(wizard.erase_speed_dropdown);
		task->erase.erase_mode = cdw_dropdown_get_current_item_id(wizard.erase_mode_dropdown);
		global_config.erase_mode = task->erase.erase_mode;

		cdw_erase_wizard_print_debug_info(task, wizard.format_mode_dropdown_visible);

	} else {
		cdw_vdm ("INFO: not attempting erasing, not displaying settings\n");
	}

	cdw_erase_wizard_destroy();

	/* redraw parent */
	cdw_main_ui_main_window_wrefresh();

	return crv;
}





cdw_rv_t cdw_erase_wizard_driver(void)
{
	int fi = f_cancel_button_i; /* initial focus on "cancel" button */
	int key = 'a'; /* safe initial value */

	while (key != CDW_KEY_ESCAPE) {
		key = cdw_form_driver(wizard.cdw_form, fi, &global_config);

		/* in this wizard driver we are interested only in
		   ENTER being pressed on either "erase" or "cancel"
		   buttons */
		if (key == CDW_KEY_ENTER) {
			fi = field_index(current_field(wizard.cdw_form->form));
			if (fi == f_erase_button_i) {

				/* flush */
				form_driver(wizard.cdw_form->form, REQ_VALIDATION);

				return CDW_OK;
			} else if (fi == f_cancel_button_i) {
				return CDW_CANCEL;
			} else {
				;
			}
		} else {
			;
		}
	}

	return CDW_CANCEL;
}





/**
   \brief Initialize wizard data structures

   \param task
   \param disc
*/
void cdw_erase_wizard_init(void)
{
	cdw_assert (LINES > window_n_lines, "ERROR: LINES is too small: %d\n", LINES);
	cdw_assert (COLS > window_n_cols, "ERROR: COLS is too small: %d\n", COLS);

	wizard.erase_mode_dropdown = (CDW_DROPDOWN *) NULL;
	wizard.erase_speed_dropdown = (CDW_DROPDOWN *) NULL;
	wizard.format_mode_dropdown = (CDW_DROPDOWN *) NULL;
	wizard.format_mode_dropdown_visible = false;

	wizard.erase_button = (CDW_BUTTON *) NULL;
	wizard.cancel_button = (CDW_BUTTON *) NULL;

	wizard.format_mode_dropdown_visible = false;

	return;
}







/**
   \brief Cleanup function for erase wizard

   Function deallocates all resources used by wizard window.
   Call this function when you close wizard in normal mode or when
   you want to clean up after unsuccessful creation of wizard.
*/
void cdw_erase_wizard_destroy(void)
{
	if (wizard.format_mode_dropdown_visible) {
		cdw_dropdown_delete(&wizard.format_mode_dropdown);
		cdw_assert (wizard.format_mode_dropdown == (CDW_DROPDOWN *) NULL,
			    "ERROR: delete() didn't set pointer to NULL\n");
	}

	cdw_dropdown_delete(&wizard.erase_mode_dropdown);
	cdw_dropdown_delete(&wizard.erase_speed_dropdown);
	cdw_assert (wizard.erase_mode_dropdown == (CDW_DROPDOWN *) NULL,
		    "ERROR: delete() didn't set pointer to NULL\n");
	cdw_assert (wizard.erase_speed_dropdown == (CDW_DROPDOWN *) NULL,
		    "ERROR: delete() didn't set pointer to NULL\n");

	cdw_button_delete(&wizard.erase_button);
	cdw_button_delete(&wizard.cancel_button);
	cdw_assert (wizard.erase_button == (CDW_BUTTON *) NULL,
		    "ERROR: delete() didn't set button pointer to NULL\n");
	cdw_assert (wizard.cancel_button == (CDW_BUTTON *) NULL,
		    "ERROR: delete() didn't set button pointer to NULL\n");

	if (wizard.cdw_form->window != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->window);
		wizard.cdw_form->window = (WINDOW *) NULL;
	}

	if (wizard.cdw_form->subwindow != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->subwindow);
		wizard.cdw_form->subwindow = (WINDOW *) NULL;
	}

	if (wizard.cdw_form != (cdw_form_t *) NULL) {
		cdw_form_delete_form_objects(wizard.cdw_form);
		cdw_form_delete(&(wizard.cdw_form));
	}

	return;
}





/**
   \brief Show information about possible problems with erasing of some disc types

   Function warns user about probability of problems that may occur
   when erasing of certain types of disc is about to be performed. User
   can cancel erasing procedure.

   \param disc_type - variable describing exact disc type

   \return CDW_OK either when disc type isn't problematic or when user agrees to continue with problematic disc
   \return CDW_CANCEL when user don't want to continue
*/
cdw_rv_t cdw_erase_wizard_initial_warning(cdw_disc_type_t disc_type)
{
	if (disc_type == CDW_DVD_RWP || disc_type == CDW_DVD_RW_RES) {

		/* it is important to warn user at this point */
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t w = cdw_buttons_dialog(_("Attempting to erase a DVD+/-RW disc"),
						/* 2TRANS: this is warning message in dialog window.
						   User can press OK or Cancel button. */
						_("This will (partially) erase information from disc making it unreadable.\nThe process may take some time.\n\nYou don't have to erase DVD+RW or DVD-RW Restricted Overwrite disc to create new compilation - just choose \"Start new, appendable disc\" in Write wizard.\n\nDo you want to continue with erasing?"),
						CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);

		if (w != CDW_OK) {
			return CDW_CANCEL;
		}
	} else if (disc_type == CDW_DVD_RW || disc_type == CDW_DVD_RW_SEQ) {
		/* it is important to warn user at this point */
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t w = cdw_buttons_dialog(_("Attempting to erase a DVD-RW disc"),
						/* 2TRANS: this is warning message in dialog window.
						   User can press OK or Cancel button. */
						_("This will (partially) erase information from disc making it unreadable.\nThe process may take some time.\n\nExcessive erasing of DVD may limit lifetime of media.\n\nDo you want to continue with erasing?"),
						CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
		if (w != CDW_OK) {
			return CDW_CANCEL;
		}
	} else {
		return CDW_OK;
	}

	return CDW_OK;
}





/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_GEN_ERROR on failure
*/
cdw_rv_t cdw_erase_wizard_build(cdw_task_t *task, cdw_disc_t *disc)
{
	if (disc->type == CDW_DVD_RW_RES
	    || disc->type == CDW_DVD_RW_SEQ
	    || disc->type == CDW_DVD_RW) {

		if (task->erase.tool.id == CDW_TOOL_DVD_RW_FORMAT) {
			wizard.format_mode_dropdown_visible = true;
		}
	}

	wizard.cdw_form = cdw_form_new(CDW_ERASE_WIZARD_N_FIELDS);
	if (wizard.cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_GEN_ERROR;
	}

	wizard.cdw_form->fields = wizard_fields;

	int begin_y = ((LINES - window_n_lines) / 2) - 2;
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window = cdw_ncurses_window_new((WINDOW *) NULL,
							 window_n_lines, window_n_cols,
							 begin_y, begin_x,
							 CDW_COLORS_DIALOG,
							 /* 2TRANS: this is title of wizard window; "erase" as in
							    "Erasing an optical disc" */
							 _("Erase wizard"),
							 /* 2TRANS: this is tip at the bottom of window - user can
							    switch between window elements using tab key */
							 _("Use 'Tab' key to move"));
	if (wizard.cdw_form->window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_GEN_ERROR;
	}

	wizard.cdw_form->subwindow = cdw_ncurses_window_new(wizard.cdw_form->window,
							    subwindow_n_lines, subwindow_n_cols,
							    3, 1,
							    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);

	if (wizard.cdw_form->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_GEN_ERROR;
	}

	/* *** main message in wizard window *** */
	mvwprintw(wizard.cdw_form->window, top_label_row, first_col,
		  /* 2TRANS: this is message to user: current action will
		     be erasing "%s" disc, where "%s" is DVD or CD */
		  _("Select parameters for erasing %s disc"), disc->simple_type_label);


	cdw_form_descr_t descr[] = {
		/*    widget type       begin_y               begin_x         n_cols                   n_lines    field enum         data1                        data2 */

		/* 2TRANS: this is label in erase wizard */
		{ CDW_WIDGET_LABEL,     erase_mode_row,       first_col,      label_n_cols,                1,    f_erase_mode_l,     _("Select mode of erasing:"),   0 },
		{ CDW_WIDGET_DROPDOWN,  erase_mode_row + 1,   first_col + 1,  wide_dropdown_n_cols - 2,    1,    f_erase_mode_i,     (void *) NULL,                  0 },

		/* 2TRANS: this is label in erase wizard */
		{ CDW_WIDGET_LABEL,     erase_speed_row,      first_col,      label_n_cols,                1,    f_erase_speed_l,    _("Select speed of erasing:"),  0 },
		{ CDW_WIDGET_DROPDOWN,  erase_speed_row + 1,  first_col + 1,  narrow_dropdown_n_cols - 2,  1,    f_erase_speed_i,    (void *) NULL,                  0 },

		/* 2TRANS: this is label in erase wizard, it's about target
		   format for a DVD-RW disc */
		{ CDW_WIDGET_LABEL,     format_mode_row,      first_col,      label_n_cols,                1,    f_format_mode_l,    _("Select DVD-RW format:"),     0 },
		{ CDW_WIDGET_DROPDOWN,  format_mode_row + 1,  first_col + 1,  wide_dropdown_n_cols - 2,    1,    f_format_mode_i,    (void *) NULL,                  0 },

		{ CDW_WIDGET_BUTTON,    buttons_row,          3,              2,                           1,    f_erase_button_i,   (void *) NULL,                  0 },
		{ CDW_WIDGET_BUTTON,    buttons_row,          15,             2,                           1,    f_cancel_button_i,  (void *) NULL,                  0 },

		/* guard */
		{ -1,                   0,                    0,              0,                           0,    0,                  (void *) NULL,                 0 }};

	/* this function also sets which dropdown item is selected at the beginning */
	wizard.erase_mode_dropdown = cdw_erase_wizard_make_erase_mode_dropdown(disc->type);
	if (wizard.erase_mode_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create session mode dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_erase_mode_i].data1 = (void *) wizard.erase_mode_dropdown;
	}

	/* this function also sets which dropdown item is selected at the beginning */
	wizard.erase_speed_dropdown = cdw_erase_wizard_make_speed_dropdown(disc);
	if (wizard.erase_speed_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create speed dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_erase_speed_i].data1 = (void *) wizard.erase_speed_dropdown;
	}

	if (wizard.format_mode_dropdown_visible) {
		/* this function also sets which dropdown item is selected at the beginning */
		wizard.format_mode_dropdown = cdw_erase_wizard_make_format_mode_dropdown(disc);
		if (wizard.format_mode_dropdown == (CDW_DROPDOWN *) NULL) {
			cdw_vdm ("ERROR: failed to create session mode dropdown\n");
			return CDW_GEN_ERROR;
		} else {
			descr[f_format_mode_i].data1 = (void *) wizard.format_mode_dropdown;
		}
	}


	/* *** buttons at the bottom *** */
	wizard.erase_button = cdw_button_new(wizard.cdw_form->subwindow, buttons_row, 3,
					     /* 2TRANS: button label, it refers to erasing optical disc */
					     _("Erase"), CDW_COLORS_DIALOG);
	descr[f_erase_button_i].data1 = (void *) wizard.erase_button;

	wizard.cancel_button = cdw_button_new(wizard.cdw_form->subwindow, buttons_row, 15,
					      /* 2TRANS: button label */
					      _("Cancel"), CDW_COLORS_DIALOG);
	descr[f_cancel_button_i].data1 = (void *) wizard.cancel_button;

	wizard.cdw_form->n_fields = CDW_ERASE_WIZARD_N_FIELDS;
	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	wizard.cdw_form->form = cdw_form_new_form(wizard.cdw_form->window,
						  wizard.cdw_form->subwindow,
						  wizard.cdw_form->fields);
	if (wizard.cdw_form->form == (FORM *) NULL) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_GEN_ERROR;
	}


	/* current items are set in functions creating respective
	   dropdowns; now just display them */
	cdw_dropdown_display_current_item(wizard.erase_speed_dropdown);
	cdw_dropdown_display_current_item(wizard.erase_mode_dropdown);
	if (wizard.format_mode_dropdown_visible) {
		cdw_dropdown_display_current_item(wizard.format_mode_dropdown);
	} else {
		field_opts_off(wizard.cdw_form->fields[f_format_mode_i], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_format_mode_l], O_VISIBLE);
	}

	cdw_button_unfocus(wizard.erase_button);
	cdw_button_unfocus(wizard.cancel_button);

	cdw_form_add_return_char(wizard.cdw_form, CDW_KEY_ENTER);
	cdw_form_add_return_char(wizard.cdw_form, CDW_KEY_ESCAPE);

	wizard.cdw_form->handle_enter = cdw_erase_wizard_driver_handle_enter;

	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





/**
   \brief Check which element has currently keyboard focus, and handle Enter key for this element

   Function checks which element in wizard window (identified by field
   index \p fi) has currently keyboard focus and executes related code that
   performs action appropriate for given element.

   "elements" are buttons and dropdowns. Checkboxes don't "recognize" Enter key.

   \param cdw_form - form in which elements are placed
   \param fi - field index, index of form field that has keyboard focus
   \param data - pointer to additional data that may be used by some elements
*/
void cdw_erase_wizard_driver_handle_enter(cdw_form_t *cdw_form, int fi, __attribute__((unused)) void *data)
{
	if (cdw_form->is_dropdown_field_index(cdw_form, fi)) {
		CDW_DROPDOWN *dropdown = cdw_form->get_dropdown_from_form(cdw_form, fi);
		if (dropdown != (CDW_DROPDOWN *) NULL) {
			cdw_rv_t crv = cdw_dropdown_driver(dropdown);
			if (crv == CDW_OK) {
				const char *label = cdw_dropdown_get_current_item_label(dropdown);
				FIELD **fields = form_fields(cdw_form->form);
				/* setting buffer makes field to display
				   string, but slightly moved to left */
				set_field_buffer(*(fields + fi), 0, label);

				/* call to cdw_dropdown_focus() fixes it */
				cdw_dropdown_focus(dropdown);

				form_driver(cdw_form->form, REQ_VALIDATION);
			} /* else field buffer is not updated and nothing changes */
		}
	} else {
		;
	}

	return;
}





CDW_DROPDOWN *cdw_erase_wizard_make_erase_mode_dropdown(cdw_disc_type_t disc_type)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(wizard.cdw_form->subwindow,
						  erase_mode_row + 1, first_col,
						  wide_dropdown_n_cols, /* width */
						  2,                    /* n_items_max */
						  CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}


	if (disc_type == CDW_DVD_RWP) {
		/* 2TRANS: this string means "perform erasing in fast mode" */
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, CDW_ERASE_MODE_FAST, _("Erase in fast mode"));
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create erase mode dropdown label 0\n");
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	} else {
		/* 2TRANS: this string means "perform erasing in fast mode" */
		cdw_rv_t crv1 = cdw_dropdown_add_item(dropdown, CDW_ERASE_MODE_FAST, _("Erase in fast mode"));
		/* 2TRANS: this string means "erase all data from disc" */
		cdw_rv_t crv2 = cdw_dropdown_add_item(dropdown, CDW_ERASE_MODE_ALL, _("Erase all content (long time)"));

		if (crv1 != CDW_OK || crv2 != CDW_OK) {

			cdw_vdm ("ERROR: failed to create erase mode dropdown label 0 or 1\n");
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		cdw_dropdown_set_current_item_by_id(dropdown, global_config.erase_mode);
		return dropdown;
	}
}





CDW_DROPDOWN *cdw_erase_wizard_make_speed_dropdown(cdw_disc_t *disc)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(wizard.cdw_form->subwindow,
						  erase_speed_row + 1, first_col,
						  narrow_dropdown_n_cols, /* width */
						  disc->write_speeds.n_speeds,   /* n_items_max */
						  CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	char label[5];
	for (int i = 0; i < disc->write_speeds.n_speeds; i++) {
		snprintf(label, 4 + 1, "%d", disc->write_speeds.speeds[i]);
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, disc->write_speeds.speeds[i], label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create dropdown label #%d of value %d\n", i, disc->write_speeds.speeds[i]);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}


	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		/* the function returns index of speed based on wizard.speed_range;
		   cdw_disc module guarantees it to be >= 0 */
		int speed_index = cdw_disc_get_initial_write_speed_index(disc);
		cdw_dropdown_set_current_item_by_ind(dropdown, speed_index);
		return dropdown;
	}
}





CDW_DROPDOWN *cdw_erase_wizard_make_format_mode_dropdown(cdw_disc_t *disc)
{
	/* remember that gettext() result strings must not be free()d */

	/* 2TRANS: this is label in dropdown: it means "don't change
	   format of DVD-RW during process of erasing disc" */
	char *dont_change = _("Don't change format of DVD-RW");
	/* 2TRANS: this is label in dropdown: 'Sequential' is one of two
	   possible formats of DVD-RW */
	char *seq = _("Change format to Sequential");
	/* 2TRANS: this is label in dropdown: 'Restricted' is one of two
	   possible formats of DVD-RW */
	char *res = _("Change format to Restricted");

	if (seq == (char *) NULL || res == (char *) NULL || dont_change == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create initial labels for format mode dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	CDW_DROPDOWN *dropdown = cdw_dropdown_new(wizard.cdw_form->subwindow,
						  format_mode_row + 1, first_col,
						  wide_dropdown_n_cols, /* width of dropdown */
						  3,                    /* n_items_max */
						  CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	cdw_rv_t crv1 = CDW_GEN_ERROR;
	cdw_rv_t crv2 = CDW_GEN_ERROR;
	if (disc->type == CDW_DVD_RW_SEQ) {
		crv1 = cdw_dropdown_add_item(dropdown, CDW_ERASE_DVD_RW_FORMAT_SEQ, dont_change);
		crv2 = cdw_dropdown_add_item(dropdown, CDW_ERASE_DVD_RW_FORMAT_RES, res);


	} else if (disc->type == CDW_DVD_RW_RES) {
		crv1 = cdw_dropdown_add_item(dropdown, CDW_ERASE_DVD_RW_FORMAT_RES, dont_change);
		crv2 = cdw_dropdown_add_item(dropdown, CDW_ERASE_DVD_RW_FORMAT_SEQ, seq);


	} else { /* we don't know if current disc is in Sequential or Restricted format */
		crv1 = cdw_dropdown_add_item(dropdown, CDW_ERASE_DVD_RW_FORMAT_RES, res);
		crv2 = cdw_dropdown_add_item(dropdown, CDW_ERASE_DVD_RW_FORMAT_SEQ, seq);

	}

	if (crv1 != CDW_OK || crv2 != CDW_OK) {
		cdw_vdm ("ERROR: failed to initialize format mode dropdown label 0 or 1\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}


	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		/* 0 - first item on list will be default, regardless of
		   what the ID of this item is */
		cdw_dropdown_set_current_item_by_ind(dropdown, 0);
		return dropdown;
	}
}





void cdw_erase_wizard_print_debug_info(cdw_task_t *task, bool format_mode_dropdown_visible)
{
	/* for printing debug information we use values saved in "task"
	   variable, not in "wizard" struct, because we want to check what
	   really was saved in "task" */
	cdw_vdm ("INFO:              selected speed = \"%d\"\n", task->erase.erase_speed);
	cdw_vdm ("INFO:         selected erase mode = \"%s\"\n", task->erase.erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");

	if (format_mode_dropdown_visible) {
		if (task->erase.format_mode == CDW_ERASE_DVD_RW_FORMAT_SEQ) {
			cdw_vdm ("INFO: selected DVD-RW format mode = \"Sequential\"\n");
		} else if (task->erase.format_mode == CDW_ERASE_DVD_RW_FORMAT_RES) {
			cdw_vdm ("INFO: selected DVD-RW format mode = \"Restricted overwrite\"\n");
		} else {
			cdw_vdm ("ERROR: incorrect format mode = %d\n", task->erase.format_mode);
		}
	} else {
		cdw_vdm ("INFO: DVD-RW format mode dropdown is NOT visible\n");
	}
	return;
}



