#ifndef H_CDW_TASK
#define H_CDW_TASK

#include <stdbool.h>

#include "main.h"
#include "cdw_disc.h"
//#include "external_tools.h"

#define CDW_TOOL_STATUS_OK                                       0x00000000

/* mkisofs */
#define CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME             0x00000001
#define CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT                 0x00000002
#define CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE             0x00000004
#define CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR              0x00000008
#define CDW_TOOL_STATUS_MKISOFS_BAD_OPTION                       0x00000010
#define CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW            0x00000020
#define CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT                    0x00000040
#define CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH               0x00000080
#define CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME                 0x00000100
#define CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW                    0x00000200
#define CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR                      0x00000300
#define CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE               0x00000800
#define CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL                    0x00001000
#define CDW_TOOL_STATUS_MKISOFS_SHORT_READ                       0x00002000

/* cdrecord */
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE              0x00000001
#define CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR             0x00000002
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET           0x00000004
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH  0x00000008
#define CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT                0x00000010
#define CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER                     0x00000020
#define CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT                0x00000040
#define CDW_TOOL_STATUS_CDRECORD_BAD_OPTION                      0x00000080
#define CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK              0x00000100

/* growisofs */
#define CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES             0x00000001
#define CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE           0x00000002
#define CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION                   0x00000004
#define CDW_TOOL_STATUS_GROWISOFS_IO_ERROR                       0x00000008
#define CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE             0x00000010
#define CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY                   0x00000020


/* in the process of verification there may be used two different tools:
   standard code reading files, or library code (from libcdio
   or similar library) reading optical disc; in the first case
   there should be displayed some generic system message (e.g.
   can't find some file in user's system), in the second case
   there should be displayed information about problems with
   accessing optical disc or about problems with optical drive */
#define CDW_TOOL_STATUS_VERIFY_SETUP_FAIL                        0x00000001
#define CDW_TOOL_STATUS_VERIFY_READ_FAIL                         0x00000002
#define CDW_TOOL_STATUS_VERIFY_WRONG_TRACK                       0x00000004

/* libcdio */
#define CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL                       0x00080001
#define CDW_TOOL_STATUS_LIBCDIO_READ_FAIL                        0x00080002




/* mode of writing files to a CD/DVD disc */
typedef enum {
	CDW_SESSION_MODE_INIT = -1,
	CDW_SESSION_MODE_START_MULTI = 0,  /**< \brief Start writing to disc in mode allowing further appending of data (don't close disc after writing) */
	CDW_SESSION_MODE_CREATE_SINGLE,    /**< \brief Start writing to disc in mode NOT allowing further appending of data (close disc after writing)*/
	CDW_SESSION_MODE_CONTINUE_MULTI,   /**< \brief Continue writing to disc in mode allowing further appending of data (don't close disc after writing) */
	CDW_SESSION_MODE_WRITE_FINAL,      /**< \brief Write to disc in mode NOT allowing further appending of data (close disc after writing) */
	CDW_SESSION_MODE_ERROR,
	CDW_SESSION_MODE_N_MAX
} cdw_session_mode_t;


/* mode of writing files to a CD/DVD disc */
typedef enum {
	CDW_DISC_MODE_INIT = -1,
	CDW_DISC_MODE_UNSPECIFIED = 0, /**< \brief Let a tool select its own mode */
	CDW_DISC_MODE_TAO,      /**< \brief Track at once */
	CDW_DISC_MODE_DAO,    /**< \brief Disc at once */
	CDW_DISC_MODE_SAO,   /**< \brief Session at once */
	CDW_DISC_MODE_RAW,      /**< \brief Raw */
	CDW_DISC_MODE_RAW96P,
	CDW_DISC_MODE_RAW96R,
	CDW_DISC_MODE_RAW16,
	CDW_DISC_MODE_ERROR,
	CDW_DISC_MODE_N_MAX
} cdw_disc_mode_t;




typedef struct  {
	enum {
		CDW_TASK_NONE,            /**< no task currently assigned */

		CDW_TASK_BURN_FROM_IMAGE,      /**< burn image to optical disc */
		CDW_TASK_BURN_FROM_FILES,      /**< burn files to optical disc */
		CDW_TASK_READ_DISC,       /**< read content of optical disc */
		CDW_TASK_ERASE_DISC,      /**< erase optical disc - this includes formatting */
		CDW_TASK_CLONE_DISC,      /**< clone one optical disc to another optical disc */

		CDW_TASK_CREATE_IMAGE,    /**< create iso image on hdd, using files */

		CDW_TASK_CHECK_MD5SUM,    /**< check md5sum of specified object */

		CDW_TASK_MEDIA_INFO        /**< get check media information */
	} id; /** \brief Identifier of given task */

	struct {
		bool ok;
		size_t mkisofs;
		size_t cdrecord;
		size_t verify;
		size_t growisofs;
		size_t dvd_rw_mediainfo;
		size_t libcdio;
	} tool_status;


	struct {
		/* "input" parameters of erasing */
		int format_mode; /* format of DVD-RW: Sequential or Restricted Overwrite */
		int erase_speed; /* just a numeric value */
		int erase_mode;  /* erasing disc: all or fast */
		struct {
			int id;
			char *fullpath;
		} tool;


		/* "output" parameter of erasing (but only for CD-RW): if
		   after attempting to perform erasing of CD-RW value of
		   erase_time is still zero, then erasing failed */
		size_t erase_time;
	} erase;

	struct {
		/* single sesion, append, finalize, etc. */
		int session_modes[CDW_SESSION_MODE_N_MAX];
		int session_modes_n_labels;
		cdw_session_mode_t session_mode;

		int speed;

		/* tao, sao, etc. */
		int disc_modes[CDW_DISC_MODE_N_MAX];
		int disc_modes_n_labels;
		cdw_disc_mode_t disc_mode;

		bool verify;

		struct {
			int id;
			char *fullpath;
		} tool;

		/* size of data being burned to disc; this variable should not store
		   _exact_ size of data, value stored in this variable should be
		   used only when approximate data size needs to be known, e.g. in
		   progress window when displaying approximate progress of task;
		   depending on type of task this variable stores size of iso image
		   file being burned to optical disc, or size of files selected for
		   burning to optical disc */
		double data_size_mb;
	} burn;

	struct {
		struct {
			int id;
			char *fullpath;
		} tool;
	} create_image;

	struct {
		struct {
			int id;
			char *fullpath;
		} tool;
	} media_info;

	struct {
		struct {
			int id;
			char *fullpath;
		} tool;
		bool from_cd;

		char *iso_fullpath;

		int data_source_fd;
		int md5sum_stdin_fd;

		void * (*read_and_write)(void *task);

	} verify;
} cdw_task_t; /** \brief Data structure describing task that is currently
                         performed or is about to be performed */


void cdw_task_check_tool_status(cdw_task_t *task);
cdw_rv_t cdw_task_select_tools_for_task(cdw_task_t *task, cdw_disc_t *disc);
cdw_rv_t cdw_task_resolve_allowed_writing_modes(cdw_task_t *task, cdw_disc_t *disc);
void cdw_task_reset_tool_status(cdw_task_t *task);

void cdw_task_reset_state(cdw_task_t *task);

const char *cdw_task_get_session_mode_label(int mode);
const char *cdw_task_get_disc_mode_label(int mode);

/* macro, but very useful */
#define CDW_TASK_CREATE_TASK(task_name, task_id) \
	cdw_task_t task_name; \
	cdw_task_reset_state(&task_name); \
	task_name.id = task_id;

#endif /* H_CDW_TASK */
