/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* asprintf() */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h> /* PATH_MAX */

#include <errno.h>  /* EINVAL */

#include "gettext.h"
#include "cdw_task.h"

#include "cdw_ext_tools.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"
#include "cdw_thread.h"
#include "cdw_logging.h"
#include "cdw_which.h"
#include "cdw_config_ui.h"
#include "cdw_config_ui_internals.h"

/**
   \file external_tools.c

   \brief File implements code searching for tools such as mkisofs or
   growisofs, storing full paths in file system to these tools,
   implements getter function providing pointer to path to specified
   tool.

   The file also implements code that creates labels for dropdowns, with
   which user can select one of (possibly) few available implementations
   of given tool.

   This file uses term "instance" - its meaning is "executable file of
   tool X, being placed in given directory in file system". Example:
   on my system tool cdrecord has two instances:
   \li /home/acerion/bin/cdrecord/
   \li /usr/bin/cdrecord (which is a symbolic link to /usr/bin/wodim)

   Labels mentioned above can be used to create a dropdown widget in
   configuration window, with which user can select one of these two
   instances to perform writing to CD.
*/



extern struct cdw_config_ext_tools_dropdowns_t ext_tools_dropdowns;


typedef struct {
	char version[30];
	int id; /* currently unused */
	char *label;
	char *fullpath; /* path to executable, that was found by "which" utility */

	/* if a  fullpath is a symbolic link, then it would be
	   nice to have path (at least relative) to real program, so that we
	   can show it to a user (to give him more information). Here is a
	   place for this path (potentially full, but can be relative) to
	   'target' file. Note that only some (or none, or all) of tools from
	   fullpath will have 'targets' */
	char *target_path;
	int capabilities;
} cdw_tool_instance_t;



typedef struct {
	/* name of executable, 20 chars should be enough for everyone */
	char name[20];

	/* is the tool available in system at all? */
	bool available;
	/* single tool can have multiple instances (implementations) located
	   in different locations in file system */
	cdw_tool_instance_t instances[CDW_EXT_TOOLS_N_INSTANCES_MAX + 1];
	int n_instances;

	/* dropdown index, index of currently selected item in dropdown, and
	   thus index of tool implementation selected by user (can point to
	   "selected by cdw" or "system default" as well; initially set by
	   external tools module to some default value, then can be modified
	   by user in configuration window */
	int current_instance_ind;

} cdw_tool_t;


/* maximal number of families of tools:
   dvd+rw-tools, cdrtools, libburnia tools, <your tool name here>;
   used with "Tools family to use for DVDs" dropdown in configuration window */
#define CDW_EXT_TOOLS_N_FAMILIES_MAX 3


#define CDW_TOOL_MKISOFS_NAME            "mkisofs"
#define CDW_TOOL_CDRECORD_NAME           "cdrecord"
#define CDW_TOOL_GROWISOFS_NAME          "growisofs"
#define CDW_TOOL_DVD_RW_FORMAT_NAME      "dvd+rw-format"
#define CDW_TOOL_DVD_RW_MEDIAINFO_NAME   "dvd+rw-mediainfo"
#define CDW_TOOL_MD5SUM_NAME             "md5sum"


static struct {
	struct {
		bool manual_selection;

		struct {
			/* cdrecord (+ xorriso) */
			/* dropdown labels and IDs for dropdowns displayed in config
			   window; user can select cd tools family when manual
			   configuration is enabled */
			cdw_dropdown_item_t items[CDW_EXT_TOOLS_N_FAMILIES_MAX];
			int n_items;
			/* tools family used when manual tool configuration is not
			   selected */
			int default_id;
			int current_family_ind;
		} cd_family;

		struct {
			/* cdrecord + growisofs (+ xorriso) */
			/* dropdown labels and IDs for dropdowns displayed in config
			   window; user can select dvd tools family when manual
			   configuration is enabled */
			cdw_dropdown_item_t items[CDW_EXT_TOOLS_N_FAMILIES_MAX];
			int n_items;
			/* tools family used when manual tool configuration is not
			   selected */
			int default_id;
			int current_family_ind;
		} dvd_family;
	} config; /* note that there are no tool dropdowns in "config" struct,
		     there are only tool families; this is not an example
		     of consistency and good design :/ */

	cdw_tool_t tools[CDW_N_TOOLS];
} cdw_ext_tools;



static void cdw_ext_tools_print_fullpaths(void);
static void cdw_ext_tools_init_labels(void);
static cdw_rv_t cdw_ext_tools_build_dropdown_labels(void);
static void cdw_ext_tools_print_labels(void);
static void cdw_ext_tools_find_external_tools(void);
static int cdw_ext_tools_process_tool_instances(int t, char **instances);
static void cdw_ext_tools_init_cdw_tools(void);
static void cdw_ext_tools_check_tool_id_assertions(void);
static void cdw_ext_tools_build_cd_family(void);
static void cdw_ext_tools_build_dvd_family(void);
// static cdw_rv_t cdw_ext_tools_check_versions(void);


static const char *cdw_ext_tools_get_instance_manual(int tool_id);
static const char *cdw_ext_tools_get_instance_auto(int tool_id);

static const char *cdw_ext_tools_get_system_default_instance(int tool_id);

//static size_t cdw_ext_tools_select_tool_impl_intelligently(enum cdw_tool tool_id);


static char *cdw_ext_tools_get_current_instance_label(int tool_id);
static int cdw_ext_tools_get_current_instance_ind(int tool_id);

static CDW_DROPDOWN *cdw_ext_tools_tool_dropdown_new(WINDOW *window, int row, int col, int width, cdw_tool_t *tool);


/* \brief Label in tools dropdown indicating that a tool is not available */
static char label_not_available[20];
static char label_system_default[20];
static char label_no_tool_available[20];
static char dvd_rw_tools_label[] = "dvd+rw-tools";

/**
   \brief Initialize "external tools" module

   The function checks availability of "which" utility, and runs it to
   check whether and where tools such as mkisofs or cdrecord are available
   in the system. Collected information is put into cdw_tools[].

   Function does not return error value when some (or all) tools are
   unavailable. It calls a function which sets cdw_tool[<tool id>].available
   flag, so checking this flag (directly or using access function) is a correct
   way of checking tool status.

   \return CDW_NO if "which" tool is not available
   \return CDW_GEN_ERROR on other errors
   \return CDW_OK on success
*/
cdw_rv_t cdw_ext_tools_init(void)
{
	cdw_ext_tools_check_tool_id_assertions();
	cdw_ext_tools_init_labels();
	cdw_ext_tools_init_cdw_tools();
	cdw_ext_tools_find_external_tools();
	// cdw_ext_tools_check_versions();
	cdw_ext_tools_build_cd_family();
	cdw_ext_tools_build_dvd_family();

	cdw_rv_t crv = cdw_ext_tools_build_dropdown_labels();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build dropdown results\n");
		cdw_ext_tools_clean();
		return CDW_GEN_ERROR;
	}

#ifndef NDEBUG  /* only for debug purposes */
	cdw_vdm ("\n\nexternal tool fullpaths initialized as:\n\n");
	cdw_ext_tools_print_fullpaths();
	cdw_ext_tools_print_labels();
#endif
	return CDW_OK;
}





/**
   \brief Assign very initial values to fields in cdw_tools[] variable

   Call this function to initialize fields in cdw_tools[]. This is a first
   step to have correct and useful values in cdw_tools[]. After calling
   this function you are ready to call cdw_ext_find_external_tools().
*/
void cdw_ext_tools_init_cdw_tools(void)
{
	for (int t = 0; t < CDW_N_TOOLS; t++) {

		for (int i = 0; i < CDW_EXT_TOOLS_N_INSTANCES_MAX; i++) {
			cdw_ext_tools.tools[t].instances[i].fullpath = (char *) NULL;
			cdw_ext_tools.tools[t].instances[i].target_path = (char *) NULL;
			cdw_ext_tools.tools[t].instances[i].version[0] = '\0';
			cdw_ext_tools.tools[t].instances[i].capabilities = 0;

			cdw_ext_tools.tools[t].instances[i].label = (char *) NULL;
			cdw_ext_tools.tools[t].instances[i].id = 0;
		}

		cdw_ext_tools.tools[t].available = false;
		cdw_ext_tools.tools[t].n_instances = 0;
		cdw_ext_tools.tools[t].current_instance_ind = 0;
	}

	/* it is cdw logic that (by default) selects tools for performing
	   tasks, including selecting proper kind of tool for given task
	   (cdrecord / growisofs / xorriso) if more than one is available */
	cdw_ext_tools.config.manual_selection = false;

	cdw_ext_tools.config.dvd_family.default_id = 0;
	cdw_ext_tools.config.dvd_family.current_family_ind = 0;
	cdw_ext_tools.config.dvd_family.n_items = 0;
	cdw_ext_tools.config.cd_family.default_id = 0;
	cdw_ext_tools.config.cd_family.current_family_ind = 0;
	cdw_ext_tools.config.cd_family.n_items = 0;
	for (int t = 0; t < CDW_EXT_TOOLS_N_FAMILIES_MAX; t++) {
		cdw_ext_tools.config.dvd_family.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.dvd_family.items[t].label = (char *) NULL;

		cdw_ext_tools.config.cd_family.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.cd_family.items[t].label = (char *) NULL;
	}

	strcpy(cdw_ext_tools.tools[CDW_TOOL_MKISOFS].name, CDW_TOOL_MKISOFS_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_CDRECORD].name, CDW_TOOL_CDRECORD_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_GROWISOFS].name, CDW_TOOL_GROWISOFS_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_DVD_RW_FORMAT].name, CDW_TOOL_DVD_RW_FORMAT_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_DVD_RW_MEDIAINFO].name, CDW_TOOL_DVD_RW_MEDIAINFO_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_MD5SUM].name, CDW_TOOL_MD5SUM_NAME);

	return;
}





/**
   \brief Call "which"-like function with names of external tools

   The function calls cdw_which() for every tool in cdw_tools[] table
   to collect full paths to all instances of every tool. Then the function
   checks potential target files of links if the full paths are paths of
   symbolic links.
*/
void cdw_ext_tools_find_external_tools(void)
{
	for (int t = 0; t < CDW_N_TOOLS; t++) {
		/* instances = table with paths to every executable named
		   tools[t].name, that is available in paths specified in
		   environment variable PATH */
		char **instances = cdw_which(cdw_ext_tools.tools[t].name);
		if (instances == (char **) NULL) {
			cdw_vdm ("ERROR: can't get instances table for tool \"%s\"\n",
				 cdw_ext_tools.tools[t].name);
		}

		cdw_ext_tools.tools[t].n_instances = cdw_ext_tools_process_tool_instances(t, instances);

		cdw_assert (cdw_ext_tools.tools[t].n_instances <= CDW_EXT_TOOLS_N_INSTANCES_MAX,
			    "ERROR: too many tool instances: %d, limit is %d\n",
			    cdw_ext_tools.tools[t].n_instances, CDW_EXT_TOOLS_N_INSTANCES_MAX);

		/* paths to instances are strdup()ed by process_instances(),
		   so instances are no longer needed */
		for (int i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {
			cdw_assert (instances[i] != (char *) NULL, "ERROR: instance #%d is NULL (n_instances = %d)\n",
				    i, cdw_ext_tools.tools[t].n_instances);
			free(instances[i]);
			instances[i] = (char *) NULL;
		}
		if (instances != (char **) NULL) {
			free(instances);
			instances = (char **) NULL;
		}

		if (cdw_ext_tools.tools[t].n_instances == 0) {
			cdw_ext_tools.tools[t].available = false;
			cdw_ext_tools.tools[t].current_instance_ind = 0;
		} else {
			cdw_ext_tools.tools[t].available = true;
			/* this sets "system default" as default dropdown value */
			cdw_ext_tools.tools[t].current_instance_ind = cdw_ext_tools.tools[t].n_instances;
		}
	}

	return;
}




int cdw_ext_tools_process_tool_instances(int t, char **instances)
{
	int i = 0;
	while (instances[i] != (char *) NULL) { /* while (!guard) */
		cdw_ext_tools.tools[t].instances[i].fullpath = strdup(instances[i]);
		cdw_vdm ("INFO: fullpath to executable = \"%s\"\n",
			 cdw_ext_tools.tools[t].instances[i].fullpath);

		char link_path[PATH_MAX - 2];
		ssize_t r = readlink(instances[i], link_path, PATH_MAX - 2);
		if (r != -1) {
			link_path[r] = '\0';
			cdw_ext_tools.tools[t].instances[i].target_path = strdup(link_path);
			cdw_vdm ("INFO:\tpath to real executable = \"%s\"\n",
				 cdw_ext_tools.tools[t].instances[i].target_path);
		} else if (r == EINVAL) {
			/* "The named file is not a symbolic link." */
		} else {
			; /* TODO: check rest of values of r, some of them might
			     indicate serious problems */
		}

		i++;
	}

	return i;
}





/**
   \brief Check assertions about variable describing default paths, print current ext tools settings

   User can use configuration window and its dropdown to select
   default instance of any available tool. Configuration code modifies
   cdw_tools[<tool id>].index, and goal of this function is to
   check (only check) if cdw_tools[<tool id>].index has correct
   value (value within certain range).

   Since it uses assert()s, it is only useful in debug builds.
*/
void cdw_ext_tools_debug_print_config(void)
{
	if (!cdw_ext_tools.config.manual_selection) {
		cdw_vdm ("INFO: automatic selection, not displaying external tools settings\n\n");
		return;
	}

	cdw_vdm ("INFO: manual selection, displaying external tools settings\n");

	int i = cdw_ext_tools.config.dvd_family.current_family_ind;
	cdw_assert (cdw_ext_tools.config.dvd_family.items[i].label != (char *) NULL,
		    "ERROR: label #%d of DVD tool family is NULL\n", i);
	cdw_vdm ("INFO: DVD tool family #%d label = \"%s\"\n",
		 i, cdw_ext_tools.config.dvd_family.items[i].label);

	for (int t = 0; t < CDW_N_TOOLS; t++) {
		bool available = cdw_ext_tools.tools[t].available;
		int ind = cdw_ext_tools.tools[t].current_instance_ind;
		char *name = cdw_ext_tools.tools[t].name;

		if (available) {
			/* "1" is for 1 special label "system default" */
			cdw_assert (ind < cdw_ext_tools.tools[t].n_instances + 1,
				    "ERROR: %s: saved dropdown index is out of range (%d >= %d)\n",
				    name, ind, cdw_ext_tools.tools[t].n_instances + 1);
			if (ind < cdw_ext_tools.tools[t].n_instances) {
				cdw_vdm ("INFO: fullpath for tool %s resolved as \"%s\" with dropdown index %d\n",
					 name, cdw_ext_tools.tools[t].instances[ind].fullpath, ind);

			} else { /* ind == cdw_ext_tools.tools[t].n_instances;
				    in case of manual selection it means that
				    user selected "system default", in case of
				    automatic selection it means that ... */
				cdw_vdm ("INFO: fullpath for tool %s not yet resolved, will use resolver function\n",
					 name);
			}
		} else {
			cdw_vdm ("INFO: tool %s is not available\n", name);
			/* tool is not available, so following conditions must be true: */
			cdw_assert (ind == 0,
				    "ERROR: tool %s is marked as not available, but value of its dropdown index is non-zero: %d\n",
				    name, ind);

			cdw_assert (cdw_ext_tools.tools[t].instances[ind].label == label_not_available,
				    "ERROR: tool %s is marked as not available, but its only dropdown label is not \"not available\": \"%s\"\n",
				    name, cdw_ext_tools.tools[t].instances[ind].label);

			cdw_assert (cdw_ext_tools.tools[t].instances[ind].fullpath == (char *) NULL,
				    "ERROR: tool %s is marked as not available, but fullpath %d of the tool is not null: \"%s\"\n",
				    name, ind, cdw_ext_tools.tools[t].instances[ind].fullpath);

		}
	}

	cdw_vdm ("(newline)\n");

	return;
}





/**
   \brief Deallocate all resources allocated by "external tools" module

   The function deallocates all resources (labels, other strings and other
   memory) used by this module. You should call this function when closing
   application.
*/
void cdw_ext_tools_clean(void)
{
	for (int t = 0; t < CDW_N_TOOLS; t++) {
		for (int i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {

			if (cdw_ext_tools.tools[t].instances[i].fullpath != (char *) NULL) {
				free(cdw_ext_tools.tools[t].instances[i].fullpath);
				cdw_ext_tools.tools[t].instances[i].fullpath = (char *) NULL;
			}

			if (cdw_ext_tools.tools[t].instances[i].target_path != (char *) NULL) {
				free(cdw_ext_tools.tools[t].instances[i].target_path);
				cdw_ext_tools.tools[t].instances[i].target_path = (char *) NULL;
			}

			/* label_not_available is allocated by gettext system
			   and must not be free()d */
			if (cdw_ext_tools.tools[t].instances[i].label != (char *) NULL
			    && cdw_ext_tools.tools[t].instances[i].label != label_not_available) {

				free(cdw_ext_tools.tools[t].instances[i].label);
				cdw_ext_tools.tools[t].instances[i].label = (char *) NULL;
			}
		}
	}

	return;
}





/**
   \brief Print values of "fullpath" variables of each tool in cdw_tools[]

   Function walks through all tools in cdw_tools[] and prints to stderr all
   "fullpath" fields available.

   Purpose of this function is purely for debugging.
*/
void cdw_ext_tools_print_fullpaths(void)
{
	for (int t = 0; t < CDW_N_TOOLS; t++) {
		if (cdw_ext_tools.tools[t].available) {

			cdw_vdm ("fullpaths for %s tool:\n", cdw_ext_tools.tools[t].name);

			for (int i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {
				cdw_assert (cdw_ext_tools.tools[t].instances[i].fullpath != (char *) NULL,
					    "ERROR: fullpath %d is null\n", i);

				fprintf (stderr, "full path %d = \"%s\"", i, cdw_ext_tools.tools[t].instances[i].fullpath);
				if (cdw_ext_tools.tools[t].instances[i].target_path != (char *) NULL) {
					fprintf(stderr, " -> \"%s\"", cdw_ext_tools.tools[t].instances[i].target_path);
				}
				fprintf(stderr, "\n");
			}

			if (cdw_ext_tools.tools[t].current_instance_ind < cdw_ext_tools.tools[t].n_instances) {
				int ind = cdw_ext_tools.tools[t].current_instance_ind;
				fprintf(stderr, "path to selected instance is \"%s\"\n",
					cdw_ext_tools.tools[t].instances[ind].fullpath );
			} else if (cdw_ext_tools.tools[t].current_instance_ind == cdw_ext_tools.tools[t].n_instances) {
				fprintf(stderr, "path to selected instance is \"system default\"\n");
			} else {
				cdw_assert (0, "ERROR: cdw_ext_tools.tools[%d].current_item_index = %d is out of range\n",
					    t, cdw_ext_tools.tools[t].current_instance_ind);
			}

		} else {
			fprintf(stderr, "%s is not available, no fullpaths will be printed\n",
				cdw_ext_tools.tools[t].name);
		}
	}
	fprintf(stderr, "\n");

	return;
}





/**
   \brief Create strings for configuration dropdowns in which user can select tools

   Function creates for each tool in cdw_tools[] set of labels, that can
   be used in configuration window as labels for dropdowns, in which user
   can select one of instances of a tool. These strings are stored in
   cdw_tools[<tool id>].labels[]

   If a given tool is not available, then the only label is sth like
   "not available". If a given tool is available, the labels are very
   similar to fullpaths, the only difference is that if a fullpath is
   a symbolic link, the label also includes target of the link (from
   target_paths[] table).

   \return CDW_MEM_ERROR on memory allocation error
   \return CDW_OK on success
*/
cdw_rv_t cdw_ext_tools_build_dropdown_labels(void)
{
	for (int t = 0; t < CDW_N_TOOLS; t++) {

		if (cdw_ext_tools.tools[t].available) {
			cdw_assert (cdw_ext_tools.tools[t].instances[0].fullpath != (char *) NULL,
				    "ERROR: tool %s is available, but its first fullpath = NULL\n",
				    cdw_ext_tools.tools[t].name);


			int i = 0;
			for (i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {
				cdw_assert (cdw_ext_tools.tools[t].instances[i].fullpath != (char *) NULL,
					    "ERROR: fullpath to instance #%d of tool %d is NULL\n", i, t);

				if (cdw_ext_tools.tools[t].instances[i].target_path == (char *) NULL) {
					/* given path is not a symbolic link,
					   label will be like "/usr/bin/cdrecord */
					cdw_ext_tools.tools[t].instances[i].label = strdup(cdw_ext_tools.tools[t].instances[i].fullpath);
				} else {
					/* label will be like "/usr/bin/cdrecord (wodim),
					   where first part is a symlink, and second part
					   is a target file */
					cdw_ext_tools.tools[t].instances[i].label = cdw_string_concat(cdw_ext_tools.tools[t].instances[i].fullpath,
										    " (",
										    cdw_ext_tools.tools[t].instances[i].target_path,
										    ")",
										    (char *) NULL);
				}
				if (cdw_ext_tools.tools[t].instances[i].label == (char *) NULL) {
					cdw_vdm ("ERROR: failed to allocate memory for label (tool %d, instance #%d)\n", t, i);
					return CDW_MEM_ERROR;
				}
			}
			cdw_ext_tools.tools[t].instances[i].label = label_system_default;
		} else {
			cdw_ext_tools.tools[t].instances[0].label = label_not_available;
		}
	}

	return CDW_OK;
}





/**
   \brief Print values of variables in cdw_tools[] which will be used as dropdown labels

   Function walks through all tools in cdw_tools[] and prints to stderr all
   "labels" fields available. These fields can be used to create dropdowns
   in configuration window.

   Purpose of this function is purely for debugging.
*/
void cdw_ext_tools_print_labels(void)
{
	for (int t = 0; t < CDW_N_TOOLS; t++) {

		for (int i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {
			cdw_assert (cdw_ext_tools.tools[t].instances[i].fullpath != (char *) NULL,
				    "ERROR: fullpath to instance #%d of tool %d is NULL\n", i, t);
			fprintf(stderr, " + %s_labels[%d] = \"%s\"\n",
				cdw_ext_tools.tools[t].name, i, cdw_ext_tools.tools[t].instances[i].label);
		}
	}

	return;
}





/**
   \brief Check if given tool is available (accessible)

   Check if cdw found given tool (specified by \p tool_id), knows full path
   to the tool, and can use it to perform tasks.
   You can call this function only for

   \param tool_id - id of a tool (as defined in header file)

   \return true if given tool is available
   \return false if given tool is not available
*/
bool cdw_ext_tools_is_tool_available(enum cdw_tool tool_id)
{
	cdw_assert (tool_id >= CDW_TOOL_MKISOFS && tool_id <= CDW_TOOL_MD5SUM,
		    "ERROR: incorrect tool id: %d\n", tool_id);

	return cdw_ext_tools.tools[tool_id].available;
}





/**
   \brief Message informing that a tool can't be found

   Use the function to inform user that cdw can't find given tool (specified
   by \p tool_id), and because of this cdw can't perform required operation.

   If all goes well then tool name (indicated by \p tool_id) is used in
   dialog message, so that user knows which tool causes problems.
   However if something goes wrong, a message without the tool name is
   displayed - user doesn't know which tool causes problems, but still knows
   that there is a problem. In the first case CDW_OK is returned, in second
   case CDW_GEN_ERROR is returned.

   \param tool_id - id of a tool (as defined in header file)

   \return CDW_OK on success
   \return CDW_GEN_ERROR on problems
*/
cdw_rv_t cdw_ext_tools_error_dialog(enum cdw_tool tool_id)
{
	cdw_assert (tool_id >= CDW_TOOL_MKISOFS && tool_id <= CDW_TOOL_MD5SUM,
		    "ERROR: incorrect tool id: %d\n", tool_id);

	char *msg = (char *) NULL;

	/* 2TRANS: this is message printed into cdw log. %s is a tool name,
	   like cdrecord or growisofs; "Tools" is a label of one of tabs
	   in Configuration window; keep escaped quotes */
	cdw_logging_write(_("ERROR: cdw can't find following tool: \"%s\".\n"), cdw_ext_tools.tools[tool_id].name);

	/* 2TRANS: this is message in dialog window; %s is a tool name,
	   like cdrecord or growisofs; "Tools" is a label of one of tabs
	   in Configuration window; keep escaped quotes */
	int rv = asprintf(&msg, _("cdw can't proceed because it can't find this tool: %s. Please check your options (\"Tools\" tab)."),
			  cdw_ext_tools.tools[tool_id].name);

	if (rv == -1) {
		cdw_vdm ("ERROR: asprintf() returns -1\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      "tool" refers to external programs like cdrecord
				      or growisofs; keep escaped quotes */
				   _("Can't find some external tool. Please check your options (\"Tools\" tab)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_GEN_ERROR;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"), msg, CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		free(msg);
		msg = (char *) NULL;
		return CDW_OK;
	}
}





/**
   \brief Check if tool IDs weren't changed during development of cdw

   This function is a wrapper for a set of assertions, so it is one big
   assert() which doesn't do anything in release build. What it does is
   it checks that certain symbolic values didn't change during development
   of cdw, that they weren't accidentally modified by developer.

   This is important because these values are used to index Very Important
   Table. I'm not 100% sure that I use these values consistently in whole
   code base, this function checks that these values match assumptions
   that I made coding this module.

   If you run compiler with all warnings enabled (well, one of them for sure,
   can't remember now which one), and all variables have correct values,
   the compiler will warn you that each line in this function "will never be
   executed" - this is an early sign that everything is ok.

   It will be sufficient if you call this function once, when initializing
   this module.
*/
void cdw_ext_tools_check_tool_id_assertions(void)
{
	/* Probably this wouldn't be necessary if I used these values
	   consistently in this file and in whole project. For now I
	   will keep using it, and in future, when my code is more
	   polished, I will consider dropping it. But maybe I will keep
	   it because you never know when you introduce some change that
	   will break some assumptions */
	cdw_assert (CDW_TOOL_NONE == -1, "you changed CDW_TOOL_NONE value, told you not to do that!\n");

	cdw_assert (CDW_TOOL_MKISOFS == 0, "you changed CDW_TOOL_MKISOFS value, told you not to do that!\n");
	cdw_assert (CDW_TOOL_CDRECORD == 1, "you changed CDW_TOOL_CDRECORD value, told you not to do that!\n");
	cdw_assert (CDW_TOOL_GROWISOFS == 2, "you changed CDW_TOOL_GROWISOFS value, told you not to do that!\n");
	cdw_assert (CDW_TOOL_DVD_RW_FORMAT == 3, "you changed CDW_TOOL_DVD_RW_FORMAT value, told you not to do that!\n");
	cdw_assert (CDW_TOOL_DVD_RW_MEDIAINFO == 4, "you changed CDW_TOOL_DVD_RW_MEDIAINFO value, told you not to do that!\n");
	cdw_assert (CDW_TOOL_MD5SUM == 5, "you changed CDW_TOOL_MD5SUM value, told you not to do that!\n");

	return;
}




#if 0


/* for now it only works for growisofs */
cdw_rv_t cdw_ext_tools_check_versions(void)
{
	int i = 0;
	char *fullpath = cdw_ext_tools.tools[CDW_TOOL_GROWISOFS].instances[i].fullpath;

	while (fullpath != (char *) NULL) {
		char *command = cdw_string_concat(fullpath, " --version", (char *) NULL);
		if (command == (char *) NULL) {
			cdw_vdm ("ERROR: failed to create command for fullpath \"%s\"\n", fullpath);
			return CDW_GEN_ERROR;
		} else {
			/* need the task only for pipe regexp dispatcher,
			   and the dispatcher checks for some specific fields,
			   which we have to set here */
			cdw_task_t task;
			task.burn.tool.id = CDW_TOOL_GROWISOFS;
			task.id = CDW_TASK_BURN_FROM_IMAGE;

			run_command(command, &task);
			free(command);
			command = (char *) NULL;

			i++;
			fullpath = cdw_ext_tools.tools[CDW_TOOL_GROWISOFS].instances[i].fullpath;
		}
	}

	return CDW_OK;
}

#endif



void cdw_ext_tools_init_labels(void)
{
	/* 2TRANS: given tool is not available in user's environment;
	   no more than 19 chars */
	strncpy(label_not_available, _("(not available)"), 19);
	label_not_available[19] = '\0';

	/* 2TRANS: "system default" means "tool selected by default
	   in user's system";  no more than 19 chars */
	strncpy(label_system_default, _("(system default)"), 19);
	label_system_default[19] = '\0';

	/* 2TRANS: "no tool available" means "no tool that would meet some
	   specification is available"; no more than 19 chars */
	strncpy(label_no_tool_available, _("(no tool available)"), 19);
	label_system_default[19] = '\0';

	return;
}




void cdw_ext_tools_build_cd_family(void)
{
	/* current algorithm for selecting the best default family for CDs
	   is simple: cdrtools is always the best;
	   order of checking tool families availability for purposes
	   of creating "CD family" dropdown follows the same algorithm:
	   cdrtools is always the best, so check cdrtools
	   first; if the family exist then put it on top of dropdown */

	cdw_ext_tools.config.cd_family.default_id = CDW_TOOL_NONE;

	int n_items = 0;
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)) {
		/* cdrecord alone is enough to get meta info from CD disc,
		   write to CD disc and erase CD disc, so no need to check
		   availability of other members of cdrtools; if mkisofs is
		   missing then it will be signalled to user in different
		   place (if mkisofs will be needed at all for given action); */
		cdw_ext_tools.config.cd_family.items[n_items].label = cdw_ext_tools.tools[CDW_TOOL_CDRECORD].name;
		cdw_ext_tools.config.cd_family.items[n_items].id = CDW_TOOL_CDRECORD;
		n_items++;

		cdw_ext_tools.config.cd_family.default_id = CDW_TOOL_CDRECORD;
	}
	/*
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {
	        cdw_ext_tools.config.cd_family.items[n_items].label = cdw_ext_tools.tools[CDW_TOOL_XORRISO].name;
		cdw_ext_tools.config.cd_family.items[n_items].id = CDW_TOOL_XORRISO;
		n_items++;
		if (cdw_ext_tools.config.cd_family.default_id == CDW_TOOL_NONE) {
			// cdrtools were not set as default CD family
			cdw_ext_tools.config.cd_family.default_id = CDW_TOOL_XORRISO;
		}
	}
	*/

	if (n_items == 0) {
		/* there are no CD tool families available */
		cdw_ext_tools.config.cd_family.items[0].label = label_no_tool_available;
		cdw_assert (cdw_ext_tools.config.cd_family.default_id == CDW_TOOL_NONE,
			    "ERROR: zero CD families, but someone changed automatic family\n");
		n_items = 1;  /* the only item will be "not available" */
	}

	/* in all cases first label in dropdown is preselected by default */
	cdw_ext_tools.config.cd_family.current_family_ind = 0;
	cdw_ext_tools.config.cd_family.n_items = n_items;
	cdw_assert (cdw_ext_tools.config.cd_family.n_items <= CDW_EXT_TOOLS_N_FAMILIES_MAX,
		    "ERROR: collected %d families, but limit is %d\n",
		    cdw_ext_tools.config.cd_family.n_items, CDW_EXT_TOOLS_N_FAMILIES_MAX);

#ifndef NDEBUG
	for (int t = 0; t < cdw_ext_tools.config.cd_family.n_items; t++) {
		cdw_assert (cdw_ext_tools.config.cd_family.items[t].label != (char *) NULL,
			    "ERROR: label #%d of CD tool family is NULL\n", t);
		cdw_vdm ("INFO: CD tool family #%d label = \"%s\"\n",
			 t, cdw_ext_tools.config.cd_family.items[t].label);
	}
#endif

	return;
}





/* these are labels visible in configuration panel in "tools" card;
   they describe families of tools (tool packs) used for handling
   DVD discs; Currently cdw supports two families of tools that handle DVD discs:
   dvd+rw-tools and cdrtools; below I use "cdrecord" instead of cdrtools
   because "cdrecord" may be known better, and additionally I don't want
   to suggest to user that other programs from this tool pack (mkisofs, cdda2wav)
   are used or not for CD or DVD, depending on the selection */

void cdw_ext_tools_build_dvd_family(void)
{
	/* current algorithm for selecting the best family for DVDs is simple:
	   dvd+rw-tools is always the best;
	   order of checking tool families availability for purposes
	   of creating "DVD family" dropdown follows the same algorithm:
	   dvd+rw-tools is always the best, so check dvd+rw-tools
	   first; if the family exist then put it on top of dropdown */

	cdw_ext_tools.config.dvd_family.default_id = CDW_TOOL_NONE;

	int n_items = 0;
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_GROWISOFS)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_DVD_RW_MEDIAINFO)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_DVD_RW_FORMAT)) {

		/* only when all members of family are available, then the
		   family itself is available */

		cdw_ext_tools.config.dvd_family.items[n_items].label = dvd_rw_tools_label;
		/* ah! inconsistency! it should be CDW_TOOL_DVD_RW_TOOLS,
		   but I don't want to create tool label for tool family */
		cdw_ext_tools.config.dvd_family.items[n_items].id = CDW_TOOL_GROWISOFS;
		n_items++;

		cdw_ext_tools.config.dvd_family.default_id = CDW_TOOL_GROWISOFS;
	}
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)) {
		/* cdrecord alone is enough to get meta info from DVD disc,
		   write to DVD disc and erase DVD disc, so no need to check
		   availability of other members of cdrtools; if mkisofs is
		   missing then it will be signalled to user in different
		   place (if mkisofs will be needed at all for given action); */
		cdw_ext_tools.config.dvd_family.items[n_items].label = cdw_ext_tools.tools[CDW_TOOL_CDRECORD].name;
		cdw_ext_tools.config.dvd_family.items[n_items].id = CDW_TOOL_CDRECORD;
		n_items++;

		if (cdw_ext_tools.config.dvd_family.default_id == CDW_TOOL_NONE) {
			/* dvd-rw-tools were not set as default family */
			cdw_ext_tools.config.dvd_family.default_id = CDW_TOOL_CDRECORD;
		}
	}
	/*
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {
	        cdw_ext_tools.config.dvd_family.items[n_items].label = cdw_ext_tools.tools[CDW_TOOL_XORRISO].name;
		cdw_ext_tools.config.dvd_family.items[n_items].id = CDW_TOOL_XORRISO;
		n_items++;
		if (cdw_ext_tools.config.dvd_family.default_id == CDW_TOOL_NONE) {
		        cdw_ext_tools.config.dvd_family.default_id = CDW_TOOL_XORRISO;
		}
	}
	*/

	if (n_items == 0) {
		/* there are no DVD tool families available */
		cdw_ext_tools.config.dvd_family.items[0].label = label_no_tool_available;
		cdw_assert (cdw_ext_tools.config.dvd_family.default_id == CDW_TOOL_NONE,
			    "ERROR: zero DVD families, but someone changed automatic family\n");
		n_items = 1; /* the only item will be "not available" */
	}

	/* in all cases first label in dropdown is preselected by default */
	cdw_ext_tools.config.dvd_family.current_family_ind = 0;
	cdw_ext_tools.config.dvd_family.n_items = n_items;
	cdw_assert (cdw_ext_tools.config.dvd_family.n_items <= CDW_EXT_TOOLS_N_FAMILIES_MAX,
		    "ERROR: collected %d families, but limit is %d\n",
		    cdw_ext_tools.config.dvd_family.n_items, CDW_EXT_TOOLS_N_FAMILIES_MAX);

#ifndef NDEBUG
	for (int t = 0; t < cdw_ext_tools.config.dvd_family.n_items; t++) {
		cdw_assert (cdw_ext_tools.config.dvd_family.items[t].label != (char *) NULL,
			    "ERROR: label #%d of DVD tool family is NULL\n", t);
		cdw_vdm ("INFO: DVD tool family #%d label = \"%s\"\n",
			 t, cdw_ext_tools.config.dvd_family.items[t].label);
	}
#endif

	return;
}





const char *cdw_ext_tools_get_instance_manual(int tool_id)
{
	char *name = cdw_ext_tools.tools[tool_id].name;

	if (cdw_ext_tools.tools[tool_id].available) {
		/* select implementation */
		int ind = cdw_ext_tools.tools[tool_id].current_instance_ind;
		if (ind == cdw_ext_tools.tools[tool_id].n_instances) {
			/* user selected "system default" in dropdown */
			const char *fullpath = cdw_ext_tools_get_system_default_instance(tool_id);
			cdw_vdm ("INFO: selecting tool: sys default select \"%s\" of tool \"%s\"\n",
				 fullpath, name);
			return fullpath;
		} else {
			const char *fullpath = cdw_ext_tools.tools[tool_id].instances[ind].fullpath;
			cdw_vdm ("INFO: selecting tool: manual select #%d = \"%s\" of tool \"%s\"\n",
				 ind, fullpath, name);
			return fullpath;
		}
	} else {
		/* this tool should be available if "growisofs and family"
		   was put in "cdw_tools for dvd" table */
		if (tool_id == CDW_TOOL_GROWISOFS
		    || tool_id == CDW_TOOL_DVD_RW_MEDIAINFO
		    || tool_id == CDW_TOOL_DVD_RW_FORMAT) {

			int family = CDW_TOOL_GROWISOFS;
			cdw_vdm ("ERROR: tool \"%s\" is unavailable when family in manual selection = \"%s\"\n",
				 cdw_ext_tools.tools[tool_id].name, cdw_ext_tools.tools[family].name);

		} else if (tool_id == CDW_TOOL_CDRECORD) {
			int family = CDW_TOOL_CDRECORD;
			cdw_vdm ("ERROR: tool \"%s\" is unavailable when family in manual selection = \"%s\"\n",
				 cdw_ext_tools.tools[tool_id].name, cdw_ext_tools.tools[family].name);
		} else {
			cdw_vdm ("ERROR: tool \"%s\" is unavailable\n", cdw_ext_tools.tools[tool_id].name);
		}

		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      "tool" means "cdrecord", "growisofs", etc. */
				   _("Can't find any tool for current task, please check your configuration and system.\n"),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return (char *) NULL;
	}
}





const char *cdw_ext_tools_get_system_default_instance(int tool_id)
{
	return cdw_ext_tools.tools[tool_id].instances[0].fullpath;
}





bool cdw_ext_tools_config_is_manual(void)
{
	return cdw_ext_tools.config.manual_selection;
}





int cdw_ext_tools_config_get_dvd_family_id(void)
{
	if (cdw_ext_tools.config.manual_selection) {
		return cdw_ext_tools.config.dvd_family.items[cdw_ext_tools.config.dvd_family.current_family_ind].id;
	} else {
		return cdw_ext_tools.config.dvd_family.default_id;
	}
}





int cdw_ext_tools_config_get_cd_family_id(void)
{
	if (cdw_ext_tools.config.manual_selection) {
		return cdw_ext_tools.config.cd_family.items[cdw_ext_tools.config.cd_family.current_family_ind].id;
	} else {
		return cdw_ext_tools.config.cd_family.default_id;
	}
}





const char *cdw_ext_tools_get_tool_name(int tool_id)
{
	/* name of a tool is always defined, even if no implementation
	   of this tool is available */
	return cdw_ext_tools.tools[tool_id].name;
}





/* system default instance of a tool (if it is available in the system at all)
   is listed by "which" as first, and is put on list of available
   implementations as first */
#define CDW_TOOLS_SYSTEM_DEFAULT_INSTANCE 0





int cdw_ext_tools_get_current_instance_ind(int tool_id)
{
	const int sdi = CDW_TOOLS_SYSTEM_DEFAULT_INSTANCE;
	if (!cdw_ext_tools.tools[tool_id].available) {
		cdw_vdm ("ERROR: can't get tool instance label because tool is not available\n");
		return -1;
	}

	if (cdw_ext_tools_config_is_manual()) {
		/* user has selected manually one of tool instances
		   or allowed it to be "system default" */
		int ind = cdw_ext_tools.tools[tool_id].current_instance_ind;
		if (ind == cdw_ext_tools.tools[tool_id].n_instances) {
			/* user selected "system default" in dropdown */
			cdw_vdm ("INFO: manual selection, selecting system default instance \"%s\" of tool \"%s\"\n",
				 cdw_ext_tools.tools[tool_id].instances[sdi].fullpath, cdw_ext_tools.tools[tool_id].name);
			return sdi;
		} else {
			cdw_vdm ("INFO: manual selection, selecting instance #%d (%s) of tool \"%s\"\n",
				 ind, cdw_ext_tools.tools[tool_id].instances[ind].fullpath, cdw_ext_tools.tools[tool_id].name);
			return (int) ind;
		}
	} else {
		/* non-manual selection, cdw has to find by itself
		   which tool instance should be used; for now
		   cdw "intelligently" selects system default
		   implementation */
		cdw_vdm ("INFO: \"intelligent\" selection, selecting instance #%d (%s) of tool \"%s\"\n",
			 sdi, cdw_ext_tools.tools[tool_id].instances[sdi].fullpath, cdw_ext_tools.tools[tool_id].name);
		return sdi;
	}
}





char *cdw_ext_tools_get_current_instance_label(int tool_id)
{
	int ind = cdw_ext_tools_get_current_instance_ind(tool_id);
	if (ind == -1) {
		cdw_vdm ("ERROR: can't get tool instance index\n");
		return (char *) NULL;
	} else {
		return cdw_ext_tools.tools[tool_id].instances[ind].label;
	}
}





bool cdw_ext_tools_is_cdrecord_wodim(void)
{
	char *label = cdw_ext_tools_get_current_instance_label(CDW_TOOL_CDRECORD);
	if (label == (char *) NULL) {
		cdw_vdm ("ERROR: can't get tool instance label\n");
		return false;
	} else {
		char *c = strstr(label, "wodim");
		if (c == (char *) NULL) {
			cdw_vdm ("INFO: cdrecord instance IS NOT wodim\n");
			return false;
		} else {
			cdw_vdm ("INFO: cdrecord instance IS wodim\n");
			return true;
		}
	}
}





const char *cdw_ext_tools_get_instance_auto(int tool_id)
{
	const char *fullpath = cdw_ext_tools_get_system_default_instance(tool_id);
	cdw_vdm ("INFO: selecting tool: auto select \"%s\" of tool %s\n",
		 fullpath, cdw_ext_tools.tools[tool_id].name);
	return fullpath;
}





const char *cdw_ext_tools_get_tool_fullpath(int tool_id)
{
	if (cdw_ext_tools.config.manual_selection) {
		cdw_vdm ("INFO: selecting tool: manual selection\n");
		return cdw_ext_tools_get_instance_manual(tool_id);
	} else {
		cdw_vdm ("INFO: selecting tool: automatic selection\n");
		return cdw_ext_tools_get_instance_auto(tool_id);
	}
}





cdw_rv_t cdw_ext_tools_create_config_dropdowns(WINDOW *window, int row, int col, int width)
{
	ext_tools_dropdowns.dvd_family = cdw_dropdown_new(window, row, col, width,
							  cdw_ext_tools.config.dvd_family.n_items,
							  CDW_COLORS_DIALOG);
	for (int i = 0; i < cdw_ext_tools.config.dvd_family.n_items; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(ext_tools_dropdowns.dvd_family,
						     cdw_ext_tools.config.dvd_family.items[i].id,
						     cdw_ext_tools.config.dvd_family.items[i].label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item %d / \"%s\"\n", i, cdw_ext_tools.config.dvd_family.items[i].label);
		}
	}
	cdw_dropdown_finalize(ext_tools_dropdowns.dvd_family);
	cdw_dropdown_set_current_item_by_ind(ext_tools_dropdowns.dvd_family, (size_t) cdw_ext_tools.config.dvd_family.current_family_ind);



	ext_tools_dropdowns.mkisofs = cdw_ext_tools_tool_dropdown_new(window, row + 2, col, width, &cdw_ext_tools.tools[CDW_TOOL_MKISOFS]);
	ext_tools_dropdowns.cdrecord = cdw_ext_tools_tool_dropdown_new(window, row + 4, col, width, &cdw_ext_tools.tools[CDW_TOOL_CDRECORD]);
	ext_tools_dropdowns.growisofs = cdw_ext_tools_tool_dropdown_new(window, row + 6, col, width, &cdw_ext_tools.tools[CDW_TOOL_GROWISOFS]);
	ext_tools_dropdowns.dvd_rw_mediainfo = cdw_ext_tools_tool_dropdown_new(window, row + 8, col, width, &cdw_ext_tools.tools[CDW_TOOL_DVD_RW_MEDIAINFO]);
	ext_tools_dropdowns.dvd_rw_format = cdw_ext_tools_tool_dropdown_new(window, row + 10, col, width, &cdw_ext_tools.tools[CDW_TOOL_DVD_RW_FORMAT]);

	return CDW_OK;

}





CDW_DROPDOWN *cdw_ext_tools_tool_dropdown_new(WINDOW *window, int row, int col, int width, cdw_tool_t *tool)
{
	/* "tool->n_instances + 1" in lines below is either for "system
	   default" label if n_instances > 0, or for "not available" label
	   if n_instances == 0 */
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window, row, col, width, tool->n_instances + 1, CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown for \"%s\"\n", tool->name);
		return (CDW_DROPDOWN *) NULL;
	}
	for (int i = 0; i < tool->n_instances + 1; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown,
						     i, /* id == ind */
						     tool->instances[i].label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item %d / \"%s\" for \"%s\" dropdown\n",
				 i, tool->instances[i].label, tool->name);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}
	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize \"%s\" dropdown\n", tool->name);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
	}
	crv = cdw_dropdown_set_current_item_by_ind(dropdown, (size_t) tool->current_instance_ind);
	cdw_assert (crv == CDW_OK, "ERROR: failed to set current item of \"%s\" dropdown\n", tool->name);

	return dropdown;
}





void cdw_ext_tools_save_dropdown_states(void)
{
	cdw_ext_tools.config.dvd_family.current_family_ind = (int) cdw_dropdown_get_current_item_ind(ext_tools_dropdowns.dvd_family);
	cdw_ext_tools.tools[CDW_TOOL_MKISOFS].current_instance_ind = (int) cdw_dropdown_get_current_item_ind(ext_tools_dropdowns.mkisofs);
	cdw_ext_tools.tools[CDW_TOOL_CDRECORD].current_instance_ind = (int) cdw_dropdown_get_current_item_ind(ext_tools_dropdowns.cdrecord);
	cdw_ext_tools.tools[CDW_TOOL_GROWISOFS].current_instance_ind = (int) cdw_dropdown_get_current_item_ind(ext_tools_dropdowns.growisofs);
	cdw_ext_tools.tools[CDW_TOOL_DVD_RW_MEDIAINFO].current_instance_ind = (int) cdw_dropdown_get_current_item_ind(ext_tools_dropdowns.dvd_rw_mediainfo);
	cdw_ext_tools.tools[CDW_TOOL_DVD_RW_FORMAT].current_instance_ind = (int) cdw_dropdown_get_current_item_ind(ext_tools_dropdowns.dvd_rw_format);

	return;
}


void cdw_ext_tools_save_manual_selection_state(bool ms)
{
	cdw_ext_tools.config.manual_selection = ms;
	return;
}



bool cdw_ext_tools_get_manual_selection_state(void)
{
	return cdw_ext_tools.config.manual_selection;
}

