#ifndef H_CDW_CONFIG
#define H_CDW_CONFIG

#include <stdbool.h>
#include <stddef.h> /* size_t */

#include "main.h"

/** \brief Maximal length of some char tables storing long option values */
#define OPTION_FIELD_LEN_MAX 950

/* does not include ending '\0' */
#define PADSIZE_FIELD_LEN_MAX 4

/** \brief Maximal length of CD/DVD/iso label, does not include ending '\0'; */
#define VOLUME_ID_LEN_MAX 32 /* the value is taken from mkisofs man page */


int cdw_config_module_init(void);
void cdw_config_module_clean(void);

cdw_rv_t cdw_config_read_from_file(void);
cdw_rv_t cdw_config_write_to_file(void);


/* volume size - maximum size of selected files, used
   in "selected files info" view in main ui window; it is tempting
   to use "disc type" from cdw_disc.h, but it's not that simple to
   unify the two sets */
enum {
	CDW_CONFIG_VOLUME_SIZE_CD74 = 0,
	CDW_CONFIG_VOLUME_SIZE_CD80,
	CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC, /* smallest common denominator for DVDs */
	CDW_CONFIG_VOLUME_SIZE_DVD_R,
	CDW_CONFIG_VOLUME_SIZE_DVD_RP,
	CDW_CONFIG_VOLUME_SIZE_DVD_RW,
	CDW_CONFIG_VOLUME_SIZE_DVD_RWP,
	CDW_CONFIG_VOLUME_SIZE_DVD_R_DL,
	CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,
	CDW_CONFIG_VOLUME_SIZE_CUSTOM,
	CDW_CONFIG_VOLUME_SIZE_AUTO };


/** \brief Pair of name-value strings that make an option */
typedef struct {
	char *name; /**< \brief Name of option, string that stands on left-hand side of '=' char in configuration file*/
	char *value; /**< \brief Value of option, string that stands on right-hand side of '=' char in configuration file */
} cdw_option_t;

/* this is in header file only because unit testing code requires prototype */
cdw_option_t cdw_config_split_options_line(char *line);



/** \brief Main cdw configuration variable, storing all major options used by cdw */
typedef struct {

	/* page A - writing */

	/* these two fields are obsolete, remove in future */
	char dao[2];         /**< \brief Controls mode of writing of track to disc */
	int depreciated_speed;

	bool pad;            /**< \brief Pad written track with blanks (cdrecord) */
	int pad_size;
	int erase_mode;     /**< \brief Select mode of blanking a disc: fast / all */
	bool eject;          /**< \brief Should cdw eject drive tray after finished writing or blanking? */
	int speed_range;     /**< \brief See enum speed_ranges below */
	bool burnproof;      /**< \brief Should a tool use technique that helps avoiding buffer underrun? */
	bool dummy;          /**< \brief Perform dummy write */
	char other[OPTION_FIELD_LEN_MAX + 1]; /**< \brief Old version of variable, kept for backward compatibility */
	char *other_cdrecord_options;     /**< \brief Place for other cdrecord options, specified manually */
	char *other_growisofs_options; /**< \brief Place for other growisofs options, specified manually */



	/* page B - hardware */

	/* device name in form of /dev/xxx - it is used as default device
	   name in cdrecord, but SCSI descriptor can be also used if needed;
	   custom_device is also used in other tools code - these tools use
	   only this entry to access reading/writing device (I don't mention
	   here cddb nor legacy cdda2wav code, because it is currently
	   unmaintained) */
	char custom_drive[OPTION_FIELD_LEN_MAX + 1]; /**< \brief Path to CD/DVD reader/burner */
	/* state of "drive" dropdown in configuration window */
	char selected_drive[OPTION_FIELD_LEN_MAX + 1];
	/* SCSI device (reader/writer) descriptor in form of X:Y:Z;
	   it should be used only in cdrecord code, and user should set this
	   string only in cases when cdrecord can't work with /dev/xxx device */
	char scsi[OPTION_FIELD_LEN_MAX + 1];        /**< \brief Variable specifying scsi device in form preferred by cdrecord: bus:target:lun */
	/* currently unused, probably was used as device name
	   of read-only or secondary drive */
	char cdrom[OPTION_FIELD_LEN_MAX + 1];
	/* currently unused in maintained code */
	char mountpoint[OPTION_FIELD_LEN_MAX + 1];


	/* page C - audio */
	char *audiodir;



	/* page D - iso filesystem */
	/* with size limit (max number of chars, without ending '\0') = VOLUME_ID_LEN_MAX */
	char *volumeid;         /**< \brief Label of disc visible in file managers */
	bool showvol;           /**< \brief Show dialog asking for volume ID before creating iso image or burning files to disc */
	int iso_level;          /**< \brief Level of conformance with ISO9660; valid values are 1 to 4 */
	bool joliet;            /**< \brief Turn on creting Joliet filenames (-J, mkisofs option) */
	bool rockridge;         /**< \brief Use "normal" RR attributes, -R (mkisofs) */
	bool useful_rr;         /**< \brief Use rationalized (useful) RR attributes, -r (mkisofs) */
	bool joliet_long;       /**< \brief "Allow  Joliet  filenames  to  be  up to 103 Unicode characters, instead of 64." */
	bool follow_symlinks;   /**< \brief Follow symbolic links when generating the filesystem */

	char *iso_image_full_path;     /**< \brief Current full path to ISO9660 image file */
	char *boot_disc_options;       /**< \brief Options for creating a bootable disc */
	char *other_mkisofs_options;   /**< \brief Place for other mkisofs options, specified manually */



	/* page E: external tools options */
	/* external tools have their own configuration variable, and there
	   are no settings (yet) that should be stored in config file */



	/* page F: log file settings */
	char *log_full_path; /**< \brief Full path to cdw log file */
	bool showlog;        /**< \brief Show cdw log after finishing operation */



	/* other */
	/* these two are read and written to config file */
        int      volume_size_id;             /**< \brief ID of preferred standard size of target media/iso file */
	long int volume_size_custom_value;   /**< \brief Custom size of target media/iso file, MB */
	/* this one should be most often used by code not related to managing configuration */
	long int volume_size_value;          /**< \brief Preferred standard size of target media/iso file */



	/* ** audio - unused ** */

	char autodic[2];
	char bitsperchn[3];
	char stereo[2];
	char echosound[2];
	char encode[2];
	char lame[2];
	char highq[2];
	char bitrate[4];
	char cdbhost[255];
	char cdbuser[20];
	char sqlite_file[256];


	/* options not displayed in configuration window and not saved to config file */
	bool support_dvd_rp_dl;
	bool show_dvd_rw_support_warning;

} cdw_config_t;



cdw_rv_t cdw_config_var_copy(cdw_config_t *dest, cdw_config_t *src);
cdw_rv_t cdw_config_var_validate(cdw_config_t *config, int *page, int *field);


cdw_rv_t cdw_config_var_init_fields(cdw_config_t *config);
cdw_rv_t cdw_config_var_free_fields(cdw_config_t *config);

bool cdw_config_has_scsi_device(void);
const char *cdw_config_get_custom_drive(void);

void cdw_config_debug_print_config(cdw_config_t *config);

long int cdw_config_volume_size_by_id(int id);
int cdw_config_volume_id_by_label(const char *label);

/** \brief Writing speed ranges that user can preselect in configuration window */
enum speed_ranges {
	SPEED_RANGE_AUTO, /**< \brief Default value read from drive */
	SPEED_RANGE_LOWEST, /**< \brief Lowest value available for given disc */
	SPEED_RANGE_MIDDLE,  /**< \brief In-between value available for given disc */
	SPEED_RANGE_HIGHEST /**< \brief Highest value available for given disc */
};


/* unit tests */

void cdw_config_run_tests(void);

#endif /* H_CDW_CONFIG */
