#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>

#include <qbutton.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qdial.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qlcdnumber.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qsize.h>
#include <qslider.h>
#include <qtabwidget.h>

#include "ChannelEditorDlg.h"
#include "VideoDeviceInput.h"
#include "VideoDeviceLinux.h"
 #include "VideoDeviceInput.h"
#include "VideoSettingsDlg.h"
#include "pwc-ioctl.h"

static short int_sizes[7][2] =
{
  { 128,  96 },
  { 160, 120 },
  { 176, 144 },
  { 320, 240 },
  { 352, 288 },
  { 640, 480 },
  {  -1,  -1 }
};

static int frames[8] =  { 4, 5, 8, 10, 15, 20, 25, 30 };

#define MAX_TUNER_BUTTONS 4


/**
  \brief Constructor of Linux video settings dialog
  \param video Opened video device


*/
CVideoSettingsDlg::CVideoSettingsDlg(CVideoDeviceLinux *video)
        : CVideoSettings(0, "CVideoSettingsDlg"), m_AutoTimer(this)
{
   pVideo = video;
   m_WBMode = -1;
   m_WBRed = m_WBBlue = 0;
   m_Fps = m_OrgFps = 0;
   m_InputTunerDone = false;
   m_Philips = false;

   // Initialization is delayed until the device is opened.
   connect(pVideo, SIGNAL(Opened()), SLOT(Init()));
   // misc. stuff
   connect(pVideo, SIGNAL(Closed()), SLOT(reject()));
   connect(&m_AutoTimer, SIGNAL(timeout()), SLOT(UpdateAuto()));
}

CVideoSettingsDlg::~CVideoSettingsDlg()
{
  // Nothing to see here...
}

// private

void CVideoSettingsDlg::GetCamSizeFrame(bool First)
{
   int ifs;
   QSize ns;

   ns = pVideo->GetSize();
   ifs = pVideo->GetFramerate();
   if (First) {
     org_size = ns;
     m_OrgFps  = ifs;
   }
   SizeChanged(ns);
   FramerateChanged(ifs);
}

void CVideoSettingsDlg::Init()
{
   int i, n, c;
   QString vidname;

   /* Size & Framerate */
   // Find maximum size, set label on button if larger than 640x480,
   // then disable all radio buttons that are too large
   m_MaxSize = pVideo->GetMaxSize();
   m_MinSize = pVideo->GetMinSize();
   for (i = 0; i < 6; i++) {
      sizes[i].setWidth(int_sizes[i][0]);
      sizes[i].setHeight(int_sizes[i][1]);
   }
   if (m_MaxSize.width() > 640 || m_MaxSize.height() > 480) {
     sizes[6] = m_MaxSize;
     MaxSizeRadioButton->setText(tr("Max (%1x%2)").arg(m_MaxSize.width()).arg(m_MaxSize.height()));
   }
   else
     MaxSizeRadioButton->setEnabled(false);
   for (i = 0; i < 7; i++) {
      if (m_MaxSize.width()  < sizes[i].width() ||
          m_MaxSize.height() < sizes[i].height())
        SizeSelection->find(i)->setEnabled(false);
      if (m_MinSize.width()  > sizes[i].width() ||
          m_MinSize.height() > sizes[i].height())
        SizeSelection->find(i)->setEnabled(false);
   }
   FramerateSelection->setEnabled(pVideo->HasFramerate());
   GetCamSizeFrame(true);

   /* brightness, contrast, gamma, saturation & hue sliders */
   i = pVideo->GetBrightness();
   if (i == 0xffff)
     m_BrightnessSlider->setDisabled(true);
   else
     m_BrightnessSlider->setValue(i);
   i = pVideo->GetContrast();
   if (i == 0xffff)
     m_ContrastSlider->setDisabled(true);
   else
     m_ContrastSlider->setValue(i);
   i = pVideo->GetWhiteness();
   if (i == 0xffff)
     m_GammaSlider->setDisabled(true);
   else
     m_GammaSlider->setValue(i);
   i = pVideo->GetColour();
   if (i == 0xffff)
     m_SaturationSlider->setDisabled(true);
   else
     m_SaturationSlider->setValue(i);
   i = pVideo->GetHue();
   if (i == 0xffff)
     m_HueSlider->setDisabled(true);
   else
     m_HueSlider->setValue(i);

   if (m_InputTunerDone)
     return;

   /* input selection */
   // Make sure we get nice columns in the boxes
   m_InputSelection->setColumnLayout(1, Horizontal);
   m_TunerSelection->setColumnLayout(1, Horizontal);
   // Determine number of inputs, set names in "Input selector", possibly
   // find out which is the current tuner.
   pTuner = 0;
   n = pVideo->GetNumberOfInputs();
   for (i = 0; i < n; i++) {
      CVideoDeviceInput *input = 0;
      QRadioButton *button = 0;

      input = pVideo->GetVideoInput(i);
      if (input == 0) {
        qWarning("Could not get input %d of video device", i);
      }
      else {
        button = new QRadioButton(input->GetName(), m_InputSelection);
        m_InputSelection->insert(button, i);
      }
   }
   m_InputSelection->setButton(pVideo->GetInput());

   n = pVideo->GetNumberOfTuners();
   for (i = 0; i < n; i++) {
      CVideoDeviceTuner *tuner = 0;
      QRadioButton *button = 0;

      tuner = pVideo->GetVideoTuner(i);
      if (tuner == 0) {
        qWarning("Could not get tuner %d of video device", i);
      }
      else {
        button = new QRadioButton(tuner->GetName(), m_TunerSelection);
        m_TunerSelection->insert(button, i);
      }
   }
   m_TunerSelection->setButton(pVideo->GetTuner());

   /* Try to find out if this is a Philips cam */
   m_Philips = false;
   vidname = pVideo->GetIntfName();
   if (vidname.find("Philips") == 0 && vidname.find("webcam") > 0)
     m_Philips = true;
   else {
     struct pwc_probe probe;

     memset(&probe, 0, sizeof(probe));
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCPROBE, &probe) == 0) {
       if (vidname == probe.name)
         m_Philips = true;
     }
   }

   if (m_Philips) {
     int agc, contour, compr, backlight, flicker, dynnoise;
     struct pwc_whitebalance wb;
     struct pwc_wb_speed wbs;
     struct pwc_mpt_range range;
     struct pwc_mpt_angles cur_angles;

     qDebug("Philips webcam detected, enabling extensions");

     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGAGC, &agc) < 0) {
       qDebug("No VIDIOCPWCGAGC");
       AGCBox->setEnabled(false);
       AGCValue->setEnabled(false);
     }
     else {
       AGCBox->setChecked(agc < 0);
       AGCValue->setValue(abs(agc) / 256);
     }

     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGCONTOUR, &contour) < 0) {
       qDebug("No VIDIOCPWCGCONTOUR");
       ContourBox->setEnabled(false);
       ContourValue->setEnabled(false);
     }
     else {
       ContourBox->setChecked(contour < 0);
       if (contour >= 0)
         ContourValue->setValue(contour / 256);
     }

     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGBACKLIGHT, &backlight) < 0) {
       qDebug("No VIDIOCPWCGBACKLIGHT");
       BacklightBox->setEnabled(false);
     }
     else
       BacklightBox->setChecked(backlight != 0);

     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGFLICKER, &flicker) < 0) {
       qDebug("No VIDIOCPWCGFLICKER");
       FlickerBox->setEnabled(false);
     }
     else
       FlickerBox->setChecked(flicker != 0);

     ShutterBox->setChecked(true);


     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGDYNNOISE, &dynnoise) < 0) {
       qDebug("No VIDIOCPWCGDYNNOISE");
       m_NoiseReduction->setEnabled(false);
     }
     else
       m_NoiseReduction->setCurrentItem(dynnoise);

     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGCQUAL, &compr) < 0) {
       qDebug("VIDIOCPWCGCQUAL failed");
       m_CompressionSelection->setEnabled(false);
     }
     else
       m_CompressionSelection->setCurrentItem(compr);

     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGAWB, &wb) < 0) {
       m_WhiteBalance->setEnabled(false);
       m_RedDial->setEnabled(false);
       m_BlueDial->setEnabled(false);
     }
     else {
       m_WhiteBalance->setEnabled(true);
       m_WhiteBalance->setCurrentItem(wb.mode);
       ChangedWB(wb.mode);
     }
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGAWBSPEED, &wbs) < 0) {
       m_WBSpeed->setEnabled(false);
       m_WBDelay->setEnabled(false);
     }

     /* Try pan/tilt options */
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCMPTGRANGE, &range) < 0) {
       Tabs->removePage(PanTiltTab);
     }
     else {
       PanDial->setMinValue(range.pan_min / 100);
       PanDial->setMaxValue(range.pan_max / 100);
       TiltDial->setMinValue(range.tilt_min / 100);
       TiltDial->setMaxValue(range.tilt_max / 100);
       if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCMPTGANGLE, &cur_angles) == 0) {
         PanDial->setValue(cur_angles.pan / 100);
         TiltDial->setValue(cur_angles.tilt / 100);
       }
     }
   }
   else {
     qDebug("No Philips webcam detected, removing extension tab");
     Tabs->removePage(PhilipsTab);
     Tabs->removePage(PanTiltTab);
   }
   m_InputTunerDone = true;
}



// private slots

/**
  \brief Pick up size changes (even if we do it ourselves!)
*/
void CVideoSettingsDlg::SizeChanged(const QSize &ns)
{
   int i, sz;

qDebug("CVideoSettingsDlg::SizeChanged(%dx%d)", ns.width(), ns.height());

   // Determine image size (this better be exact)
   sz = -1;
   for (i = 0; i < 7; i++) {
      if (ns.width()  == sizes[i].width() &&
          ns.height() == sizes[i].height()) {
        sz = i;
        break;
      }
   }
   // Set radio button group
   if (sz != -1)
     SizeSelection->setButton(sz);
   if (ns != size)
     size = ns;
}

void CVideoSettingsDlg::FramerateChanged(int ifs)
{
   int i, fs, nfs, diff, dev;

qDebug("CVideoSettingsDlg::FramerateChanged(%d)", ifs);
   // Determine closest matching framerate
   fs = -1;
   nfs = -1;
   dev = 9999;
   for (i = 0; i < 8; i++) {
      diff = abs(frames[i] - ifs);
      if (diff < dev) { // Closest so far...
        dev = diff;
        nfs = frames[i];
        fs = i;
      }
   }
   if (fs != -1)
     FramerateSelection->setButton(fs);
   if (nfs != m_Fps)
     m_Fps = nfs;
}

/**
  This is called once a second to update the sliders in the Philips section
*/
void CVideoSettingsDlg::UpdateAuto()
{
   int agc;
   struct pwc_whitebalance wb;

   if (m_Philips) {
     if (AGCBox->isChecked()) {
       if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGAGC, &agc) == 0) {
         AGCValue->setValue(-agc / 256);
       }
     }
     if (m_WBMode == PWC_WB_AUTO) {
       if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGAWB, &wb) == 0) {
         m_RedDial->setValue(wb.read_red / 256);
         m_BlueDial->setValue(wb.read_blue / 256);
       }
     }
   }
}





// protected


/// We broadcast the fact that the dialog got hidden, to update the toolbar.
void CVideoSettingsDlg::hideEvent(QHideEvent *e)
{
   m_AutoTimer.stop();
   emit DialogClosed();
}

void CVideoSettingsDlg::showEvent(QShowEvent *e)
{
   if (m_Philips)
     m_AutoTimer.start(1000);
}


// protected slots
/// Called when the user clicked on a Size radio button
void CVideoSettingsDlg::ClickedSize(int s)
{
   if (pVideo == NULL || s < 0 || s > 6)
     return;

   /* Try to set the cam to the desired size; this may trigger a SizeChanged() signal */
   pVideo->SetSize(sizes[s]);
   GetCamSizeFrame();
}

/// Called when the user clicked on a Framerate radio button
void CVideoSettingsDlg::ClickedRate(int f)
{
   if (pVideo == NULL || f < 0 || f > 7)
     return;
   /* Leave flags intact */
   pVideo->SetFramerate(frames[f]);
   GetCamSizeFrame();
}

void CVideoSettingsDlg::MovedBrightness(int value)
{
   if (pVideo == NULL)
     return;
   pVideo->SetBrightness(value);
}

void CVideoSettingsDlg::MovedContrast(int value)
{
   if (pVideo == NULL)
     return;
   pVideo->SetContrast(value);
}

void CVideoSettingsDlg::MovedGamma(int value)
{
   if (pVideo == NULL)
     return;
   pVideo->SetWhiteness(value);
}

void CVideoSettingsDlg::MovedSaturation(int value)
{
   if (pVideo == NULL)
     return;
   pVideo->SetColour(value);
}

void CVideoSettingsDlg::MovedHue(int value)
{
   if (pVideo == NULL)
     return;
   pVideo->SetHue(value);
}


// 3rd group

void CVideoSettingsDlg::ClickedInputSelector(int input)
{
   int i, n, c;
   CVideoDeviceInput *pVI;
   CVideoDeviceTuner *pVT;
   QButton *pB;

qDebug("CVideoSettingsDlg::ClickedInputSelector(%d)", input);
   if (pVideo == NULL)
     return;
   pVideo->SetInput(input);

   pVI = pVideo->GetVideoInput(input);
   if (pVI == 0)
     return;
#if 0
   /* determine number of available tuners  */
   pTuner = 0;
   n = pVI->GetNumberOfTuners();
   c = pVI->GetCurrentTuner();

   for (i = 0; i < MAX_TUNER_BUTTONS; i++) {
      pB = m_TunerSelection->find(i);
      if (pB == 0)
        continue;

      if (i < n) {
        pVT = pVI->GetTuner(i);
        if (pVT != 0) {
          pB->setText(pVT->GetName());
          pB->setEnabled(true);
          pB->show();
        }
      }
      else {
        pB->setEnabled(false);
        pB->hide();
      }
   }
   if (n == 1) // If we have only one tuner, might as well select it
   {
     ClickedTunerSelector(0);
     c = 0;
   }
   m_TunerSelection->setButton(c);

   m_TunerSelection->setEnabled(n > 0);
   m_ChannelSelector->setEnabled(n > 0);
#endif
}

void CVideoSettingsDlg::ClickedTunerSelector(int tuner)
{
   CVideoDeviceInput *pVI;

qDebug("CVideoSettingsDlg::ClickedTunerSelector(%d)", tuner);
   if (pVideo == 0)
     return;
   pVideo->SetTuner(tuner);
}

void CVideoSettingsDlg::ClickedChannelSetup()
{
   if (pVideo != 0) {
     pVideo->ShowTunerDialog();
   }
}

void CVideoSettingsDlg::ToggledAGC(bool on)
{
   AGCValue->setEnabled(!on);
   if (on) {
     int agc = -1;
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAGC, &agc) < 0)
       perror("VIDIOCPWCSAGC");
   }
}

void CVideoSettingsDlg::MovedAGC(int value)
{
   int agc;

   if (!AGCBox->isChecked()) {
     agc = value * 256;
     ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAGC, &agc);
   }
}

void CVideoSettingsDlg::ToggledShutter(bool on)
{
   if (on)
     ShutterValue->hide();
   else
     ShutterValue->show();
   if (on) {
     int speed = -1;
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSSHUTTER, &speed) < 0)
       perror("VIDIOCPWCSSHUTTER");
   }
}

void CVideoSettingsDlg::MovedShutter(int value)
{
   int speed;

   if (!ShutterBox->isChecked()) {
     speed = value * 256;
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSSHUTTER, &speed) < 0)
       perror("VIDIOCPWCSSHUTTER");
   }
}

void CVideoSettingsDlg::ToggledContour(bool on)
{
   if (on)
     ContourValue->hide();
   else
     ContourValue->show();
   if (on) {
     int contour = -1;
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSCONTOUR, &contour) < 0)
       perror("VIDIOCPWCSCONTOUR");
   }
}

void CVideoSettingsDlg::MovedContour(int value)
{
   int contour;

   if (!ContourBox->isChecked()) {
     contour = value * 256;
     if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSCONTOUR, &contour) < 0)
       perror("VIDIOCPWCSCONTOUR");
   }
}

void CVideoSettingsDlg::ToggledBacklight(bool on)
{
   int backlight;
   backlight = on ? 1 : 0;
   qDebug("ToggledBacklight to %d", backlight);
   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSBACKLIGHT, &backlight) < 0)
     perror("VIDIOCPWCSBACKLIGHT");
}

void CVideoSettingsDlg::ToggledFlicker(bool on)
{
   int flicker;
   flicker = on ? 1 : 0;
   qDebug("ToggledFlicker to %d", flicker);
   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSFLICKER, &flicker) < 0)
     perror("VIDIOCPWCSFLICKER");
}


void CVideoSettingsDlg::ChangedNoise(int noise)
{
   qDebug("ChangedNoise to %d", noise);
   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSDYNNOISE, &noise) < 0)
     perror("VIDIOCPWCSDYNNOISE");
}

void CVideoSettingsDlg::ChangedCompression(int compression)
{
   qDebug("ChangedCompression to %d", compression);
   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSCQUAL, &compression) < 0)
     perror("VIDIOCPWCSCQUAL");
}

/**
  \brief Called when user chooses different white balance

  This function will modify the dialog. When using a preset (indooor/outdoor),
  the red and blue dials are hidden. In auto mode, the dials track the
  values as reported by the camera. In manual mode, the dials can be set
  by the user.
*/
void CVideoSettingsDlg::ChangedWB(int new_mode)
{
   struct pwc_whitebalance wb;

   m_WBMode = new_mode;
   wb.mode = m_WBMode;
   wb.manual_red = m_RedDial->value() * 256;
   wb.manual_blue = m_BlueDial->value() * 256;
   ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAWB, &wb);

   if (m_WBMode == PWC_WB_MANUAL || m_WBMode == PWC_WB_AUTO) {
     m_RedDial->show();
     m_BlueDial->show();
     m_RedDial->setEnabled(m_WBMode == PWC_WB_MANUAL);
     m_BlueDial->setEnabled(m_WBMode == PWC_WB_MANUAL);
     if (m_WBMode == PWC_WB_MANUAL) {
       m_RedDial->setValue(wb.manual_red / 256);
       m_BlueDial->setValue(wb.manual_blue / 256);
     }
     else {
       if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCGAWB, &wb) < 0)
         perror("VIDIOCPWCGAWB 3");
       m_RedDial->setValue(wb.read_red / 256);
       m_BlueDial->setValue(wb.read_blue / 256);
     }
   }
   else {
     m_RedDial->hide();
     m_BlueDial->hide();
   }
}

void CVideoSettingsDlg::MovedRed(int red)
{
   struct pwc_whitebalance wb;

   wb.mode = m_WBMode;
   wb.manual_red = m_RedDial->value() * 256;
   wb.manual_blue = m_BlueDial->value() * 256;
   ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAWB, &wb);
}

void CVideoSettingsDlg::MovedBlue(int blue)
{
   struct pwc_whitebalance wb;

   wb.mode = m_WBMode;
   wb.manual_red = m_RedDial->value() * 256;
   wb.manual_blue = m_BlueDial->value() * 256;
   ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAWB, &wb);
}

void CVideoSettingsDlg::MovedWBSpeed(int speed)
{
   struct pwc_wb_speed wbs;

   /* The speed is reversed... the more to the right, the lower the value should be */
   wbs.control_speed = (255 - speed) << 8;
   wbs.control_delay = 0;

   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAWBSPEED, &wbs) < 0)
     perror("VIDIOCPWCSAWBSPEED speed");
}

void CVideoSettingsDlg::MovedWBDelay(int delay)
{
   struct pwc_wb_speed wbs;

   wbs.control_speed = 0;
   wbs.control_delay = delay << 8;

   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCSAWBSPEED, &wbs) < 0)
     perror("VIDIOCPWCSAWBSPEED delay");
}

void CVideoSettingsDlg::ClickedSaveUser()
{
   ioctl(pVideo->GetDescriptor(), VIDIOCPWCSUSER);
}

void CVideoSettingsDlg::ClickedRestoreUser()
{
   ioctl(pVideo->GetDescriptor(), VIDIOCPWCRUSER);
}

void CVideoSettingsDlg::ClickedRestoreFactory()
{
   ioctl(pVideo->GetDescriptor(), VIDIOCPWCFACTORY);
}


// 5th group

void CVideoSettingsDlg::SetPanTiltAngle()
{
   struct pwc_mpt_angles Angles;

   Angles.absolute = 1;
   Angles.pan  = PanDial->value() * 100;
   Angles.tilt = TiltDial->value() * 100;
   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCMPTSANGLE, &Angles) < 0)
     perror("VIDIOCPWCMPTSANGLE");
}

void CVideoSettingsDlg::ResetPanTilt()
{
   int reset_flags;

   reset_flags = PWC_MPT_PAN | PWC_MPT_TILT;
   if (ioctl(pVideo->GetDescriptor(), VIDIOCPWCMPTRESET, &reset_flags) < 0)
     perror("VIDIOCPWCMPTRESET");
   else {
     PanDial->setValue(0);
     TiltDial->setValue(0);
   }
}

// public

void CVideoSettingsDlg::SetConfiguration(const QDomNode &node)
{
   m_ConfigurationXML = node;
}

QDomNode CVideoSettingsDlg::GetConfiguration(QDomDocument &dom_doc) const
{
   return m_ConfigurationXML;
}

void CVideoSettingsDlg::ShowDisplayTab()
{
   Tabs->showPage(DisplayTab);
   show();
}

void CVideoSettingsDlg::ShowFormatTab()
{
   Tabs->showPage(FormatTab);
   show();
}

void CVideoSettingsDlg::ShowSourceTab()
{
   Tabs->showPage(SourceTab);
   show();
}

