/* 
   CAM - Cpu's Audio Mixer for Linux 
   (c) 1994-1996 
   AUTHOR: Jan 'TWP' VANDENBERGHE (jvdbergh@uia.ua.ac.be) 

 */

/* 
   This program is free software; you can redistribute it and/or 
   modify it under the terms of the GNU General Public License 
   as published by the Free Software Foundation; either version 2 
   of the License, or (at your option) any later version. 

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of 
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details. 

   You should have received a copy of the GNU General Public License 
   along with this program; if not, write to the Free Software 
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */


#include "cam.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>

FILE *trt;

#define DIV (((400<<8)/COLS)+5)
#define SPACE (COLS/18)
#define LPOS (SPACE+14)
#define RPOS (((COLS/4)+1)+SPACE)

extern jmp_buf env;

int mixer_fd, recmask, devmask, stereodevs, recsrc;
char *devname[DEV_COUNT] = SOUND_DEVICE_LABELS;
char *mixer_dev = "/dev/mixer";
struct devices
  {
    int vol_left, vol_right;
    int dev_nr;
    int X;
    int Y;
  }
dev_vol[DEV_COUNT];
int dev_count = 0, dev_ind[DEV_COUNT];
int a3dse = 2, agc = 0;

void
close_mixer (void)
{
  if (close (mixer_fd) != 0)
    fprintf (stderr, "Error closing device %s", mixer_dev);
}

/* gets n bits from x starting from position p */
int
getbits (int x, int p, int n)
{
  return (x >> (p + 1 - n)) & ~(~0 << n);
}

/* sets the volume of a given device */
void
adjust (int dev_nr, int left, int right)
{
  int temp = 0;
  if (left == -1)
    return;
  if (right == -1)
    temp = left;
  else
    temp = left + (right << 8);
  if (ioctl (mixer_fd, MIXER_WRITE (dev_nr), &temp) == -1)
    {
      finish (0);
      fprintf (stderr, "MIXER_WRITE : Can't write to %s !", mixer_dev);
      exit (1);
    }

}


/* evaluates an argument */
int
eval_argv (char *arg)
{
  if ((strcmp (arg, "-v") == 0) || (strcmp (arg, "--volume") == 0))
    return 0;
  else if ((strcmp (arg, "-b") == 0) || (strcmp (arg, "--bass") == 0))
    return 1;
  else if ((strcmp (arg, "-t") == 0) || (strcmp (arg, "--treble") == 0))
    return 2;
  else if ((strcmp (arg, "-s") == 0) || (strcmp (arg, "--synth") == 0))
    return 3;
  else if ((strcmp (arg, "-p") == 0) || (strcmp (arg, "--pcm") == 0))
    return 4;
  else if ((strcmp (arg, "-S") == 0) || (strcmp (arg, "--speaker") == 0))
    return 5;
  else if ((strcmp (arg, "-l") == 0) || (strcmp (arg, "--line") == 0))
    return 6;
  else if ((strcmp (arg, "-m") == 0) || (strcmp (arg, "--mic") == 0))
    return 7;
  else if ((strcmp (arg, "-c") == 0) || (strcmp (arg, "--cd") == 0))
    return 8;
  else if ((strcmp (arg, "-I") == 0) || (strcmp (arg, "--imix") == 0))
    return 9;
  else if ((strcmp (arg, "-a") == 0) || (strcmp (arg, "--altpcm") == 0))
    return 10;
  else if ((strcmp (arg, "-r") == 0) || (strcmp (arg, "--reclev") == 0))
    return 11;
  else if ((strcmp (arg, "-i") == 0) || (strcmp (arg, "--igain") == 0))
    return 12;
  else if ((strcmp (arg, "-o") == 0) || (strcmp (arg, "--ogain") == 0))
    return 13;
  else if ((strcmp (arg, "-l1") == 0) || (strcmp (arg, "--line1") == 0))
    return 14;
  else if ((strcmp (arg, "-l2") == 0) || (strcmp (arg, "--line2") == 0))
    return 15;
  else if ((strcmp (arg, "-l3") == 0) || (strcmp (arg, "--line3") == 0))
    return 16;
  else if ((strcmp (arg, "-3d") == 0) || (strcmp (arg, "--3dse") == 0))
    return 17;
  else if ((strcmp (arg, "-agc") == 0) || (strcmp (arg, "--agc") == 0))
    return 18;

  else if ((strcmp (arg, "-h") == 0) || (strcmp (arg, "-?") == 0) || (strcmp (arg, "--help") == 0))
    return 20;
  else if (strcmp (arg, "-get") == 0)
    return 21;
  else if (strcmp (arg, "-getfile") == 0)
    return 22;
  else if (strcmp (arg, "-save") == 0)
    return 23;
  else if (strcmp (arg, "-savefile") == 0)
    return 24;
  else
    return 99;
}

/* sets sound given from command line */
void
set_sound (int argc, char *argv[])
{
  int ii, vol_left, vol_right, dummy, pos;
  int argc_count = 1, arg_code;
  FILE *save_file;
  char *SAVE_FILE, *q;
  int device;
  int left, right;

  q = getenv ("HOME");
  SAVE_FILE = malloc (strlen (q) + 8);
  strcpy (SAVE_FILE, q);
  strcat (SAVE_FILE, "/.camrc");

  init_sound ();
  do
    {
      arg_code = eval_argv (argv[argc_count]);
      if (argc_count >= argc - 1)
        switch (arg_code)
	  {
	  case 20:
	  case 21:
	  case 23:
	  case 99:
	    break;
	  case 22:
	  case 24:
	    fprintf (stderr, "mising filename for option %s\n",argv[argc_count]);
	    exit(1);
	  default:
	    fprintf (stderr, "mising value for option %s\n",argv[argc_count]);
	    exit(1);
	}
      switch (arg_code)
	{
	case 20:
	  comm_help ();
	  break;
	case 21:		/* get */
	  if ((save_file = fopen (SAVE_FILE, "r")) == NULL)
	    printf ("\nNo save file available\n\n");
	  else
	    {
	      fscanf (save_file, "%i,%i,%i\n", &dummy, &a3dse, &agc);
#ifdef SOUND_MIXER_3DSE
	      ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse);
#endif
#ifdef SOUND_MIXER_AGC
	      ioctl (mixer_fd, SOUND_MIXER_AGC, &agc);
#endif

	      for (ii = 0; ii < DEV_COUNT; ii++)
		{
		  fscanf (save_file, "%i %i\n", &vol_left, &vol_right);
		  if (DEV_EXIST (ii))
		    adjust (ii, vol_left, vol_right);
		}
	    }
	  fclose (save_file);
	  break;
	case 22:		/* getfile */
	  argc_count++;
	  if ((save_file = fopen (argv[argc_count], "r")) == NULL)
	    printf ("\nNo save file available\n\n");
	  else
	    {
	      fscanf (save_file, "%i,%i,%i\n", &dummy, &a3dse, &agc);
#ifdef SOUND_MIXER_3DSE
	      ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse);
#endif
#ifdef SOUND_MIXER_AGC
	      ioctl (mixer_fd, SOUND_MIXER_AGC, &agc);
#endif

	      for (ii = 0; ii < DEV_COUNT; ii++)
		{
		  fscanf (save_file, "%i %i\n", &vol_left, &vol_right);
		  if (DEV_EXIST (ii))
		    adjust (ii, vol_left, vol_right);
		}
	      fclose (save_file);
	    }
	  break;
	case 23:		/* save */
	  if ((save_file = fopen (SAVE_FILE, "w")) == NULL)
	    {
	      message ("Unable to save settings");
	      break;
	    }
#ifdef SOUND_MIXER_3DSE
	  a3dse = 2;
	  if (ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse) == -1)
	    {
	      a3dse = 0;
	    }
#else
	  a3dse = 0;
#endif
	  fprintf (save_file, "0,%i,0\n", a3dse);
	  for (ii = 0; ii < DEV_COUNT; ii++)
	    {
	      if (DEV_EXIST (ii))
		{
		  if (ioctl (mixer_fd, MIXER_READ (ii), &pos) == -1)
		    {
		      finish (0);
		      fprintf (stderr, "MIXER_READ : Error reading %s\n", mixer_dev);
		      exit (1);
		    }
		  vol_left = getbits (pos, 7, 8);

		  if (STEREO (ii))
		    {
		      vol_right = getbits (pos, 15, 8);
		    }
		  else
		    vol_right = -1;
		}
	      else
		{
		  vol_left = -1;
		  vol_right = -1;
		}
	      fprintf (save_file, "%i %i\n", vol_left, vol_right);
	    }
	  fclose (save_file);
	  break;
	case 24:		/* savefile */
	  argc_count++;
	  if ((save_file = fopen (argv[argc_count], "w")) == NULL)
	    {
	      message ("Unable to save settings");
	      break;
	    }
#ifdef SOUND_MIXER_3DSE
	  a3dse = 2;
	  if (ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse) == -1)
	    {
	      a3dse = 0;
	    }
#else
	  a3dse = 0;
#endif
	  fprintf (save_file, "0,%i,0\n", a3dse);
	  for (ii = 0; ii < DEV_COUNT; ii++)
	    {
	      if (DEV_EXIST (ii))
		{
		  if (ioctl (mixer_fd, MIXER_READ (ii), &pos) == -1)
		    {
		      finish (0);
		      fprintf (stderr, "MIXER_READ : Error reading %s\n", mixer_dev);
		      exit (1);
		    }
		  vol_left = getbits (pos, 7, 8);
		  if (STEREO (ii))
		    {
		      vol_right = getbits (pos, 15, 8);
		    }
		  else
		    vol_right = -1;
		}
	      else
		{
		  vol_left = -1;
		  vol_right = -1;
		}
	      fprintf (save_file, "%i %i\n", vol_left, vol_right);
	    }
	  fclose (save_file);
	  break;

	case 0:
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
	case 6:
	case 7:
	case 8:
	case 9:
	case 10:
	case 11:
	case 12:
	case 13:
	case 14:
	case 15:
	case 16:

	  device = eval_argv (argv[argc_count]);
	  argc_count++;

	  if DEV_EXIST
	    (device)
	    {
	      sscanf (argv[argc_count], "%i,%i", &left, &right);
	      adjust (device, left, right);
	    }
	  else
	    fprintf (stderr, "\nDevice not supported\n\n");
	  break;
	case 17:
	  argc_count++;
	  if (*argv[argc_count] == '1')
	    a3dse = 1;
	  else if (*argv[argc_count] == '0')
	    a3dse = 0;
	  else
	    {
	      fprintf (stderr, "\nIllegal parameter %s\n", argv[argc_count]);
	      break;
	    }
#ifdef SOUND_MIXER_3DSE
	  if (ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse) == -1)
	    {
	      fprintf (stderr, "\n3D Stereo Enhancement unsupported\n");
	    }
#endif
	  break;
	case 18:
	  argc_count++;
	  if (*argv[argc_count] == '1')
	    agc = 1;
	  else if (*argv[argc_count] == '0')
	    agc = 0;
	  else
	    {
	      fprintf (stderr, "\nIllegal parameter %s\n", argv[argc_count]);
	      break;
	    }
#ifdef SOUND_MIXER_AGC
	  if (ioctl (mixer_fd, SOUND_MIXER_AGC, &agc) == -1)
	    {
	      fprintf (stderr, "\n3D Stereo Enhancement unsupported\n");
	    }
#endif
	  break;
	case 99:
	  fprintf (stderr, "\nIllegal option %s\n", argv[argc_count]);
	  argc_count++;
	  break;
	}
      argc_count++;
    }
  while (argc_count < argc);
}



void
init_sliders (void)
{
  int X, Y;
  int dev_nr;
  int pos;
  int max_dev = ((LINES - 2 - 3) / 3) * 2;
  X = SPACE;
  Y = 2;
  a3dse = 2;
  agc = 0;			/* Turn off agc function at start */
#ifdef SOUND_MIXER_3DSE
  if (ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse) == -1)
    {
      a3dse = -1;
    }
#else
  a3dse = -1;
#endif
#ifdef SOUND_MIXER_AGC
  if (ioctl (mixer_fd, SOUND_MIXER_AGC, &agc) == -1)
    {
      agc = -1;
    }
#else
  agc = -1;
#endif
  for (dev_nr = 0; dev_nr < DEV_COUNT; dev_nr++)
    {
      if (DEV_EXIST (dev_nr))
	{
	  if (dev_count >= max_dev)
	    break;

	  if (ioctl (mixer_fd, MIXER_READ (dev_nr), &pos) == -1)
	    {
	      finish (0);
	      fprintf (stderr, "MIXER_READ : Error reading %s\n", mixer_dev);
	      exit (1);
	    }

	  if ((dev_vol[dev_count].vol_left = getbits (pos, 7, 8) / (DIV >> 8)) > COLS / 4)
	    dev_vol[dev_count].vol_left = COLS / 4;
	  if (STEREO (dev_nr))
	    {
	      if ((dev_vol[dev_count].vol_right = getbits (pos, 15, 8) / (DIV >> 8)) > COLS / 4)
		dev_vol[dev_count].vol_right = COLS / 4;
	    }
	  else
	    {
	      dev_vol[dev_count].vol_right = dev_vol[dev_count].vol_left;
	    }
	  dev_vol[dev_count].X = X;
	  dev_vol[dev_count].Y = Y;
	  draw_slider (Y, X, dev_nr);
	  if (getbits (recmask, dev_nr, 1))
	    draw_rec_lamp (Y + 2, X + 4, getbits (recsrc, dev_nr, 1) ? 1 : 2);
	  draw_handle (Y + 1, X + 14 + dev_vol[dev_count].vol_left, FALSE, 4);
	  if (STEREO (dev_nr))
	    draw_handle (Y + 2, X + 14 + dev_vol[dev_count].vol_right, FALSE, 4);
	  dev_ind[dev_nr] = dev_count;
	  dev_vol[dev_count++].dev_nr = dev_nr;
	  if (Y + 6 > LINES - 3)
	    {
	      Y = 2;
	      X = COLS - (RPOS + 14);
	    }
	  else if (STEREO (dev_nr))
	    Y += 3;
	  else
	    Y += 2;
	}
      else
	dev_ind[dev_nr] = -1;
    }
}

void
ctrl_sliders (void)
{
  int ch;
  int X, Y;
  int left = TRUE;
  int locked = FALSE;
  FILE *save_file;
  int dev_nr = 0;
  char *SAVE_FILE, *q;
  int *volume;
  int X_rest = LPOS;
  int dev_nrs;
  int pos, rec_tmp;
  int vol_left, vol_right;
  int dev_nr_tmp;

  q = getenv ("HOME");
  SAVE_FILE = malloc (strlen (q) + 8);
  strcpy (SAVE_FILE, q);
  strcat (SAVE_FILE, "/.camrc");

  init_sound ();
  init_sliders ();

  keypad (stdscr, TRUE);
  cbreak ();
  noecho ();

  volume = &dev_vol[dev_nr].vol_left;

  X = X_rest + *volume;
  Y = dev_vol[dev_nr].Y + 1;

  draw_handle (Y, X, TRUE, 4);
  do
    {

      status (((locked == TRUE) ? "  LOCKED  " : " UNLOCKED "));
      ch = getch ();
      fflush (stdin);
      message ("");
      switch (ch)
	{
	case 27:
	case 'q':
	case 'Q':
	  break;
	case KEY_LEFT:
	  if (*volume > 0)
	    {
	      *volume = *volume - 1;
	      X--;
	      draw_handle (Y, X, TRUE, 3);

	      if ((locked) && (STEREO (dev_vol[dev_nr].dev_nr)))
		{
		  dev_vol[dev_nr].vol_right--;
		  draw_handle (Y + 1, X, TRUE, 3);
		}
	    }
	  break;
	case KEY_RIGHT:
	  if (*volume < (COLS / 4))
	    {
	      *volume = *volume + 1;
	      X++;
	      draw_handle (Y, X - 1, TRUE, 1);

	      if ((locked) && (STEREO (dev_vol[dev_nr].dev_nr)))
		{
		  dev_vol[dev_nr].vol_right++;
		  draw_handle (Y + 1, X - 1, TRUE, 1);
		}
	    }
	  break;
	case KEY_UP:

	  draw_handle (Y, X, FALSE, 4);
	  if ((locked) && STEREO (dev_vol[dev_nr].dev_nr))
	    draw_handle (Y + 1, X, FALSE, 4);

	  if ((left == TRUE) || (locked == TRUE) || (!STEREO (dev_vol[dev_nr].dev_nr)))
	    {
	      dev_nr = (dev_nr + dev_count - 1) % dev_count;

	      X_rest = dev_vol[dev_nr].X + 14;
	      if (STEREO (dev_vol[dev_nr].dev_nr) && (!locked))
		Y = dev_vol[dev_nr].Y + 2;
	      else
		Y = dev_vol[dev_nr].Y + 1;

	      if (locked && (STEREO (dev_vol[dev_nr].dev_nr)))
		{
		  int temp_vol;

		  X = dev_vol[dev_nr].vol_left + X_rest;
		  draw_handle (Y, X, FALSE, 5);
		  X = dev_vol[dev_nr].vol_right + X_rest;
		  draw_handle (Y + 1, X, FALSE, 5);

		  temp_vol = (dev_vol[dev_nr].vol_left + dev_vol[dev_nr].vol_right) / 2;
		  dev_vol[dev_nr].vol_left = temp_vol;
		  dev_vol[dev_nr].vol_right = temp_vol;

		}
	      if (STEREO (dev_vol[dev_nr].dev_nr) && (!locked))
		{
		  X = dev_vol[dev_nr].vol_right + X_rest;
		  volume = &dev_vol[dev_nr].vol_right;
		  left = FALSE;
		}
	      else
		{
		  X = dev_vol[dev_nr].vol_left + X_rest;
		  volume = &dev_vol[dev_nr].vol_left;
		  left = TRUE;
		}

	    }
	  else if ((left == FALSE) && (STEREO (dev_vol[dev_nr].dev_nr)))
	    {
	      Y--;
	      volume = &dev_vol[dev_nr].vol_left;
	      X = X_rest + *volume;
	      left = TRUE;
	    }

	  draw_handle (Y, X, TRUE, 4);
	  if ((locked) && STEREO (dev_vol[dev_nr].dev_nr))
	    draw_handle (Y + 1, X, TRUE, 4);
	  break;

	case KEY_DOWN:
	  draw_handle (Y, X, FALSE, 4);
	  if ((locked) && STEREO (dev_vol[dev_nr].dev_nr))
	    draw_handle (Y + 1, X, FALSE, 4);

	  if ((left == FALSE) || (locked) || (!STEREO (dev_vol[dev_nr].dev_nr)))
	    {
	      dev_nr = (dev_nr + 1) % dev_count;

	      X_rest = dev_vol[dev_nr].X + 14;
	      Y = dev_vol[dev_nr].Y + 1;

	      if (locked && (STEREO (dev_vol[dev_nr].dev_nr)))
		{
		  int temp_vol;

		  X = dev_vol[dev_nr].vol_left + X_rest;
		  draw_handle (Y, X, FALSE, 5);
		  X = dev_vol[dev_nr].vol_right + X_rest;
		  draw_handle (Y + 1, X, FALSE, 5);

		  temp_vol = (dev_vol[dev_nr].vol_left + dev_vol[dev_nr].vol_right) / 2;
		  dev_vol[dev_nr].vol_left = temp_vol;
		  dev_vol[dev_nr].vol_right = temp_vol;

		}

	      X = dev_vol[dev_nr].vol_left + X_rest;
	      volume = &dev_vol[dev_nr].vol_left;
	      left = TRUE;
	    }
	  else if ((left == TRUE) && (STEREO (dev_vol[dev_nr].dev_nr)))
	    {
	      Y++;
	      volume = &dev_vol[dev_nr].vol_right;
	      X = X_rest + *volume;
	      left = FALSE;
	    }



	  draw_handle (Y, X, TRUE, 4);
	  if ((locked) && STEREO (dev_vol[dev_nr].dev_nr))
	    draw_handle (Y + 1, X, TRUE, 4);
	  break;

	case 'r':
	case 'R':
	  if (getbits (recmask, dev_nr, 1))
	    {
	      rec_tmp = getbits (recsrc, dev_vol[dev_nr].dev_nr, 1);
	      if (rec_tmp)
		recsrc &= ~(1 << dev_vol[dev_nr].dev_nr);
	      else
		recsrc |= (1 << dev_vol[dev_nr].dev_nr);
	      if (ioctl (mixer_fd, SOUND_MIXER_WRITE_RECSRC, &recsrc) == -1)
		{
		  finish (0);
		  fprintf (stderr, "SOUND_MIXER_WRITE_RECSRC : Error reading %s\n", mixer_dev);
		  exit (-1);
		}
	      if (ioctl (mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1)
		{
		  finish (0);
		  fprintf (stderr, "SOUND_MIXER_READ_RECSRC : Error reading %s\n", mixer_dev);
		  exit (-1);
		}
	      {
		register int i;
		for (i = 0; i < DEV_COUNT; i++)
		  {
		    if (getbits (recmask, i, 1))
		      {
			rec_tmp = getbits (recsrc, i, 1);
			draw_rec_lamp (dev_vol[i].Y + 2, dev_vol[i].X + 4, rec_tmp ? 1 : 2);
		      }
		  }
	      }
	    }
	  break;
	case 'l':
	case 'L':

	  if ((locked) && STEREO (dev_vol[dev_nr].dev_nr))
	    {
	      if (left)
		draw_handle (Y + 1, X, FALSE, 4);
	      else
		draw_handle (Y - 1, X, FALSE, 4);
	    }
	  else if ((!locked) && STEREO (dev_vol[dev_nr].dev_nr))
	    {
	      int temp_vol;

	      draw_handle (Y, X, FALSE, 5);
	      if (left)
		{
		  X = dev_vol[dev_nr].vol_right + X_rest;
		  draw_handle (Y + 1, X, FALSE, 5);
		}
	      else
		{
		  X = dev_vol[dev_nr].vol_left + X_rest;
		  draw_handle (Y - 1, X, FALSE, 5);
		}
	      temp_vol = (dev_vol[dev_nr].vol_left + dev_vol[dev_nr].vol_right) / 2;
	      dev_vol[dev_nr].vol_left = temp_vol;
	      dev_vol[dev_nr].vol_right = temp_vol;

	      X = dev_vol[dev_nr].vol_left + X_rest;
	      draw_handle (Y, X, TRUE, 4);
	      if (left)
		draw_handle (Y + 1, X, TRUE, 4);
	      else
		draw_handle (Y - 1, X, TRUE, 4);

	    }
	  locked = (locked + 1) % 2;
	  if (!left)
	    {
	      Y--;
	      left = TRUE;
	      volume = &dev_vol[dev_nr].vol_left;
	    }
	  break;
	case 's':
	case 'S':
	  if ((save_file = fopen (SAVE_FILE, "w")) == NULL)
	    {
	      message ("Unable to save settings");
	      break;
	    }
	  fprintf (save_file, "%i,%i,%i\n", locked, a3dse, agc);
	  for (dev_nrs = 0; dev_nrs < DEV_COUNT; dev_nrs++)
	    {
	      if (DEV_EXIST (dev_nrs))
		{
		  if (ioctl (mixer_fd, MIXER_READ (dev_nrs), &pos) == -1)
		    {
		      finish (0);
		      fprintf (stderr, "MIXER_READ : Error reading %s\n", mixer_dev);
		      exit (1);
		    }
		  vol_left = getbits (pos, 7, 8);
		  if (STEREO (dev_nrs))
		    {
		      vol_right = getbits (pos, 15, 8);
		    }
		  else
		    vol_right = -1;
		}
	      else
		{
		  vol_left = -1;
		  vol_right = -1;
		}
	      fprintf (save_file, "%i %i\n", vol_left, vol_right);

	    }
	  fclose (save_file);
	  message ("Settings saved");
	  break;
	case 'g':
	case 'G':

	  if ((save_file = fopen (SAVE_FILE, "r")) == NULL)
	    {
	      message ("No save file available");
	      break;
	    }
	  fscanf (save_file, "%i,%i,%i\n", &locked, &a3dse, &agc);
#ifdef SOUND_MIXER_3DSE
	  ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse);
	  if (a3dse == TRUE)
	    {
	      attrset (COLOR_PAIR (COLOR_VOL1));
	      mvaddnstr (dev_vol[SOUND_MIXER_VOLUME].Y + 2, dev_vol[SOUND_MIXER_VOLUME].X, "3D", 2);
	    }
	  else
	    {
	      attrset (COLOR_PAIR (COLOR_SLIDER));
	      mvaddnstr (dev_vol[SOUND_MIXER_VOLUME].Y + 2, dev_vol[SOUND_MIXER_VOLUME].X, "  ", 2);
	    }
#endif
#ifdef SOUND_MIXER_AGC
	  ioctl (mixer_fd, SOUND_MIXER_AGC, &agc);
	  if (agc == TRUE)
	    {
	      attrset (COLOR_PAIR (COLOR_VOL1));
	      mvaddnstr (dev_vol[SOUND_MIXER_MIC].Y + 2, dev_vol[SOUND_MIXER_MIC].X, "AGC", 3);
	    }
	  else
	    {
	      attrset (COLOR_PAIR (COLOR_SLIDER));
	      mvaddnstr (dev_vol[SOUND_MIXER_MIC].Y + 2, dev_vol[SOUND_MIXER_MIC].X, "   ", 3);
	    }
#endif
	  dev_nr_tmp = 0;
	  for (dev_nrs = 0; dev_nrs < DEV_COUNT; dev_nrs++)
	    {

	      fscanf (save_file, "%i %i\n", &vol_left, &vol_right);
	      if (DEV_EXIST (dev_nrs))
		{
		  adjust (dev_nrs, vol_left, vol_right);
		  if (vol_left != -1)
		    {
		      if ((vol_left / (DIV >> 8)) > COLS / 4)
			vol_left = COLS / 4;
		      else
			vol_left /= (DIV >> 8);
		    }
		  if (vol_right != -1)
		    {
		      if ((vol_right / (DIV >> 8)) > COLS / 4)
			vol_right = COLS / 4;
		      else
			vol_right /= (DIV >> 8);
		    }
		  if ((dev_nr_tmp = dev_ind[dev_nrs]) == -1)
		    continue;
		  draw_handle (dev_vol[dev_nr_tmp].Y + 1, dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_left, FALSE, 5);

		  dev_vol[dev_nr_tmp].vol_left = vol_left;
		  if ((dev_nr_tmp == dev_nr) && (locked || left == TRUE))
		    {
		      draw_handle (dev_vol[dev_nr_tmp].Y + 1, dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_left, TRUE, 4);
		      X = dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_left;
		    }
		  else
		    draw_handle (dev_vol[dev_nr_tmp].Y + 1, dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_left, FALSE, 4);

		  if (STEREO (dev_nrs))
		    {
		      draw_handle (dev_vol[dev_nr_tmp].Y + 2, dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_right, FALSE, 5);

		      dev_vol[dev_nr_tmp].vol_right = vol_right;
		      if ((dev_nr_tmp == dev_nr) && (locked || left == FALSE))
			{
			  draw_handle (dev_vol[dev_nr_tmp].Y + 2, dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_right, TRUE, 4);
			  X = dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_right;
			}
		      else
			draw_handle (dev_vol[dev_nr_tmp].Y + 2, dev_vol[dev_nr_tmp].X + 14 + dev_vol[dev_nr_tmp].vol_right, FALSE, 4);

		    }
		}
	    }
	  fclose (save_file);
	  message ("Settings loaded");
	  break;
#ifdef SOUND_MIXER_3DSE
	case '3':
	  if (a3dse != -1)
	    {
	      if (a3dse == FALSE)
		a3dse = TRUE;
	      else
		a3dse = FALSE;
	      ioctl (mixer_fd, SOUND_MIXER_3DSE, &a3dse);
	    }
	  if (a3dse == TRUE)
	    {
	      attrset (COLOR_PAIR (COLOR_VOL1));
	      mvaddnstr (dev_vol[SOUND_MIXER_VOLUME].Y + 2, dev_vol[SOUND_MIXER_VOLUME].X, "3D", 2);
	    }
	  else
	    {
	      attrset (COLOR_PAIR (COLOR_SLIDER));
	      mvaddnstr (dev_vol[SOUND_MIXER_VOLUME].Y + 2, dev_vol[SOUND_MIXER_VOLUME].X, "  ", 2);
	    }
	  break;
#endif
#ifdef SOUND_MIXER_AGC
	case 'a':
	case 'A':
	  if (agc != -1)
	    {
	      if (agc == FALSE)
		agc = TRUE;
	      else
		agc = FALSE;
	      ioctl (mixer_fd, SOUND_MIXER_AGC, &agc);
	    }
	  if (agc == TRUE)
	    {
	      attrset (COLOR_PAIR (COLOR_VOL1));
	      mvaddnstr (dev_vol[SOUND_MIXER_MIC].Y + 2, dev_vol[SOUND_MIXER_MIC].X, "AGC", 3);
	    }
	  else
	    {
	      attrset (COLOR_PAIR (COLOR_SLIDER));
	      mvaddnstr (dev_vol[SOUND_MIXER_MIC].Y + 2, dev_vol[SOUND_MIXER_MIC].X, "   ", 3);
	    }
	  break;
#endif
	case 'h':
	case 'H':
	  help ();
	  break;
	case 'Z':
	  clear ();
	  refresh ();
	  endwin ();
	  dev_count = 0;
	  longjmp (env, 2);
	}
      if ((ch != 27) && (ch != 'Q') && (ch != 'q'))
	{
	  adjust (dev_vol[dev_nr].dev_nr,
		  (((dev_vol[dev_nr].vol_left * DIV) >> 8) & 0xff),
		  (((dev_vol[dev_nr].vol_right * DIV) >> 8) & 0xff));
	}
    }
  while ((ch != 27) && (ch != 'Q') && (ch != 'q'));

}


/* initializes the sound */
void
init_sound ()
{

  if ((mixer_fd = open (mixer_dev, O_RDWR)) < 0)
    {
      finish (0);
      fprintf (stderr, "Error opening %s. : you probably don't have an audiocard that supports a mixer", mixer_dev);
      exit (1);
    }
  if (ioctl (mixer_fd, SOUND_MIXER_READ_DEVMASK, &devmask) == -1)
    {
      finish (0);
      fprintf (stderr, "SOUND_MIXER_READ_DEVMASK : Error reading %s\n", mixer_dev);
      exit (-1);
    }
  if (ioctl (mixer_fd, SOUND_MIXER_READ_RECMASK, &recmask) == -1)
    {
      finish (0);
      fprintf (stderr, "SOUND_MIXER_READ_RECMASK : Error reading %s\n", mixer_dev);
      exit (-1);
    }
  if (ioctl (mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1)
    {
      finish (0);
      fprintf (stderr, "SOUND_MIXER_READ_RECSRC : Error reading %s\n", mixer_dev);
      exit (-1);
    }
  if (ioctl (mixer_fd, SOUND_MIXER_READ_STEREODEVS, &stereodevs) == -1)
    {
      finish (0);
      fprintf (stderr, "SOUND_MIXER_READ_STEREODEVS : Error reading %s\n", mixer_dev);
      exit (-1);

    }

  if (!devmask)
    {
      finish (0);
      fprintf (stderr, "No device found.\n");
      exit (1);
    }
}
