# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007 Canonical Limited.
#   Author: Robert Collins.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""Activity of bzr.

This module provides Activity which supports announcing and recieving messages
about bzr activity: See the class for more detail.
"""

import socket
import time

import dbus.service
import gobject

from bzrlib.plugins.dbus import mapper
from bzrlib.revision import NULL_REVISION
from bzrlib.smart.protocol import _encode_tuple, _decode_tuple


def _get_bus(bus):
    """Get a bus thats usable."""
    if bus is None:
        # lazy import to not pollute bzr during startup.
        import dbus.mainloop.glib
        dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
        return dbus.SessionBus()
    else:
        return bus


class Activity(object):
    """bzrlib object activity tracking."""

    def __init__(self, bus=None):
        """Create an Activity object.

        :param bus: A dbus Bus object. By default this will be set to
            bus.SessionBus(). If you need a private bus or a system bus,
            supply this parameter.
        """
        self.bus = _get_bus(bus)

    def add_url_map(self, source_prefix, target_prefix):
        """Helper to invoke add_url_map on the dbus Broadcast service."""
        self._call_on_broadcast('add_url_map', source_prefix, target_prefix)

    def advertise_branch(self, branch):
        """Advertise branch to dbus.

        This is a top level convenience function to advertise a branch. No
        warranties are made about delivery of the advertisement, nor of how
        long it will be visible to users. Specifically, dbus errors are caught,
        and the advertisement is not repeated.
        :param branch: The branch to be advertised. The advertisement is done
            by announcing the tip revision and the URL of the branch.
        :return: None
        :raises: Nothing should be raised.
        """
        self.announce_revision(branch.last_revision(), branch.base)

    def announce_revision(self, revision, url):
        """Low level revision-specific announce logic. 

        The recommended API is advertise_branch, announce_revision, while 
        public is not stable or supported. Use at your own warranty.
        """
        if revision in (None, NULL_REVISION):
            revision = '' # avoid sending None or NULL_REVISION (its internal
                          # only) on the wire.
        self._call_on_broadcast('announce_revision', revision, url)

    def announce_revision_urls(self, revision, urls):
        """Low level revision-specific announce logic. 

        The recommended API is advertise_branch, announce_revision, while 
        public is not stable or supported. Use at your own warranty.

        This method does not translate urls: its expected that the urls
        being advertised are already translated.
        """
        if revision in (None, NULL_REVISION):
            revision = '' # avoid sending None or NULL_REVISION (its internal
                          # only) on the wire.
        self._call_on_broadcast('announce_revision_urls', revision, urls)

    def _call_on_broadcast(self, method_name, *args):
        """Thunk method through to the dbus Broadcast service."""
        try:
            dbus_object = self.bus.get_object(Broadcast.DBUS_NAME,
                Broadcast.DBUS_PATH)
        except dbus.DBusException, e:
            if (e.get_dbus_name() ==
                'org.freedesktop.DBus.Error.ServiceUnknown'):
                # service not available
                return
            else:
                # some other error
                raise
        dbus_iface = dbus.Interface(dbus_object, Broadcast.DBUS_INTERFACE)
        # make a non-blocking call, which we can then ignore as we dont
        # care about responses: Apparently there is some dbus foo to help
        # make this not need the stub function
        import gobject
        mainloop = gobject.MainLoop()
        def handle_reply():
            # quit our loop.
            mainloop.quit()
        def handle_error(error):
            """If an error has happened, lets raise it.
            
            Note that this will not raise it at the right point, but it should
            hit some handler somewhere.
            """
            mainloop.quit()
            raise error
        method = getattr(dbus_iface, method_name)
        method(reply_handler=handle_reply, error_handler=handle_error, *args)
        # iterate enough to emit the signal, in case we are being called from a
        # sync process.
        mainloop.run()

    def listen_for_revisions(self, callback):
        """Listen for revisions over dbus."""
        broadcast_service = self.bus.get_object(
            Broadcast.DBUS_NAME,
            Broadcast.DBUS_PATH)
        broadcast_service.connect_to_signal("Revision", callback,
            dbus_interface=Broadcast.DBUS_INTERFACE)

    def remove_url_map(self, source_prefix, target_prefix):
        """Helper to invoke remove_url_map on the dbus Broadcast service."""
        self._call_on_broadcast('remove_url_map', source_prefix, target_prefix)

    def serve_broadcast(self, when_ready=None):
        """Run a 'Broadcast' server.

        This is the core logic for 'bzr dbus-broadcast' which will be invoked
        by dbus activation.

        It starts up gobject mainloop and places a Broadcast object on that.
        When the loop exits, it returns.

        :param when_ready: An optional callback to be invoked after the server
            is ready to handle requests.
        """
        broadcaster = Broadcast(self.bus)
        mainloop = gobject.MainLoop()
        if when_ready:
            when_ready()
        mainloop.run()


class Broadcast(dbus.service.Object):

    # Dont try to put a '-' in bazaar-vcs here, its NOT dns and dbus considers -
    # illegal.
    DBUS_NAME = "org.bazaarvcs.plugins.dbus.Broadcast"
    DBUS_PATH = "/org/bazaarvcs/plugins/dbus/Broadcast"
    DBUS_INTERFACE = "org.bazaarvcs.plugins.dbus.Broadcast"

    def __init__(self, bus):
        """Create a Broadcast service.

        :param bus: The bus to serve on.
        """
        bus_name = dbus.service.BusName(
            Broadcast.DBUS_NAME, bus=bus)
        dbus.service.Object.__init__(self, bus, Broadcast.DBUS_PATH, bus_name)
        self.url_mapper = mapper.URLMapper()
        self.bus = bus

    @dbus.service.method(DBUS_INTERFACE,
                         in_signature='ss', out_signature='')
    def add_url_map(self, source_prefix, target_prefix):
        """Add a url prefix to be mapped when advertising revisions."""
        self.url_mapper.add_map(source_prefix, target_prefix)

    @dbus.service.method(DBUS_INTERFACE,
                         in_signature='ss', out_signature='')
    def announce_revision(self, revision_id, url):
        """Announce revision_id as being now present at url.

        To avoid information disclosure, no details are handed over the wire:
        clients should access the revision to determine its contents.
        """
        urls = self.url_mapper.map(url)
        if not urls:
            urls = [url]
        self.Revision(revision_id, urls)

    @dbus.service.method(DBUS_INTERFACE,
                         in_signature='sas', out_signature='')
    def announce_revision_urls(self, revision_id, urls):
        """Announce revision_id as being now present at urls.

        To avoid information disclosure, no details are handed over the wire:
        clients should access the revision to determine its contents.
        """
        self.Revision(revision_id, urls)

    @dbus.service.method(DBUS_INTERFACE,
                         in_signature='ss', out_signature='')
    def remove_url_map(self, source_prefix, target_prefix):
        """Remove a previously mapped url prefix."""
        self.url_mapper.remove_map(source_prefix, target_prefix)

    @dbus.service.signal(DBUS_INTERFACE, 'sas')
    def Revision(self, revision, urls):
        """A revision has been observed at url."""


class LanGateway(object):
    """A gateway for bazaar commit notifications to the local lan."""

    def __init__(self, bus=None, mainloop=None, activity=None):
        """Create a LanGateway object.

        :param bus: A dbus Bus object. By default this will be set to
            bus.SessionBus(). If you need a private bus or a system bus,
            supply this parameter.
        """
        self.bus = _get_bus(bus)
        if mainloop is None:
            self.mainloop = gobject.MainLoop()
        else:
            self.mainloop = mainloop
        if activity is None:
            self.activity = Activity(self.bus)
        else:
            self.activity = activity
        self.seen_revisions = {}

    def broadcast_data(self, data):
        """Transmit data on the LAN in a broadcast packet."""
        self.sock.sendto(data, ('<broadcast>', 4155))

    def catch_dbus_revision(self, revision_id, urls):
        """Catch a published revision_id from dbus."""
        packet_args = ['announce_revision', revision_id]
        seen_urls = set()
        for revisions in self.seen_revisions.values():
            if revision_id in revisions:
                seen_urls.update(set(revisions[revision_id]))
        allowed_urls = []
        for url in urls:
            if not url.startswith('file:///') and not url in seen_urls:
                allowed_urls.append(url)
        self.note_network_revision(time.time(), revision_id, allowed_urls)
        packet_args.extend(allowed_urls)
        if len(packet_args) == 2:
            # no valid urls.
            return
        self.broadcast_data(_encode_tuple(packet_args))

    def handle_network_data(self, data):
        """Handle data from the network."""
        args = _decode_tuple(data)
        assert args[0] == 'announce_revision'
        rev_id = args[1]
        announce = True
        for revisions in self.seen_revisions.values():
            if rev_id in revisions:
                announce = False
        self.note_network_revision(time.time(), args[1], args[2:])
        if announce:
            self.activity.announce_revision_urls(args[1], args[2:])

    def handle_network_packet(self, source, condition):
        """Handle a packet from the network."""
        data = source.recvfrom(65535)
        self.handle_network_data(data[0])
        return True

    def note_network_revision(self, now, revid, urls):
        """Note that revid was seen at urls now.

        This will remove stale cache entries.

        :param now: the result of time.time().
        """
        keys_to_remove = []
        for key in self.seen_revisions.keys():
            if key + 5 < int(now/60):
                keys_to_remove.append(key)
        for key in keys_to_remove:
            del self.seen_revisions[key]
        self.seen_revisions.setdefault(int(now/60), {})[revid] = urls

    def run(self, _port=4155):
        """Start the LanGateway.
        
        The optional _port parameter is used for testing.
        """
        self.start(_port)
        try:
            self.mainloop.run()
        finally:
            self.stop()

    def start(self, _port=4155):
        """Start the LanGateway.
        
        The optional _port parameter is used for testing.
        """
        # listen for network events
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
        self.sock.bind(('', _port))
        if _port == 0:
            self.port = self.sock.getsockname()[1]
        else:
            self.port = _port
        gobject.io_add_watch(self.sock, gobject.IO_IN, self.handle_network_packet)
        # listen for dbus events
        self.activity.listen_for_revisions(self.catch_dbus_revision)

    def stop(self):
        """Stop the LanGateway."""
        self.sock.close()
        self.sock = None
