# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007 Canonical Limited.
#   Author: Robert Collins.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""D-Bus integration for bzr/bzrlib. 

This plugin provides integration with D-Bus and optional local-LAN commit
broadcasting. Please see README for installation instructions which will
correctly configure the D-Bus service included in the plugin.

The integration involves a D-Bus advertising daemon and hooks into the bzr
library to reflect events (e.g. push/pull/commit/uncommit) across to D-Bus. 
This should be useful for IDE's and other editors working in the same space as
bzr, or for integration with network services.


Commands provided:
------------------

 * dbus-broadcast: D-Bus commit/branch advertising daemon. The D-Bus service
   which is activated on demand if it has been correctly installed. If for some
   reason you cannot correctly install the .service file this command can be
   run by hand.
   This service is fully documented in its python code - see 'pydoc
   bzrlib.plugins.dbus.activity.Broadcast'.
 * lan-notify: Provide a bi-directional gateway of commit-notifications to the
   local LAN. Only the URL and revision-id are disclosed, no commit content is
   transmitted. This command is typically put into the background - e.g. ``bzr
   lan-notify &``. ``lan-notify`` is very useful in local LAN collaboration to
   keep multiple developers in sync.


Related commands:
-----------------

 * bzr commit-notify. This command, shipped in the 'bzr-gtk' plugin, will
   provide a gui notification when a branch has changed. When combined with
   lan-notify commits made to published branches become visible to your
   peers on a local network.
 * bzr serve: Running a bzr server provides a convenient way to share your
   branches locally. When you commit into a branch which is currently served
   by a bzr server, then the address of the server is used in the D-Bus
   notifications. This can be very useful for ad-hoc sharing of branches when
   accessing the original data over the internet is slow and or expensive.
   Note that ``bzr server`` does not currently calculate the URL it is running
   at correctly *by default*, so check your machines ip address and then run
   bzr serve manually. e.g.  ``cd ~/source/project-foo; bzr serve
   --port=192.168.1.2:4155``.
"""

from bzrlib import commands

from activity import Activity, LanGateway
from hook import install_hooks


def test_suite():
    import tests
    return tests.test_suite()


install_hooks()


class cmd_dbus_broadcast(commands.Command):
    """A dbus service to reflect revisions to subscribers.
    
    This service runs the bzrlib.plugins.dbus.activity.Broadcast service on the
    session dbus.

    It can be contacted on org.bazaarvcs.plugins.dbus.Broadcast, as
    /org/bazaarvcs/plugins/dbus/Broadcast with interface
    org.bazaarvcs.plugins.dbus.Broadcast.

    The method announce_revision(revision_id, url) will cause the signal
    'Revision' to be raised with two parameters - revision_id and url.
    """

    def run(self):
        Activity().serve_broadcast()

commands.register_command(cmd_dbus_broadcast)


class cmd_lan_notify(commands.Command):
    """Reflect dbus commit notifications onto a LAN."""

    def run(self):
        LanGateway().run()

commands.register_command(cmd_lan_notify)
