/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BsC (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include <qfileinfo.h>
#include <qprocess.h>
#include "Tar.h"
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString Tar::GZIP_EXT = ".gz";
const QString Tar::TAR_EXT  = ".tar";

//*******************************************************************
// Tar
//*******************************************************************
Tar::Tar( QWidget* const parent, const QString& dstPath, const COMPRESS_MODE cm )
: QObject( parent )
, _cm      ( cm )
, _tarPath ( "" )
, _zipPath ( "" )
, _workDir ( "" )
, _process ( 0 )
, _ok      ( FALSE )
, _cmd     ( NONE )
, _idx     ( -1 )
{
	const QFileInfo fi( dstPath );
	const QDir dir = fi.dirPath( TRUE );
	
	if( TRUE == dir.exists() ) {
		if( FALSE == fi.fileName().isEmpty() ) {
			QString tmp = dstPath;
			removeExt( tmp, GZIP_EXT );
			removeExt( tmp, TAR_EXT );
			if( FALSE == tmp.isEmpty() ) {
				_tarPath = tmp + TAR_EXT;
				if( GZIP == _cm ) {
					_zipPath = _tarPath + GZIP_EXT;
				}
				_ok = TRUE;
			}
		}
	}
}
// end of Tar

//*******************************************************************
// create                                                     PUBLIC
//*******************************************************************
bool Tar::create( const QString& workDir, const std::vector<QString>& v )
{
	if( TRUE == _ok ) {
		const QFileInfo fi( workDir );
		const QDir dir = fi.dirPath( TRUE );

		if( TRUE == dir.exists() ) {
			if( FALSE == v.empty() ) {
				_v = v;
				_process = new QProcess( this );
				if( _process ) {
					_idx = 0;
					_cmd = CREATE;
					_process->setWorkingDirectory( workDir );
					connect( _process, SIGNAL( readyReadStdout() ), this, SLOT( readFromStdout() ));
					connect( _process, SIGNAL( readyReadStderr() ), this, SLOT( readFromStderr() ));
					connect( _process, SIGNAL( processExited()   ), this, SLOT( processExited()  ));
					
					process();
				}
			}
		}
	}
	return _ok;
}
// end of create

//*******************************************************************
// process                                                   PRIVATE
//*******************************************************************
void Tar::process()
{
	if(( _process!= 0 ) && ( _cmd != NONE ) && ( _idx != -1 )) {
		// Najpierw ustalmy jaka komende dla tar'a.
		QString comand;
		switch( _cmd ) {
			case CREATE:
				comand = (( 0 == _idx ) ? "--create" : "--update" );
				break;
			case EXTRACT:
				comand = "--extract";
				break;
			case REMOVE:
				comand = "--delete";
				break;
			case UPDATE:
				comand = "--update";
				break;
			case NONE:
				break;
		}
		
		// Przygotowanie polecenia, ktore ma byc wywolane.
		_process->clearArguments();
		_process->addArgument( "tar" );
		_process->addArgument( "--verbose" );
		_process->addArgument( comand );
		_process->addArgument( "--file=" + _tarPath );
		_process->addArgument( _v[_idx++] );
		
		// No to do roboty.
		if( FALSE == _process->start() ) {
			end();
		}
	}
}
// end of process

//*******************************************************************
// removeExt                                                 PRIVATE
//*******************************************************************
void Tar::removeExt( QString& fpath, const QString& ext )
{
	if( TRUE == fpath.endsWith( ext )) {
		const int idx = fpath.findRev( ext );
		if( idx != -1 ) {
			fpath.remove( idx, ext.length() );
		}
	}
}
// end of removeExt

//*******************************************************************
// end                                                       PRIVATE
//*******************************************************************
void Tar::end()
{
	delete _process;
	_process = 0;
	_cmd = NONE;
	_idx = -1;
}
// end of end

//###################################################################
//#                                                                 #
//#                           S L O T S                             #
//#                                                                 #
//###################################################################

//*******************************************************************
// readFromStdout                                       PRIVATE slot
//*******************************************************************
void Tar::readFromStdout()
{
	const QString msg = _process->readStdout();
	emit message( msg );
}
// end of readFromStdout

//*******************************************************************
// readFromStdout                                       PRIVATE slot
//*******************************************************************
void Tar::readFromStderr()
{
	const QString msg = _process->readStderr();
	emit message( msg );
}
// end of readFromStderr

//*******************************************************************
// processExited                                        PRIVATE slot
//*******************************************************************
void Tar::processExited()
{
	const int status = _process->exitStatus();
	
	if( 0 == status ) {
	}
	else {
		end();		
		emit finish();
	}
}
// end of processExited
