/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BsC (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * FILE  : FtpServerList.cpp
 * OBJECT: FtpServerList
 * INFO  : Dialog zawierajacy informacje o serwerach S/FTP.
 *         Tutaj uzytkownik moze zdefiniowac wiele serwerow,
 *         z ktorymi chce sie komunikowac.
 *         Uzytkownik ma mozliwosc definiowania nowych serwerow,
 *         modyfikowania informacji o juz istniejacych oraz usuwanie
 *         niepotrzebych serwerow.
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FtpServersList.h"
#include "FtpServer.h"
#include "Shared.h"
#include "Settings.h"
#include "Config.h"
#include <qpushbutton.h>
#include <qlistview.h>
#include <qlayout.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString FtpServersList::FTP_GROUP        = "FTP";
const QString FtpServersList::FTP_ITEMS        = "Items";
const QString FtpServersList::FTP_CURRENT_ITEM = "Current item";
const int     FtpServersList::OVER_STRETCH     = 100;
const int     FtpServersList::SPACING_WIDTH    = 2;
const int     FtpServersList::MARGIN_WIDTH     = 2;
const QString FtpServersList::CAPTION          = QT_TR_NOOP( "List of FTP servers" );
const QString FtpServersList::HDR_NAME         = QT_TR_NOOP( "Name" );
const QString FtpServersList::HDR_SSL          = "SLL";
const QString FtpServersList::HDR_ADDR         = QT_TR_NOOP( "Address" );
const QString FtpServersList::HDR_PORT         = QT_TR_NOOP( "Port" );
const QString FtpServersList::HDR_DIR          = QT_TR_NOOP( "Directory" );
const QString FtpServersList::HDR_USER         = QT_TR_NOOP( "User" );
const QString FtpServersList::BTN_DELETE       = QT_TR_NOOP( "&Delete" );
const QString FtpServersList::BTN_EDIT         = QT_TR_NOOP( "&Edit" );
const QString FtpServersList::BTN_NEW          = QT_TR_NOOP( "&New" );
const QString FtpServersList::BTN_CONNECT      = QT_TR_NOOP( "&Connect" );
const QString FtpServersList::BTN_EXIT         = QT_TR_NOOP( "&Save && Exit" );
const QString FtpServersList::BTN_CANCEL       = QT_TR_NOOP( "C&ancel" );
const QChar   FtpServersList::ITEM_SEPARATOR   = '|';

//*******************************************************************
// FtpServersList                                        CONSTRUCTOR
//*******************************************************************
FtpServersList::FtpServersList( QWidget* const in_widget )
: QDialog        ( in_widget )
, d_main_layout  ( new QHBoxLayout( this ))
, d_btn_layout   ( new QVBoxLayout )
, d_table_lv     ( new QListView( this ))
, d_delete_btn   ( new QPushButton( tr(BTN_DELETE), this ))
, d_edit_btn     ( new QPushButton( tr(BTN_EDIT), this ))
, d_new_btn      ( new QPushButton( tr(BTN_NEW), this ))
, d_connect_btn  ( new QPushButton( tr(BTN_CONNECT), this ))
, d_exit_btn     ( new QPushButton( tr(BTN_EXIT), this ))
, d_cancel_btn   ( new QPushButton( tr(BTN_CANCEL), this ))
, d_selected_idx ( -1 )
, d_item_on_start( Shared::EmptyStr )
{
	setCaption( tr(CAPTION) );
	setFont( Config::instance()->lfs_default_font()  );
	
	d_table_lv->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
	d_table_lv->addColumn( tr(HDR_NAME) );
	d_table_lv->addColumn( tr(HDR_SSL)  );
	d_table_lv->addColumn( tr(HDR_ADDR) );
	d_table_lv->addColumn( tr(HDR_PORT) );
	d_table_lv->addColumn( tr(HDR_DIR) );
	d_table_lv->addColumn( tr(HDR_USER) );
	d_table_lv->setSelectionMode( QListView::Single );
	d_table_lv->setShowSortIndicator( TRUE );
	d_table_lv->setAllColumnsShowFocus( TRUE );
	d_main_layout->addWidget( d_table_lv );

	d_btn_layout->setSpacing( SPACING_WIDTH );
	d_btn_layout->setMargin( MARGIN_WIDTH );
	d_btn_layout->addStretch( OVER_STRETCH );
	d_btn_layout->addWidget( d_delete_btn );
	d_btn_layout->addWidget( d_edit_btn );
	d_btn_layout->addWidget( d_new_btn );
	d_btn_layout->addStretch( 5 );
	d_btn_layout->addWidget( d_connect_btn );
	d_btn_layout->addWidget( d_exit_btn );
	d_btn_layout->addWidget( d_cancel_btn );
	d_main_layout->addLayout( d_btn_layout );
	
	read();

	connect( d_delete_btn , SIGNAL( clicked() ), this, SLOT( slot_delete()  ));
	connect( d_edit_btn   , SIGNAL( clicked() ), this, SLOT( slot_edit()    ));
	connect( d_new_btn    , SIGNAL( clicked() ), this, SLOT( slot_new()     ));
	connect( d_connect_btn, SIGNAL( clicked() ), this, SLOT( slot_connect() ));
	connect( d_exit_btn   , SIGNAL( clicked() ), this, SLOT( slot_exit()    ));
	connect( d_cancel_btn , SIGNAL( clicked() ), this, SLOT( slot_cancel()  ));
	connect( d_table_lv   , SIGNAL( doubleClicked( QListViewItem*, const QPoint&, int )),
	         this         , SLOT  ( slot_connect() ));
}
// end of FtpServersList

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void FtpServersList::polish()
{
	Shared::polish( this, 40, 40 );
	adjust();
}
// end of polish

//*******************************************************************
// showEvent                                       PRIVATE inherited
//*******************************************************************
void FtpServersList::showEvent( QShowEvent* e )
{
	QDialog::showEvent( e );
	if( FALSE == d_item_on_start.isEmpty() ) {
		select_on_start();
		d_item_on_start = "";
	}
	adjust();
}
// end of showEvent

//*******************************************************************
// select_on_start                                            PUBLIC
//*******************************************************************
void FtpServersList::select_on_start()
{
	QListViewItemIterator it( d_table_lv );
	while ( it.current() ) {
    	if( d_item_on_start == (*it)->text( 0 ) ) {
			d_table_lv->setSelected( *it, TRUE );
			d_table_lv->setCurrentItem( *it );
			d_table_lv->ensureItemVisible( *it );
			break;
		}
		++it;
   }
}

//*******************************************************************
// adjust                                                    PRIVATE
//*******************************************************************
void FtpServersList::adjust()
{
	for( int col = 0; col < 5; ++col ) {
		d_table_lv->adjustColumn( col );
	}
}
// end of adjust

//*******************************************************************
// select_item                                               PRIVATE
//*******************************************************************
void FtpServersList::select_item( const QString& in_name )
{
	QListViewItem* const item = d_table_lv->findItem( in_name, 0 );
	if( item ) {
		d_table_lv->sort();
		d_table_lv->setSelected( item, TRUE );
		d_table_lv->setCurrentItem( item );
		d_table_lv->ensureItemVisible( item );
	}
}
// end of select_item

//*******************************************************************
// change_item                                               PRIVATE
//*******************************************************************
void FtpServersList::change_item( const QString& in_name, const FtpServerInfo& in_fsi )
{
	QListViewItem* const item = d_table_lv->findItem( in_name, 0 );
	if( item ) {
		item->setText( 0, in_fsi.name() );
		item->setText( 1, in_fsi.ssl() ? "Yes" : "No" );
		item->setText( 2, in_fsi.addr() );
		item->setText( 3, QString::number( in_fsi.port() ));
		item->setText( 4, in_fsi.dir()  );
		item->setText( 5, in_fsi.user() );
		
		d_table_lv->sort();
		d_table_lv->setSelected( item, TRUE );
		d_table_lv->setCurrentItem( item );
		d_table_lv->ensureItemVisible( item );
		
		adjust();
	}
}
// end of select_item

//*******************************************************************
// search_data                                          PRIVATE slot
//*******************************************************************
int FtpServersList::search_data( const QString& in_name )
{
	int retval = -1;
	for( unsigned int i = 0; i < d_items.size(); ++i ) {
		if( in_name == d_items[i].name() ) {
			retval = i;
			break;
		}
	}
	return retval;
}
// end of select_item

//*******************************************************************
// add_item                                                  PRIVATE
//*******************************************************************
void FtpServersList::add_item( const FtpServerInfo& in_fsi )
{
	new QListViewItem( d_table_lv,
							/* 0 */ in_fsi.name(),
							/* 1 */ ( in_fsi.ssl() ? "Yes" : "No" ),
							/* 2 */ in_fsi.addr(),
							/* 3 */ QString::number( in_fsi.port() ),
							/* 4 */ in_fsi.dir(),
							/* 5 */ in_fsi.user() );
	adjust();
}
// end of add_item

//*******************************************************************
// read                                                      PRIVATE
//-------------------------------------------------------------------
// Odczyt danych z pliku konfiguracyjnego.
//*******************************************************************
void FtpServersList::read()
{
	Settings::read( FTP_GROUP, FTP_CURRENT_ITEM, d_item_on_start );
	//..............................................
	QStringList item_list;
	Settings::read( FTP_GROUP, FTP_ITEMS, item_list );

	if( FALSE == item_list.empty() ) {
		QStringList::Iterator it = item_list.begin();
		while( it != item_list.end() ) {
			// dla kazdej pozycji z listy tworzymy rekord z danymi.
			QStringList one_item = QStringList::split( ITEM_SEPARATOR, *it, TRUE );
			FtpServerInfo fsi(	one_item[0],
										one_item[1],
										one_item[2].toInt(), 
										one_item[3],
										one_item[4],
										one_item[5],
										( "Yes" == one_item[6] ));
										
			
			d_items.push_back( fsi );	// zapamietujemy rekord.
			add_item( fsi );				// dodajemy pozycje na liscie.
        ++it;
		}
	}
}
// end of read

//*******************************************************************
// save                                                      PRIVATE
//*******************************************************************
void FtpServersList::save()
{
	d_item_on_start = "";
	QListViewItem* const item = d_table_lv->currentItem();
	if( item ) d_item_on_start = item->text( 0 );
	Settings::save( FTP_GROUP, FTP_CURRENT_ITEM, d_item_on_start );
	//..............................................
	QStringList item_list;
	vector<FtpServerInfo>::const_iterator it = d_items.begin();
	while( it != d_items.end() ) {
		const QString item =   (*it).name() + ITEM_SEPARATOR
									+ (*it).addr() + ITEM_SEPARATOR
									+ QString::number( (*it).port() ) + ITEM_SEPARATOR
									+ (*it).dir()  + ITEM_SEPARATOR
									+ (*it).user() + ITEM_SEPARATOR
									+ (*it).pass() + ITEM_SEPARATOR
									+ ((*it).ssl() ? "Yes" : "No" );
		item_list << item;
		++it;
	}
	Settings::save( FTP_GROUP, FTP_ITEMS, item_list );
}
// end of save

//*******************************************************************
// slot_delete                                          PRIVATE slot
//*******************************************************************
void FtpServersList::slot_delete()
{
	QListViewItem* const item = d_table_lv->currentItem();
	if( item ) {
		const QString name = item->text( 0 );
		vector<FtpServerInfo>::iterator it = d_items.begin();
		while( it != d_items.end() ) {
			if( name == (*it).name() ) {
				d_items.erase( it );
				break;
			}
			++it;
		}
		d_table_lv->takeItem( item );
	}
}
// end of slot_delete

//*******************************************************************
// slot_edit                                            PRIVATE slot
//*******************************************************************
void FtpServersList::slot_edit()
{
	QListViewItem* const item = d_table_lv->currentItem();
	if( item ) {
		const int idx = search_data( item->text( 0 ) );
		if( idx != -1 ) {
			FtpServerInfo fsi = d_items[idx];
			FtpServer* fs = new FtpServer( this, fsi );
			if( fs ) {
				if( QDialog::Accepted == fs->exec() ) {
					d_items[idx] = fs->get_fsi();
					change_item( fsi.name(), d_items[idx] );
				}
				delete fs;
				fs = 0;
			}
		}
	}
}
// end of slot_edit

//*******************************************************************
// slot_new                                             PRIVATE slot
//*******************************************************************
void FtpServersList::slot_new()
{
	FtpServerInfo fsi;
	FtpServer* fs = new FtpServer( this, fsi );
	if( fs ) {
		if( QDialog::Accepted == fs->exec() ) {
			fsi = fs->get_fsi();
			d_items.push_back( fsi );
			add_item( fsi );
			select_item( fsi.name() );
		}
		delete fs;
		fs = 0;
	}
}
// end of slot_new

//*******************************************************************
// slot_connect                                         PRIVATE slot
//*******************************************************************
void FtpServersList::slot_connect()
{
	bool ok = FALSE;
	QListViewItem* const item = d_table_lv->currentItem();

	save();	
	if( item ) {
		QListViewItem* const item = d_table_lv->currentItem();
		if( item ) {
			d_selected_idx = search_data( item->text( 0 ) );
			if( d_selected_idx != -1 ) {
				ok = TRUE;
			}
		}
	}
	
	ok ? accept() : reject();
}
// end of slot_connect

//*******************************************************************
// slot_exit                                            PRIVATE slot
//*******************************************************************
void FtpServersList::slot_exit()
{
	save();
	reject();
}
// end of slot_cancel


//*******************************************************************
// slot_cancel                                          PRIVATE slot
//*******************************************************************
void FtpServersList::slot_cancel()
{
	reject();
}
// end of slot_cancel
