/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BsC (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "CopyDialog.h"
#include "InfoField.h"
#include "Config.h"
#include "Shared.h"
#include <qlayout.h>
#include <qprogressbar.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qmessagebox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qframe.h>

/*------- local constants:
-------------------------------------------------------------------*/
const int         CopyDialog::ProgressMinSize  = 300;
const char* const CopyDialog::BoxInfoTitle     = QT_TR_NOOP( "Source and destination" );
const char* const CopyDialog::SrcLabel         = QT_TR_NOOP( "Source:" );
const char* const CopyDialog::DstLabel         = QT_TR_NOOP( "Destination:" );
const char* const CopyDialog::BoxProgressTitle = QT_TR_NOOP( "Progress" );
const char* const CopyDialog::FtpCaption       = QT_TR_NOOP( "FTP copy processing" );
const char* const CopyDialog::LfsCaption       = QT_TR_NOOP( "Copy processing" );
const char* const CopyDialog::RemoveLabel      = QT_TR_NOOP( "remove source file(s)" );
const char* const CopyDialog::OpenReadError    = QT_TR_NOOP( "Can't open the file for reading:\n%1" );
const char* const CopyDialog::OpenWriteError   = QT_TR_NOOP( "Can't open the file for writing:\n%1" );


//*******************************************************************
// CopyDialog                                         CONSTRUCTOR
//*******************************************************************
CopyDialog::CopyDialog( QWidget* const in_parent, bool is_ftp )
: QDialog       ( in_parent )
, d_main_layout ( new QVBoxLayout( this ))
, d_info_box    ( new QGroupBox( 2, Qt::Horizontal, tr( BoxInfoTitle ), this ))
, d_info_frm    ( new QFrame( d_info_box ))
, d_info_grid   ( new QGridLayout( d_info_frm, 2, 2, 4 ))
, d_src_info_lbl( new QLabel( tr( SrcLabel ), d_info_frm ))
, d_dst_info_lbl( new QLabel( tr( DstLabel ), d_info_frm ))
, d_src_info    ( new InfoField( d_info_frm ))
, d_dst_info    ( new InfoField( d_info_frm ))
, d_progress_box( new QGroupBox( 1, Qt::Vertical, tr( BoxProgressTitle ), this ))
, d_progress_bar( new QProgressBar( d_progress_box ))
, d_btn_layout  ( new QHBoxLayout )
, d_remove_cb   ( new QCheckBox( tr( RemoveLabel ), this ))
, d_left_btn    ( new QPushButton( Shared::RunBtnLabel, this ))
, d_right_btn   ( new QPushButton( Shared::CloseBtnLabel , this ))
, d_status      ( new InfoField( this ))
, d_caption     ( is_ftp ? FtpCaption : LfsCaption )
{
    setCaption( tr( d_caption ) );
    setFont( Config::instance()->lfs_default_font()  );

    d_main_layout->setMargin( Shared::LayoutMargin );
    d_main_layout->setSpacing( Shared::LayoutSpacing );

    d_info_grid->setColStretch( 1, Shared::OverStretch );
    d_info_grid->addWidget( d_src_info_lbl, 0, 0, Qt::AlignRight );
	d_info_grid->addWidget( d_src_info    , 0, 1 );
    d_info_grid->addWidget( d_dst_info_lbl, 1, 0, Qt::AlignRight );
	d_info_grid->addWidget( d_dst_info    , 1, 1 );
	d_main_layout->addWidget( d_info_box );

	d_progress_bar->setCenterIndicator( TRUE );
	d_main_layout->addWidget( d_progress_box );

	d_btn_layout->setMargin( Shared::LayoutMargin );
	d_btn_layout->setSpacing( Shared::LayoutSpacing );
	if( is_ftp ) d_remove_cb->hide();
	d_btn_layout->addWidget( d_remove_cb );
    d_btn_layout->addStretch( Shared::OverStretch );
    d_btn_layout->addWidget( d_left_btn );
    d_btn_layout->addWidget( d_right_btn );
    d_main_layout->addLayout( d_btn_layout );

    d_main_layout->addWidget( d_status );
}
// end of CopyDialog

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void CopyDialog::polish()
{
    Shared::polish_width( this, 40 );
}
// end of polish

//*******************************************************************
// runned                                                    PRIVATE
//*******************************************************************
void CopyDialog::runned()
{
    d_remove_cb->setEnabled( FALSE );
    d_left_btn->hide();
    d_right_btn->setText( Shared::BreakBtnLabel );
}
// end of runned

//*******************************************************************
// display_info                                            PROTECTED
//*******************************************************************
void CopyDialog::display_info( const QString& in_src_path, const QString& in_dst_path )
{
	const int MAXLEN = d_src_info->width();
	const QFontMetrics font_metrics( font() );

	QString src_path = in_src_path;
	if( font_metrics.width( src_path ) > MAXLEN ) {
		Shared::clip_path( font_metrics, d_src_info->width(), src_path );
	}
	d_src_info->setText( src_path );
	
	QString dst_path = in_dst_path;
	if( font_metrics.width( dst_path ) > MAXLEN ) {
		Shared::clip_path( font_metrics, d_dst_info->width(), dst_path );
	}
	d_dst_info->setText( dst_path );

	Shared::idle();
}
// end of display_info

//*******************************************************************
// set_status                                              PROTECTED
//*******************************************************************
void CopyDialog::set_status( const QString& in_status )
{
    d_status->setText( in_status );
    Shared::idle();
}
// end of set_status

//*******************************************************************
// set_progress                                           PROTECTED
//*******************************************************************
void CopyDialog::set_progress( const int in_value )
{
	d_progress_bar->setProgress( in_value );
	Shared::idle();
}
// end of set_progress

//*******************************************************************
// set_progress                                           PROTECTED
//*******************************************************************
void CopyDialog::set_progress( const int in_done, const int in_total )
{
	d_progress_bar->setProgress( in_done, in_total );
	Shared::idle();
}
// end of set_progress

//*******************************************************************
// set_total_progress                                      PROTECTED
//*******************************************************************
void CopyDialog::set_total_progress( const int in_value )
{
    d_progress_bar->setTotalSteps( in_value );
	Shared::idle();
}
// end of set_total_progress

//*******************************************************************
// reset_progress                                          PROTECTED
//*******************************************************************
void CopyDialog::reset_progress()
{
    d_progress_bar->reset();
    Shared::idle();
}
// end of reset_progress

//*******************************************************************
// read_open_error                                         PROTECTED
//*******************************************************************
void CopyDialog::read_open_error( const QString& in_fname )
{
    QMessageBox::information( this, tr( d_caption ), tr( OpenReadError ).arg( in_fname ) );
}
// end of read_open_error

//*******************************************************************
// write_open_error                                        PROTECTED
//*******************************************************************
void CopyDialog::write_open_error( const QString& in_fname )
{
    QMessageBox::information( this, tr( d_caption ), tr( OpenWriteError ).arg( in_fname ) );
}
// end of write_open_error
