/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Attr.h"
#include "Shared.h"
#include "Config.h"
#include <qfont.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qiconset.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qfileinfo.h>
#include <qpushbutton.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <unistd.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString Attr::CAPTION           = QT_TR_NOOP( "Access permissions" );
const QString Attr::USER_ATTR_TITEL   = QT_TR_NOOP( "User" );
const QString Attr::GROUP_ATTR_TITEL  = QT_TR_NOOP( "Group" );
const QString Attr::OTHERS_ATTR_TITEL = QT_TR_NOOP( "Others" );
const QString Attr::READ              = QT_TR_NOOP( "read" );
const QString Attr::WRITE             = QT_TR_NOOP( "write" );
const QString Attr::EXEC              = QT_TR_NOOP( "execute" );
const QString Attr::RECURSIVE_LABEL   = QT_TR_NOOP( "&Recursive" );
const QString Attr::INFO_FILE         = QT_TR_NOOP( "File: %1" );
const QString Attr::INFO_DIR          = QT_TR_NOOP( "Dir: %1" );
const QString Attr::FILE_NAME_LABEL   = QT_TR_NOOP( "File name: " );

//*******************************************************************
// Attr
//*******************************************************************
Attr::Attr( QWidget* const in_parent, const QString& in_dir, const ViewTable::SelectedItems& in_items )
: QDialog            ( in_parent )
, d_dir              ( in_dir )
, d_items            ( in_items )
, d_recursiv_checkbox( new QCheckBox  ( tr(RECURSIVE_LABEL), this ))
, d_accept_button    ( new QPushButton( tr(Shared::AcceptBtnLabel), this ))
, d_close_button     ( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_fname_label      ( new QLabel( tr(FILE_NAME_LABEL), this ) )
{
	setCaption( tr(CAPTION) );
	setFont( Config::instance()->lfs_default_font() );

	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin  );
	main_layout->setSpacing( Shared::LayoutSpacing );

	main_layout->addWidget( d_fname_label );

	QHBoxLayout* const current_layout = new QHBoxLayout;
	current_layout->setMargin( Shared::LayoutMargin  );
	current_layout->setSpacing( Shared::LayoutSpacing );

	current_layout->addWidget( make_user_attr() );
	current_layout->addWidget( make_group_attr() );
	current_layout->addWidget( make_others_attr() );
	set_check_boxes();
	main_layout->addLayout( current_layout );
	
	QHBoxLayout* const bottom_layout = new QHBoxLayout;
	bottom_layout->setMargin( Shared::LayoutMargin  );
	bottom_layout->setSpacing( Shared::LayoutSpacing );

	main_layout->addLayout( bottom_layout );
			
	QVBoxLayout* const recursiv_layout = new QVBoxLayout;
	bottom_layout->addLayout( recursiv_layout );
	recursiv_layout->addStretch( Shared::OverStretch );
	recursiv_layout->addWidget( d_recursiv_checkbox );
	if( FALSE == are_dirs() ) d_recursiv_checkbox->hide();

	bottom_layout->addStretch( Shared::OverStretch );
			
	QVBoxLayout* const button_layout = new QVBoxLayout;
	bottom_layout->addLayout( button_layout );
				
	button_layout->addWidget( d_accept_button );
	button_layout->addWidget( d_close_button );
	Shared::add_icon( d_accept_button, Shared::AcceptIcon );
	Shared::add_icon( d_close_button, Shared::CloseIcon );
				
	connect( d_accept_button, SIGNAL( clicked() ), this, SLOT( accept() ));
	connect( d_close_button, SIGNAL( clicked() ), this, SLOT( reject() ));
}
// end of Attr

//*******************************************************************
// make_user_attr                                            PRIVATE
//*******************************************************************
QWidget* Attr::make_user_attr()
{
	QGroupBox* const box  = new QGroupBox( 3, Qt::Vertical, tr(USER_ATTR_TITEL), this );
	d_user_read_checkbox  = new QCheckBox( tr(READ) , box );
	d_user_write_checkbox = new QCheckBox( tr(WRITE), box );
	d_user_exec_checkbox  = new QCheckBox( tr(EXEC) , box );
	return box;
}
// end of make_user_attr

//*******************************************************************
// make_group_attr                                           PRIVATE
//*******************************************************************
QWidget* Attr::make_group_attr()
{
	QGroupBox* const box   = new QGroupBox( 3, Qt::Vertical, tr(GROUP_ATTR_TITEL), this );
	d_group_read_checkbox  = new QCheckBox( tr(READ) , box );
	d_group_write_checkbox = new QCheckBox( tr(WRITE), box );
	d_group_exec_checkbox  = new QCheckBox( tr(EXEC) , box );
	return box;
}
// end of make_group_attr

//*******************************************************************
// make_others_attr                                          PRIVATE
//*******************************************************************
QWidget* Attr::make_others_attr()
{
	QGroupBox* const box    = new QGroupBox( 3, Qt::Vertical, tr(OTHERS_ATTR_TITEL), this );
	d_others_read_checkbox  = new QCheckBox( tr(READ) , box );
	d_others_write_checkbox = new QCheckBox( tr(WRITE), box );
	d_others_exec_checkbox  = new QCheckBox( tr(EXEC) , box );
	return box;
}
// end of make_others_attr

//*******************************************************************
// accept                                                    PRIVATE
//-------------------------------------------------------------------
// Uzytkownik nacisnal klawisz 'Accept'.
// Korzystajac z ustawien check-boxow w dialogu przystepujemy do
// zmiany atrybutow.
//*******************************************************************
void Attr::accept()
{
	const QChar InvertedCommas = '\"';
	
	// Odczytanie atrybutow, ktore wybral uzytkownik.
	const QString attr =	get_selected_attr();
	qWarning( attr );

	// Przygotowanie komendy, ktora bedzie wykonywana.
	QString command = "chmod -f ";
	if( TRUE == d_recursiv_checkbox->isChecked() ) command += "-R ";
	command += ( "0" + attr + Shared::Spc );
	
	// Petla robocza.
	ViewTable::SelectedItems::const_iterator it = d_items.begin();
	while( it != d_items.end() ) {
		display_file_name( (*it)->name() );
		const int retval = system( command + InvertedCommas + (*it)->path() + InvertedCommas );
		if( -1 == retval ) {
		}
		++it;
	}

	// Zakonczenie pracy dialogu.
	QDialog::accept();
}
// end of accept

//*******************************************************************
// show                                            PRIVATE inherited
//*******************************************************************
void Attr::show()
{
	QDialog::show();
	if( FALSE == d_items.empty() ) {
		display_file_name( d_items[0]->name() );
	}
}
// end of show

//*******************************************************************
// display_file_name                                         PRIVATE
//*******************************************************************
void Attr::display_file_name( const QString& in_fname )
{
	d_fname_label->setText( tr(FILE_NAME_LABEL) + in_fname );
	Shared::idle();
}
// end of display_file_name

//*******************************************************************
// are_dirs                                                  PRIVATE
//-------------------------------------------------------------------
// Zwraca informacje, czy wsrod zazmaczonych plikow sa katalogi.
//*******************************************************************
bool Attr::are_dirs()
{
	bool retval = FALSE;
	
	ViewTable::SelectedItems::const_iterator it = d_items.begin();
	while( it != d_items.end() ) {
		if( (*it)->is_dir() ) {
			retval = TRUE;
			break;
		}
		++it;
	}
	return retval;
}
// end of are_dirs

//*******************************************************************
// set_check_boxes                                           PRIVATE
//*******************************************************************
void Attr::set_check_boxes()
{
	// Aktualne atrybuty wyswietlamy tylko i wylacznie wtedy
	// gdy wybrano jeden plik. Jesli jest wiecej plikow
	// nie wiadomo jakie atrybuty nalezalo by wyswietlic.
	if( 1 == d_items.size() ) {
		const QFileInfo& fi = d_items[0]->fi();
		d_user_read_checkbox->setChecked   ( fi.permission( QFileInfo::ReadUser   ));
		d_user_write_checkbox->setChecked  ( fi.permission( QFileInfo::WriteUser  ));
		d_user_exec_checkbox->setChecked   ( fi.permission( QFileInfo::ExeUser    ));
		d_group_read_checkbox->setChecked  ( fi.permission( QFileInfo::ReadGroup  ));
		d_group_write_checkbox->setChecked ( fi.permission( QFileInfo::WriteGroup ));
		d_group_exec_checkbox->setChecked  ( fi.permission( QFileInfo::ExeGroup   ));
		d_others_read_checkbox->setChecked ( fi.permission( QFileInfo::ReadOther  ));
		d_others_write_checkbox->setChecked( fi.permission( QFileInfo::WriteOther ));
		d_others_exec_checkbox->setChecked ( fi.permission( QFileInfo::ExeOther   ));
	}
}
// end of set_check_boxes

//*******************************************************************
// get_selected_attr                                         PRIVATE
//*******************************************************************
QString Attr::get_selected_attr()
{
	int user_attr = 0;
	if( TRUE == d_user_read_checkbox->isChecked()  ) user_attr += 4;
	if( TRUE == d_user_write_checkbox->isChecked() ) user_attr += 2;
	if( TRUE == d_user_exec_checkbox->isChecked()  ) user_attr += 1;
	
	int group_attr = 0;
	if( TRUE == d_group_read_checkbox->isChecked()  ) group_attr += 4;
	if( TRUE == d_group_write_checkbox->isChecked() ) group_attr += 2;
	if( TRUE == d_group_exec_checkbox->isChecked()  ) group_attr += 1;
	
	int others_attr = 0;
	if( TRUE == d_others_read_checkbox->isChecked()  ) others_attr += 4;
	if( TRUE == d_others_write_checkbox->isChecked() ) others_attr += 2;
	if( TRUE == d_others_exec_checkbox->isChecked()  ) others_attr += 1;

	return ( QString::number( user_attr ) 
				+ QString::number( group_attr )
				+ QString::number( others_attr ));
}
// end of get_selected_attr
