/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <glib.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "kozo-private.h"
#include "kozo-db-private.h"
#include "kozo-entry-private.h"
#include "kozo-field-private.h"

#include "kozo-util.h"

/* Tweakable parameters */

/* Amount of array structures per memory block */
#define ARRAY_BLOCK_SIZE 16

static GMutex *array_mem_lock;
static GMemChunk *array_mem_chunk;
static GTrashStack *free_array_structs = NULL;

void
kozo_entry_array_init (KozoEntryArray *array,
                       guint           begin_size)
{
        array->array = g_new (KozoEntry *, begin_size);

        array->len = 0;
        array->max = begin_size;

        obstack_init (&array->obstack);
}

void
kozo_entry_array_resize (KozoEntryArray *array,
                         guint           new_size)
{
        array->array = g_renew (KozoEntry *, array->array, new_size);

        array->max = new_size;
}

void
kozo_entry_array_destroy (KozoEntryArray *array)
{
        obstack_free (&array->obstack, NULL);

        g_free (array->array);
}

KozoEntryArray *
kozo_entry_array_new (guint begin_size)
{
        KozoEntryArray *ret;

        g_mutex_lock (array_mem_lock);

        ret = g_trash_stack_pop (&free_array_structs);
        if (!ret)
                ret = g_chunk_new (KozoEntryArray, array_mem_chunk);

        g_mutex_unlock (array_mem_lock);

        kozo_entry_array_init (ret, begin_size);

        return ret;
}

void
kozo_entry_array_free (KozoEntryArray *array)
{
        kozo_entry_array_destroy (array);

        g_mutex_lock (array_mem_lock);

        g_trash_stack_push (&free_array_structs, array);

        g_mutex_unlock (array_mem_lock);
}

void
kozo_entry_array_append (KozoEntryArray *array,
                         guint8         *data,
                         guint32         length)
{
        if (array->len == array->max)
                kozo_entry_array_resize (array, array->max * 2);

        array->array[array->len] = obstack_alloc (&array->obstack,
                                                  sizeof (KozoEntry));
        array->array[array->len]->data = data;
        array->array[array->len]->length = length;
        array->len++;
}

static void
ensure_directory (void)
{
        int ret;
        char *dirname;

        dirname = g_build_filename (g_get_home_dir (), ".kozo",
                                    "databases", NULL);
        ret = g_mkdir_with_parents (dirname, 0777);
        if (ret < 0) {
                g_warning ("Error creating %s: %s (%d)", dirname,
                           g_strerror (errno), errno);
        }
        g_free (dirname);
}

void
kozo_util_init (void)
{
        ensure_directory ();

        array_mem_lock = g_mutex_new ();

        array_mem_chunk = g_mem_chunk_create (KozoEntryArray,
                                              ARRAY_BLOCK_SIZE,
                                              G_ALLOC_ONLY);
}

void
kozo_util_shutdown (void)
{
        g_mem_chunk_destroy (array_mem_chunk);

        g_mutex_free (array_mem_lock);
}

void
kozo_init (void)
{
        kozo_util_init ();
        kozo_entry_init ();
        kozo_field_init ();
        kozo_db_init ();
}

void
kozo_shutdown (void)
{
        kozo_db_shutdown ();
        kozo_field_shutdown ();
        kozo_entry_shutdown ();
        kozo_util_shutdown ();
}

