/*
 * Copyright (c) 2005, 2006 Jeremy Erickson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <qapplication.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qmessagebox.h>
#include <qiodevice.h>
#include <qtabwidget.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include "qt_types.h"
#include "preferencesact.h"
#include "settings.h"
#include "biblepluginmeta.h"
#include "bibleplugin.h"
#include "plugininfoact.h"
#include <vector>
#include <iostream>
using namespace std;
using namespace bmemcore;

namespace bmemui
{

PreferencesAct::PreferencesAct(QWidget *parent, const char *name)
:PreferencesUI(parent, name), mDirectory(Settings::getPluginDir())
{
    populateList();
    QString file = Settings::getPluginFile();
    if (!file.isEmpty() && QFile::exists(file))
    {
        //int i = mFiles.findIndex(BiblePluginMeta(file));
        //if (i != -1)
        //    mPluginListBox->setSelected(i, true);
        int currIndex = 0;
        for (std::vector<BiblePluginMeta>::iterator it = mFiles.begin();
                it != mFiles.end(); ++it)
        {
            if (it->getMetaFileName() == file)
            {
                mPluginListBox->setSelected(currIndex, true);
                break;
            }
            currIndex++;
        }
    }
    mUseLastCheckBox->setChecked(Settings::getOpenLastFile());
    mDefaultTranslationComboBox->setCurrentText(
            Settings::getDefaultTranslation());
}

BiblePluginMeta PreferencesAct::getMetaStatic(bool &success)
{
    success = true;
    if (!Settings::getPluginFile().isEmpty())
    {
        BiblePluginMeta toReturn(Settings::getPluginFile());
        return toReturn;
    }
    QString directory = Settings::getPluginDir();
    std::vector<BiblePluginMeta> files;
    files = getPluginList(directory);
    bool worked = true;
    if (files.empty())
    {
        //Where the executable resides, important to guessing a path.
        QString appDirPath = qApp->applicationDirPath();
        //First guess is probable on Unix-like systems.  e.g.
        //app at /usr/bin/biblememorizer, plugins in
        // /usr/share/apps/biblememorizer/plugins
        QDir attempt1(appDirPath);
        worked = attempt1.cd("../share/apps/biblememorizer/plugins");
        if (worked)
        {
            directory = attempt1.absPath();
            files = getPluginList(directory);
            worked = !files.empty();
        }
        //Value of "worked" may have changed from last if statement.
        if (worked)
        {
            //Plugins found, set settings for the future.
            Settings::setPluginDir(directory);
        }
        else
        {
            //Second guess more probable on systems like Windows.  e.g.
            //app at C:\Program Files\BibleMemorizer\biblememorizer.exe,
            //plugins in C:\Program Files\BibleMemorizer\plugins
            QDir attempt2(appDirPath + "/plugins");
            worked = attempt2.exists();
            if (worked)
            {
                directory = attempt2.absPath();
                files = getPluginList(directory);
                worked = !files.empty();
            }
            //Again, value of "worked" may have changed.
            if (worked)
            {
                //Plugins found, set settings for the future.
                Settings::setPluginDir(directory);
            }
            else
            {
                //All other attempts have failed, must prompt user.
                while (!worked)
                {
                    int response = QMessageBox::warning(0,
                            tr("Plugin Directory Not Found"),
                            tr("The plugins for BibleMemorizer could not be "
                            "found.\n"
                            "Click OK to select a directory manually, or "
                            "Cancel to abort.\n\n"
                            "If you are seeing this message on the startup of "
                            "BibleMemorizer,\n"
                            "Cancelling will cause BibleMemorizer to exit.  If "
                            "you do not\n"
                            "know what to do, contact the person who gave you "
                            "BibleMemorizer\n"
                            "or post a message in the help forum at "
                            "http://biblememorizer.sf.net"),
                            QMessageBox::Ok | QMessageBox::Default,
                            QMessageBox::Cancel | QMessageBox::Escape);
                    if (response == QMessageBox::Ok)
                    {
                        directory = 
                                BQFileDialog::getExistingDirectory(appDirPath,
                                0, "Get Plugin Directory",
                                tr("Choose Plugin Directory"));
                        files = getPluginList(directory);
                        worked = !files.empty();
                    }
                    else
                    {
                        success = false;
                        break;
                    }
                }
                //Not the case if "break" occured.
                if (worked)
                {
                    Settings::setPluginDir(directory);
                }
            }
        }
    }
    if (worked)
    {
        PreferencesAct selector;
        selector.selectPluginTab();
        int result = selector.exec();
        if (result == QDialog::Accepted)
        {
            return selector.getMeta();
        }
        else
        {
            success = false;
            BiblePluginMeta toReturn;
            return toReturn;
        }
    }
    else
    {
        success = false;
        BiblePluginMeta toReturn;
        return toReturn;
    }
}

BiblePluginMeta PreferencesAct::getMeta()
{
    return mFiles[mPluginListBox->index(mPluginListBox->selectedItem())];
}

void PreferencesAct::selectPluginTab(){
    mTabWidget->setCurrentPage(mTabWidget->indexOf(mPluginTab));
}

void PreferencesAct::mOKButton_clicked()
{
    Settings::setPluginDir(mDirectory);
    Settings::setPluginFile(mFiles[mPluginListBox->index(
            mPluginListBox->selectedItem())].getMetaFileName());
    Settings::setOpenLastFile(mUseLastCheckBox->isChecked());
    Settings::setDefaultTranslation(mDefaultTranslationComboBox->currentText());
    accept();
}

void PreferencesAct::mDirButton_clicked()
{
    QString newDirectory = BQFileDialog::getExistingDirectory(mDirectory,
        this, "Get Plugin Directory", tr("Choose Plugin Directory"));
    if (populateList(newDirectory))
    {
        mDirectory = newDirectory;
    }
    else
    {
        if (!newDirectory.isEmpty())
            int response = QMessageBox::warning(this,
                    tr("Bad Plugin Directory"),
                    tr("No valid plugins found in that directory.\n\n"
                    "Using previous directory."));
    }
}

bool PreferencesAct::populateList()
{
    return populateList(mDirectory);
}

bool PreferencesAct::populateList(const QString& path)
{
    //Assuming failure until proven successful
    bool toReturn = false;
    if (!path.isEmpty())
    {
        std::vector<BiblePluginMeta> files = getPluginList(path);
        if (!files.empty())
        {
            //while (!mFiles.empty())
            //{
            //    BiblePluginMeta* toDelete = mFiles.back();
            //    mFiles.pop_back();
            //    delete toDelete;
            //}
            mFiles = files;
            mPluginListBox->clear();
            std::vector<BiblePluginMeta>::iterator it3 = mFiles.begin();
            while (it3 != mFiles.end())
            {
                mPluginListBox->insertItem((*it3).getName());
                it3++;
            }
            mPluginListBox->setSelected(0, true);
            toReturn = true;
        }
    }
    return toReturn;
}

std::vector<bmemcore::BiblePluginMeta> PreferencesAct::getPluginList(const
        QString& path)
{
    std::vector<BiblePluginMeta> files;
    if (!path.isEmpty())
    {
        //Includes the top directory and immediate subdirectories.
        QDir searchTopSubs(path, "*", QDir::Name | QDir::IgnoreCase,
            QDir::Dirs | QDir::Readable);
        //Store all searchable directories.
        std::vector<QDir> allSearches;
        //Add all the subdirectories.
        for (int i = 0; i < searchTopSubs.count(); i++)
        {
            allSearches.push_back(QDir(searchTopSubs.absFilePath(
                    searchTopSubs[i]), "*.plugin",
                    QDir::Name | QDir::IgnoreCase,
                    QDir::Files | QDir::Readable));
        }
        //Create a list of found files.
        std::vector<QDir>::iterator it1 = allSearches.begin();
        while (it1 != allSearches.end())
        {
            QStringList newEntries = (*it1).entryList();
            //Retrieve absolute versions of the paths, because
            //The files could be located in different directories.
            QStringList::iterator it2 = newEntries.begin();
            while (it2 != newEntries.end())
            {
                files.push_back(BiblePluginMeta((*it1).absFilePath(*it2)));
                it2++;
            }
            it1++;
        }
        //Clean up, since directory objects were dynamically allocated.
        //while (!allSearches.empty())
        //{
        //    QDir *toDelete = allSearches.back();
        //    allSearches.pop_back();
        //    delete toDelete;
        //}
    }
    return files;
}

PreferencesAct::~PreferencesAct()
{
    //while (!mFiles.empty())
    //{
    //    BiblePluginMeta* toDelete = mFiles.back();
    //    mFiles.pop_back();
    //    delete toDelete;
    //}
}

void PreferencesAct::mInfoButton_clicked()
{
    PluginInfoAct infoFrame(getMeta());
    infoFrame.exec();
    //PluginInfoAct(getMeta()).exec();
}

void PreferencesAct::mPluginListBox_selected(int index)
{
    BiblePlugin ourPlugin(mFiles[index]);
    QString oldTranslation = mDefaultTranslationComboBox->currentText();
    mDefaultTranslationComboBox->clear();
    if (ourPlugin.translationsAvailable()){
        mDefaultTranslationComboBox->insertStringList(
                ourPlugin.getTranslations());
    }
    mDefaultTranslationComboBox->setCurrentText(oldTranslation);
}

}

