/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include <glib.h>
#include "printproject.h"
#include "utilities.h"
#include "bible.h"
#include "usfmtools.h"
#include <config.h>
#include "pdfviewer.h"
#include "xmlutils.h"
#include "paper.h"
#include "xmlfo-utils.h"
#include "style.h"
#include "stylesheetutils.h"
#include "formatter.h"
#include "constants.h"
#include "gwrappers.h"
#include "gtkwrappers.h"
#include "directories.h"
#include "fonts.h"
#include "portion_utils.h"
#include "projectutils.h"
#include "books.h"


PrintProject::PrintProject (ProgressWindow * progresswindow)
: genconfig (0),
  projectconfig (""),
  usfm (projectconfig.stylesheet())
{
  // Save and initialize variables.
  myprogresswindow = progresswindow;
  scripture_reordered = NULL;
  usfm_inline_markers = NULL;
  xslfofootnote = NULL;
  xslfoendnote = NULL;
  xslfoxref = NULL;  
  portion_print = false;
  id_page_break_count = 0;
}


PrintProject::~PrintProject ()
{
  if (scripture_reordered) delete scripture_reordered;
  if (usfm_inline_markers) delete usfm_inline_markers;
  if (xslfofootnote) delete xslfofootnote;
  if (xslfoendnote) delete xslfoendnote;
  if (xslfoxref) delete xslfoxref;
}


void PrintProject::print ()
// Formats the project and shows it in a pdf viewer.
{
  // Progressbar.
  myprogresswindow->set_text ("Starting");
  if (myprogresswindow->cancel) {
    return;
  }

  // Check on basic markers.
  if (!usfm_basic_markers_present (usfm, true)) return;

  // Scripture related data.
  // Possible reordering or exclusion of books.
  scripture_reordered = new ScriptureReordered (genconfig.project());
  if (scripture_reordered->books.empty()) {
    gtkw_dialog_info (NULL, "There were no books to print\nSelect some books and try again");
    return;
  }

  // Data for progress bar.
  myprogresswindow->set_iterate (0, 1, scripture_reordered->books.size());
  
  // Prepare for inline text markers, notes and xrefs.
  usfm_inline_markers = new UsfmInlineMarkers (usfm);
  xslfofootnote = new XslFoFootnote (usfm, true);
  xslfoendnote = new XslFoEndnote (usfm, true);
  xslfoxref = new XslFoXref (usfm, true);

  // Prepare a bit
  table_of_contents_running_headers_prepare ();
  
  // Start building the xml-fo file.
  // The various elements of this file are created using objects.
  // When the object goes out of scope, it writes the closing elements.
  {
    XmlFoRoot xmlforoot (&xslfo_lines);
    {
      XmlFoLayoutMasterSet layoutmasterset (&xslfo_lines, true);
    }
    
    // Generate the cover.
    cover_page ();
    
    // Progress bar.
    myprogresswindow->set_iterate (0, 1, scripture_reordered->books.size());

    // Make page sequence, static content, and flow for the Scripture body.
    XmlFoPageSequence pagesequence (&xslfo_lines, false);
    {
      XmlFoStaticContent staticcontent (&xslfo_lines);
    }
    XmlFoFlow flow (&xslfo_lines);
    
    // Generate all the books.
    for (unsigned int i = 0; i < scripture_reordered->books.size(); i++) {
      // Update progress bar.
      myprogresswindow->iterate ();
      myprogresswindow->set_text (scripture_reordered->books[i]);
      if (myprogresswindow->cancel) {
        return;
      }
      // Special handling of some books.
      unsigned int bookid = books_english_to_id (scripture_reordered->books[i]);
      bool spancolumns = false;
      // Skip "Other Material".
      if (books_id_to_type (bookid) == btOtherMaterial) continue;
      // Some books have only one column.
      if (books_id_to_type (bookid) == btFrontBackMatter)
        spancolumns = true;
      // Get chapter count of this book.
      chaptercount = project_get_chapters (genconfig.project(), books_english_to_id (scripture_reordered->books[i])).size();
      // Signal new book to footnotes object.
      xslfofootnote->new_book ();
      // Insert elements for running header.
      headers_insert_anchor (i);
      // Open the book (it cleans the lines on its own).
      vector<ustring> book_lines;
      book_lines = project_retrieve_book (genconfig.project(), books_english_to_id (scripture_reordered->books[i]));
      // Deal with inclusion of a possible portion only.
      portion_print = false;
      select_portion_get_values (genconfig.project(), books_english_to_id (scripture_reordered->books[i]), 
                                 scripture_reordered->portions[i],
                                 portion_chapter_from, portion_verse_from, 
                                 portion_chapter_to, portion_verse_to);
      if (portion_chapter_from == 0)
        if (portion_verse_from == "0") 
          portion_print = true;
      // Format the lines of the book.
      format_general (book_lines, spancolumns);
      // End of book reached: attend to endnotes.
      xslfoendnote->end_book(xslfo_lines);
    }
    // End of project reached: attend to endnotes.
    xslfoendnote->end_project(xslfo_lines);
    // In case there are still some endnotes, dump them.
    xslfoendnote->dump (xslfo_lines);
  }    
  
  // Make a temporary directory where to put the working files and the resulting
  // .pdf file.information. This directory is not removed, because the pdf viewer 
  // needs the .pdf file to be there during viewing or printing.
  string working_directory = gw_build_filename(directories_get_temp(), "project");
  create_directory (working_directory);
  // Produce filename of .fo file.
  string fofilename = gw_build_filename(working_directory, "document.fo");
  // Write the document.
  write_lines (fofilename, xslfo_lines);
  // Show progress.
  if (myprogresswindow->cancel) {
    return;
  }
  string pdffilename = gw_build_filename(working_directory, "document.pdf");
  // Convert the xsl-fo document to .pdf.
  int conversion_result = formatter_convert_to_pdf (fofilename, pdffilename, * myprogresswindow, xslfofootnote->footnotecaller, xslfoxref->footnotecaller);

  // Progressbar: ready.
  myprogresswindow->set_fraction (1);
  myprogresswindow->set_text ("Ready");
  if (myprogresswindow->cancel) {
    return;
  }
  
  // View the .pdf document.
  pdfviewer (pdffilename);

  // Give message if there were errors.
  if (conversion_result != 0) {
    string message = "While formatting the text problems were encountered.\n"
                     "See menu Help - System log for more details.\n"
                     "See the helpfile for a possible solution.";
    gtkw_dialog_error (NULL, message);
  };
}


void PrintProject::format_general (vector <ustring>& lines, bool spancolumns)
/*
Handles general formatting of the markers.
lines: data to be formatted.
spancolumns: forced spanning of the two columns.
*/
{
  // Portion variables.
  bool portion_print_off_next_verse = false;
  unsigned int current_chapter = 0;
  // Chapter number variables.
  vector <ustring> chapter_number_lines;
  unsigned int last_paragraph_before_verse = 0;
  // Go through all the lines.
  myxmlfoblock = NULL;
  for (unsigned int ln = 0; ln < lines.size(); ln++) {
    // Get the line.
    ustring line = lines[ln];
    // Change certain characters to xml entities.
    xml_handle_entities (line, NULL);
    // Deal with possible "inserted" and "removed" marks.
    replace_text (line, STRIKE_THROUGH_BEGIN, "<fo:inline text-decoration=\"line-through\">");
    replace_text (line, STRIKE_THROUGH_END, "</fo:inline>");
    replace_text (line, BOLD_BEGIN, "<fo:inline font-weight=\"bold\">");
    replace_text (line, BOLD_END, "</fo:inline>");
    // Deal with footnotes.
    xslfofootnote->transform (myxmlfoblock, line);
    // Deal with crossreferences.
    xslfoxref->transform (myxmlfoblock, line);
    // Deal with endnotes. 
    if (portion_print)
      xslfoendnote->transform (myxmlfoblock, line, usfm_inline_markers);
    // Deal with inline text.
    usfm_handle_inline_text (line, usfm_inline_markers, myxmlfoblock, imXslFo, NULL);
    // Get the style belonging to the marker.
    ustring marker = usfm_extract_marker (line);
    if (usfm.is_identifier (marker)) {
      // Handle some identifiers.
      if (!portion_print)
        continue;
      IdentifierType identifiertype = usfm.identifier_get_subtype (marker);
      // Table of contents.
      if ((identifiertype == itLongTOC) || (identifiertype == itShortTOC))
        table_of_contents_insert_anchor ();
      // New book.
      if (identifiertype == itBook) {
        if (myxmlfoblock) delete myxmlfoblock;
        myxmlfoblock = NULL;
        ustring line;
        if (usfm.userbool1()) {
          // The first id in a project should not cause a page break, else
          // XEP gives an error.
          if (id_page_break_count > 0) {
            line.append ("<fo:block break-after=\"");
            if (usfm.userbool2()) line.append ("odd-page");
            else line.append ("page");
            line.append ("\"></fo:block>");
          }
          id_page_break_count++;
        }
        xslfo_lines.push_back (line);
        // Add the anchor for the running header.
        for (unsigned int i = 0; i < running_headers_anchor.size(); i++) {
          xslfo_lines.push_back (running_headers_anchor[i]);
        }
        running_headers_anchor.clear();
      }
    } else if (usfm.is_verse_number (marker)) {
      // Because of dealing with portions to include/exclude, handle verse first.
      // Get verse number. Handle combined verses too, e.g. 10-12b, etc.
      size_t position = line.find (" ");
      position = CLAMP (position, 0, line.length());
      ustring versenumber = line.substr (0, position);
      position++;
      line.erase (0, position);
      // See whether this chapter.verse is within our portion to print.
      if (portion_print_off_next_verse)
        portion_print = false;
      if ((current_chapter == portion_chapter_from) && (versenumber == portion_verse_from)) {
        portion_print = true;
      }
      if ((current_chapter >= portion_chapter_to) && (versenumber == portion_verse_to))
        portion_print_off_next_verse = true;
      if (!portion_print)
        continue;
      // There was a bug that the chapter number was missing from a book,
      // and that results in inline text being put straight under the flow, 
      // which is an error in XSL. Solve that here by checking whether
      // a block has been opened. If not, open a default block.
      if (myxmlfoblock == NULL) {
        myxmlfoblock = new XmlFoBlock (&xslfo_lines, 12, 100, OFF, OFF, OFF, OFF, JUSTIFIED, 0, 0, 0, 0, 0, false, false);
      }
      // If a chapter number is pending, print it here, and omit verse 1.
      bool printversenumber = true;
      if (!chapter_number_lines.empty()) {
        for (unsigned int cl = 0; cl < chapter_number_lines.size(); cl++)
          xslfo_lines.push_back (chapter_number_lines[cl]);
        chapter_number_lines.clear();
        if (versenumber == "1") printversenumber = false;
      }
      // When the usfm is a verse number, then the number is put in the 
      // format specified by the stylesheet, but the remaining part of the
      // line inherits the formatting from the block it is in, usually
      // a paragraph block.
      if (printversenumber) {
        xslfo_lines.push_back(XmlFoInlineText (versenumber, myxmlfoblock, usfm.fontpercentage(), usfm.italic(), usfm.bold(), usfm.underline(), usfm.smallcaps(), usfm.superscript(), usfm.color()));
      }
      xslfo_lines.push_back(line);
    } else if (usfm.is_starting_paragraph (marker)) {
      if (!portion_print)
        continue;
      if (myxmlfoblock) {
        delete myxmlfoblock;
      }
      bool span_columns = usfm.spancolumns();
      if (spancolumns) span_columns = true;
      double firstlineindent = usfm.firstlineindent();
      if (ln == last_paragraph_before_verse) firstlineindent = 0;
      myxmlfoblock = new XmlFoBlock (&xslfo_lines, 
        usfm.fontsize(), 100, usfm.italic(), usfm.bold(), usfm.underline(), 
        usfm.smallcaps(), usfm.justification(),
        usfm.spacebefore(), usfm.spaceafter(), usfm.leftmargin(),
        usfm.rightmargin(), firstlineindent, span_columns,
        usfm.paragraph_get_subtype (marker) != ptNormalParagraph);
      if (!line.empty()) xslfo_lines.push_back(line);
    } else if (usfm.is_inline_text (marker)) {
      if (!portion_print)
        continue;
      // Inline text, has been dealt with before (therefore should never occur here).
      xslfo_lines.push_back(XmlFoInlineText (line, myxmlfoblock, usfm.fontpercentage(), usfm.italic(), usfm.bold(), usfm.underline(), usfm.smallcaps(), usfm.superscript(), usfm.color()));
    } else if (usfm.is_chapter_number (marker)) {
      // See whether this chapter is within our portion to print.
      current_chapter = convert_to_int (line);
      if ((current_chapter == portion_chapter_from) && (portion_verse_from == "0")) {
        portion_print = true;
      }
      if (current_chapter > portion_chapter_to)
        portion_print = false;
      if (!portion_print)
        continue;
      // Print no chapter number for books with only one chapter: chapter 0 and 1.
      if (chaptercount == 2)
        continue;
      // See whether to print the chapter number here, or later on.
      bool printchapterlater = usfm.userbool1();
      // Close possible open block.
      if (!printchapterlater) {
        if (myxmlfoblock) {
          delete myxmlfoblock;
        }
      }
      // Signal new chapter to footnotes object.
      xslfofootnote->new_chapter ();      
      // Insert marker for chapter number.
      xslfo_lines.push_back ("<fo:block>");
      xslfo_lines.push_back ("  <fo:marker marker-class-name=\"chapter\">");
      xslfo_lines.push_back (line);
      xslfo_lines.push_back ("  </fo:marker>");
      xslfo_lines.push_back ("</fo:block>");
      // Insert or prepare chapter text.
      if (printchapterlater) {
        chapter_number_lines.push_back ("  <fo:float float=\"start\">");
        {
          XmlFoBlock xmlfoblk (&chapter_number_lines,
            usfm.fontsize(), usfm.userint1(), 
            usfm.italic(), usfm.bold(), usfm.underline(),
            usfm.smallcaps(), usfm.justification(),
            usfm.spacebefore(), usfm.spaceafter(), usfm.leftmargin(),
            usfm.rightmargin(), usfm.firstlineindent(), usfm.spancolumns(), 
            true); 
          chapter_number_lines.push_back (line);
        }
        chapter_number_lines.push_back ("  </fo:float>");
        last_paragraph_before_verse = get_last_paragraph_before_verse (ln, lines);
      } else {
        myxmlfoblock = new XmlFoBlock (&xslfo_lines,
          usfm.fontsize(), usfm.userint1(), 
          usfm.italic(), usfm.bold(), usfm.underline(),
          usfm.smallcaps(), usfm.justification(),
          usfm.spacebefore(), usfm.spaceafter(), usfm.leftmargin(),
          usfm.rightmargin(), usfm.firstlineindent(), usfm.spancolumns(),
          true);
        xslfo_lines.push_back(line);
      }
    } else if (usfm.is_peripheral (marker)) {
      if (!portion_print)
        continue;
      if (usfm.peripheral_get_subtype (marker) == ptTableOfContents)
        table_of_contents_insert_body ();
    } else if (usfm.is_picture (marker)) {
      if (!portion_print)
        continue;
      picture (marker, line);
    } else if (usfm.is_pagebreak (marker)) {
      if (!portion_print)
        continue;
      xslfo_lines.push_back ("<fo:block break-after=\"page\"></fo:block>");
    } else if (xslfoendnote->new_marker (marker)) {
      // Printing of endnotes when encountering a certain marker.
      if (myxmlfoblock) {
        delete myxmlfoblock;
        myxmlfoblock = NULL;
      }
      xslfoendnote->dump (xslfo_lines);
    } else if (marker == ELASTIC_MARKER) {
      // Elastics
      if (myxmlfoblock) delete myxmlfoblock;
      myxmlfoblock = new XmlFoBlock (&xslfo_lines, 12, 100, OFF, OFF, OFF, OFF, JUSTIFIED, 0, 0, 0, 0, 0, true, false);
      xslfo_lines.push_back (ELASTIC_XEP);
    } else {
      if (!portion_print)
        continue;
      // Fallback for unknown marker or no marker.
      xslfo_lines.push_back(line);
    }
  } 
  // Close possible last block of text.
  if (myxmlfoblock) {
    delete myxmlfoblock;
  }
  // Balance text columns using a block that spans the columns.
  xslfo_lines.push_back ("<fo:block span=\"all\"></fo:block>");
}


void PrintProject::cover_page ()
// Generate the cover.
{
  // Store the lines containing the cover.  
  vector <ustring> lines;
  
  // Collect the lines: Look for the \cov section in the Other Material book. 
  for (unsigned int i = 0; i < scripture_reordered->books.size(); i++) {
    if (books_id_to_type (books_english_to_id (scripture_reordered->books[i])) != btOtherMaterial) 
      continue;
    vector <ustring> rawlines = project_retrieve_chapter (genconfig.project(), books_english_to_id (scripture_reordered->books[i]), 0);
    bool within_cover_section = false;
    for (unsigned int i2 = 0; i2 < rawlines.size(); i2++) {
      ustring line = rawlines[i2];
      ustring marker = usfm_extract_marker (line);
      if (usfm.is_peripheral (marker))
        if (usfm.peripheral_get_subtype (marker) != ptCover)
          within_cover_section = false;
      if (within_cover_section) {
        lines.push_back (rawlines[i2]);
      }
      if (usfm.is_peripheral (marker)) {
        if (usfm.peripheral_get_subtype (marker) == ptCover)
          within_cover_section = true;
      }
    }
  }
  
  // Cover found?
  if (lines.empty()) return;
    
  // Format cover.
  XmlFoPageSequence pagesequence (&xslfo_lines, true);
  XmlFoFlow flow (&xslfo_lines);
  portion_print = true;
  portion_verse_from.clear();
  portion_verse_to.clear();
  format_general (lines, true);
}


void PrintProject::headers_insert_anchor (unsigned int index)
// Inserts an anchor in the text to allow for retrieval of the running headers.
{
  XmlFoBlock block (&running_headers_anchor, 12, 100, OFF, OFF, OFF, OFF, "last-justify", 0, 0, 0, 0, 0, true, false);
  running_headers_anchor.push_back("        <fo:marker marker-class-name=\"book\">");
  running_headers_anchor.push_back(running_headers[index]);
  running_headers_anchor.push_back("        </fo:marker>");
}


void PrintProject::table_of_contents_running_headers_prepare ()
// Prepare for the table of contents. 
{
  // Go through all the books and collect data for the running headers and the
  // table of contents.
  myprogresswindow->set_text ("Preparing");
  for (unsigned int i = 0; i < scripture_reordered->books.size(); i++) {
    // Progress.
    myprogresswindow->iterate();
    if (myprogresswindow->cancel) {
      return;
    }

    // Get the running header from chapter zero.
    ustring runningheader;
    vector <ustring> lines;
    lines = project_retrieve_chapter (genconfig.project(), books_english_to_id (scripture_reordered->books[i]), 0);
    // Search for data in these lines.
    for (unsigned int i2 = 0; i2 < lines.size(); i2++) {
      ustring line = lines[i2];
      ustring marker = usfm_extract_marker (line);
      if (usfm.is_identifier (marker)) {
        IdentifierType id_type = usfm.identifier_get_subtype (marker);
        if ((id_type == itRunningHeader) || 
            (id_type == itLeftRunningHeader) || 
            (id_type == itRightRunningHeader))
          runningheader = line;
      }
    }      
    // Store for later use.
    running_headers.push_back (runningheader);
    
    // Get the toc markers from the whole book.
    lines.clear();
    lines = project_retrieve_book (genconfig.project(), books_english_to_id (scripture_reordered->books[i]));
    // Search for data in these lines.
    for (unsigned int i2 = 0; i2 < lines.size(); i2++) {
      ustring line = lines[i2];
      ustring marker = usfm_extract_marker (line);
      if (usfm.is_identifier (marker)) {
        IdentifierType type = usfm.identifier_get_subtype (marker);
        if ((type == itLongTOC) || (type == itShortTOC)) {
          // Store text.
          table_of_contents_texts.push_back (line);
          // Store the id to be used. Spaces are not allowed in the ids, 
          // neither can they start with a number.
          ustring toc_id = "toc_" + convert_to_string (books_english_to_id (scripture_reordered->books[i])) + "_" + convert_to_string (i2);
          table_of_contents_identifiers.push_back (toc_id);
        }
      }
    }
    
  }
}


void PrintProject::table_of_contents_insert_body ()
// Generate the table of contents.
{
  if (table_of_contents_texts.empty()) return;
  if (myxmlfoblock) delete myxmlfoblock;
  myxmlfoblock = NULL;
  for (unsigned int i = 0; i < table_of_contents_texts.size(); i++) {
    XmlFoBlock block (&xslfo_lines, 12, 100, OFF, OFF, OFF, OFF, "last-justify", 0, 0, 0, 0, 0, true, false);
    xslfo_lines.push_back(table_of_contents_texts[i]);
    xslfo_lines.push_back("        <fo:leader leader-pattern=\"dots\"/>");
    ustring line;
    line = "        <fo:page-number-citation ref-id=\"";
    line.append (table_of_contents_identifiers[i]);
    line.append ("\"/>");
    xslfo_lines.push_back(line);
  }
}


void PrintProject::table_of_contents_insert_anchor ()
// Inserts an anchor in the text so the TOC can reference to it.
{
  if (table_of_contents_texts.empty()) return;
  XmlFoBlock block (&xslfo_lines, 12, 100, OFF, OFF, OFF, OFF, "last-justify", 0, 0, 0, 0, 0, false, false);
  ustring line = "      <fo:block id=\"";
  line.append (table_of_contents_identifiers[0]);
  line.append ("\">");
  xslfo_lines.push_back(line);
  xslfo_lines.push_back("      </fo:block>");
  table_of_contents_texts.erase (table_of_contents_texts.begin());
  table_of_contents_identifiers.erase (table_of_contents_identifiers.begin());
}


void PrintProject::picture (const ustring& marker, const ustring& line)
// Prints a picture.
{
  // The pictures comes in the form of, e.g.  |biblesociety.gif|col||||\fig*
  // The first marker has already been taken out.
  
  // Working copy.
  ustring myline (line);
  
  // Extract the relevant part of it, and any remainder.
  ustring remainder;
  ustring endmarker = "\\" + marker + "*";
  size_t position = myline.find (endmarker);
  if (position != string::npos) {
    remainder = myline.substr (position, 1000000);
    remainder.erase (0, endmarker.length());
    myline.erase (position, 1000000);
  }

  // Parse the figure data into its pieces.
  vector <ustring> elements;
  myline.append ("|");
  size_t separatorpos = myline.find ("|");
  while (separatorpos != string::npos) {
    ustring word = myline.substr (0, separatorpos);
    elements.push_back (word);
    myline.erase (0, ++separatorpos);
    separatorpos = myline.find ("|");
  }
  ustring desc;
  ustring file;
  ustring size;
  ustring loc;
  ustring copy;
  ustring cap;
  ustring ref;
  for (unsigned int i = 0; i < elements.size(); i++) {
    if (i == 0) desc = elements[i];
    if (i == 1) file = elements[i];
    if (i == 2) desc = elements[i];
    if (i == 3) loc = elements[i];
    if (i == 4) copy = elements[i];
    if (i == 5) cap = elements[i];
    if (i == 6) ref = elements[i];
  }
  
  // Make an existing path to the picture.
  // If a relative path is given, it looks in the pictures data directory, then
  // in the package data directory.
  if (file.substr (0, 1) != G_DIR_SEPARATOR_S) {
    ustring file2;
    file2 = gw_build_filename (directories_get_pictures(), file);
    if (g_file_test (file2.c_str(), G_FILE_TEST_IS_REGULAR))
      file = file2;
    else {
      file2 = gw_build_filename (PACKAGE_DATA_DIR, file);
      if (g_file_test (file2.c_str(), G_FILE_TEST_IS_REGULAR))
        file = file2;
    }
  }
  // If the picture file is there, fine, else quit.
  if (!g_file_test (file.c_str(), G_FILE_TEST_IS_REGULAR)) {
    gw_warning ("Picture " + file + " does not exist");
    return;
  }

  // Open a new block, insert graphic, and close it again.
  XmlFoBlock * myblock = new XmlFoBlock (&xslfo_lines, 12, 100, OFF, OFF, OFF, OFF, CENTER, 0, 0, 0, 0, 0, false, false);
  ustring graphic = "<fo:external-graphic src=\"url('";
  graphic.append (file);
  graphic.append ("')\" content-height=\"100%\" content-width=\"100%\"/>");
  xslfo_lines.push_back (graphic);
  delete myblock;
  // Just to be sure, write the remaining text.
  xslfo_lines.push_back (remainder);
}


unsigned int PrintProject::get_last_paragraph_before_verse (unsigned int currentline, vector <ustring>& lines)
/*
Starting from "currentline" it searches forward through "lines" to find the line
number that has a usfm marker that opens a new paragraph, and that is the last
such line before any verse marker.
The purpose of this is to set both the left margin and indent of the paragraph 
with verse one to zero, if the chapter number is printed beside the verse 
number, so that the layout of the page looks better.
*/
{
  unsigned int linenumber = 0;
  for (unsigned int i = currentline; i < lines.size(); i++) {
    ustring line = lines[i];
    ustring marker = usfm_extract_marker (line);    
    if (usfm.is_starting_paragraph (marker)) {
      linenumber = i;
    } else if (usfm.is_verse_number (marker)) {
      return linenumber;
    }
  }
  return 0;
}
