/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/*
The network preferences are stored in
/Library/Preferences/SystemConfiguration/preferences.plist, which must not be
accessed directly, but using the SCPreferences and/or SCPreferencesPath APIs.
The layout of the preferences file is as follows (only relevant data is shown):

{
	CurrentSet = $SET_PATH
	NetworkServices = {
		// multiple services
		$GLOBAL_SERVICE_ID = {
			Proxies = {
				// proxy configuration
			}
		}
	}
	Sets = {
		// multiple sets
		$SET_ID = {
			UserDefinedName = $SET_NAME
			Network = {
				Service = {
					// multiple service references
					$LOCAL_SERVICE_ID = {
						__LINK__ = $GLOBAL_SERVICE_PATH
					}
				}
				Global = {
					IPv4 = {
						ServiceOrder = (
							// ordered list of service id's
							$LOCAL_SERVICE_ID
						)
					}
					// There is no IPv6 -> ServiceOrder list.
					// I think it's because ServiceOrder is inapropriately
					// placed and should be located one level up.
				}
			}
		}
	}
}

Some explanations:
Set: a network profile, also known as Location.
Service: a configuration of a particular network interface.
$SET_ID: a string (usually GUID), uniquely identifying a set.
$SET_PATH: "/Sets/$SET_ID"
$SET_NAME: what you see in Apple Menu -> Locations
$GLOBAL_SERVICE_ID: a string (usually GUID), uniquely identifying a service.
$GLOBAL_SERVICE_PATH: "/NetworkServices/$GLOBAL_SERVICE_ID"
$LOCAL_SERVICE_ID: a string (usually GUID), serving as a reference to
a particular service in /NetworkServices, within a particular set.

Currently, a $LOCAL_SERVICE_ID has the same value as the referenced
$GLOBAL_SERVICE_ID, but that's not documented anywhere, so we don't
rely on this fact. When generating $LOCAL_SERVICE_ID's however, we
make them equal to the corresponging $GLOBAL_SERVICE_ID's.
To avoid possible errors, we never pass/return $GLOBAL_SERVICE_ID's,
but instead we do $GLOBAL_SERVICE_PATH's.

Theoretically, several sets could share the same services, but in practice
that confuses the Network Preferences application.  Because of that, we
duplicate services whenever we duplicate the set referencing them.
When deleting a set however, we don't delete all the referenced services,
but only the ones that are not referenced by other sets.
*/

#ifndef NETWORKPREFS_H_
#define NETWORKPREFS_H_

#include <SystemConfiguration/SystemConfiguration.h>
#include "NonCopyable.h"
#include "RefPtr.h"
#include "OutputFunction.h"
#include <utility>

class NetworkPrefs
{
	DECLARE_NON_COPYABLE(NetworkPrefs)
public:
	typedef std::pair<StringRef, StringRef> StringPair;
	
	NetworkPrefs(CFStringRef client_name);
	
	~NetworkPrefs();
	
	RefPtr<SCPreferencesRef> const& getRep() { return m_ptrPrefs; }
	
	bool lock();
	
	bool tryLock();
	
	bool isLocked() const { return m_isLocked; }
	
	bool applyChanges();
	
	StringRef getCurrentSetId() const;
	
	void setCurrentSetId(StringRef const& set_id);
	
	StringRef getSetName(StringRef const& set_id) const;
	
	// Outputs pairs of set_id, set_name
	void enumSets(OutputFunction<StringPair>& output) const;
	
	// Outputs global_service_path's
	void enumGlobalServices(OutputFunction<StringRef>& output) const;
	
	// Outputs pairs of local_service_id, global_service_path
	void enumServicesReferencedBySet(
		StringRef const& set_id, OutputFunction<StringPair>& output) const;
	
	// Outputs pairs of set_id, set_name
	void enumSetsReferencingService(
		StringRef const& service_path, OutputFunction<StringPair>& output) const;
	
	// returns the new set_id
	StringRef duplicateSet(
		StringRef const& set_id, StringRef const& new_set_name,
		StringRef const& new_set_id_hint = StringRef());
	
	void duplicateSetServices(StringRef const& set_id);
	
	void deleteSet(StringRef const& set_id);
	
	void deleteUnreferencedServices();
	
	void switchCurrentSetIfRemoved();
private:
	class StringMapper;
	
	bool doLock(bool wait);
	
	void setServiceReferences(StringRef const& set_id, DictRef const& refs);
	
	void updateServiceOrder(StringRef const& set_id, StringMapper const& mapper);
	
	static void addServiceReference(MutableDictRef const& dict,
		StringRef const& local_service_id, StringRef const& global_service_path);
	
	static DictRef createLink(StringRef const& target);
	
	RefPtr<SCPreferencesRef> m_ptrPrefs;
	bool m_isLocked;
};

#endif
