/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef WXGUI_CONTENTFILTERS_H_
#define WXGUI_CONTENTFILTERS_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "OutputFunction.h"
#include <ace/config-lite.h>
#include <wx/filename.h>
#include <memory>

class FilterGroupTag;

namespace wxGUI
{

class ContentFilterGroup;

/**
 * \brief A collection of ContentFilterGroup objects.
 */
class ContentFilters
{
	DECLARE_NON_COPYABLE(ContentFilters)
public:
	ContentFilters(wxFileName const& conf_dir);
	
	~ContentFilters();
	
	void swap(ContentFilters& other);
	
	void load();
	
	void apply() const;
	
	void loadAndApply() { load(); apply(); }
	
	/**
	 * \brief Find a filter group by tag.
	 * \return A pointer to the group found, or 0 if the group
	 *         wasn't found.
	 */
	ContentFilterGroup const* findGroup(FilterGroupTag const& tag) const;
	
	/**
	 * \return true on success, false if such group already exists.
	 */
	bool addGroup(ContentFilterGroup const& group);
	
	/**
	 * \return true on success, false if the group wasn't found.
	 */
	bool eraseGroup(FilterGroupTag const& tag);
	
	/**
	 * \return true on success, false if the group wasn't found.
	 */
	bool updateGroup(ContentFilterGroup const& group);
	
	/**
	 * \brief Apply a functor to each ContentFilterGroup.
	 *
	 * Functors will be called in the order determined
	 * by FilterGroupTag::OrderComp.
	 */
	template<typename Func>
	void foreach(Func f) const;
private:
	class Impl;
	
	void foreachImpl(OutputFunction<ContentFilterGroup>& func) const;
	
	std::auto_ptr<Impl> m_ptrImpl;
};


template<typename Func>
inline void
ContentFilters::foreach(Func f) const
{
	TemplateOutputFunction<ContentFilterGroup, Func> func(f);
	foreachImpl(func);
}

inline void swap(ContentFilters& o1, ContentFilters& o2)
{
	o1.swap(o2);
}

} // namespace wxGUI

#endif
