/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "BasicConfigDialog.h"
#include "Application.h"
#include "OperationLog.h"
#include "GlobalState.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "ProxyDescriptor.h"
#include "StringUtils.h"
#include "SymbolicInetAddr.h"
#include "CompiledImages.h"
#include "ArraySize.h"
#include "EncOps.h"
#include "types.h"
#include <boost/lexical_cast.hpp>
#include <gtkmm/window.h>
#include <gtkmm/entry.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/radiobutton.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/button.h>
#include <gtkmm/alignment.h>
#include <gtkmm/label.h>
#include <gtkmm/frame.h>
#include <gtkmm/stock.h>
#include <gtkmm/table.h>
#include <gtkmm/messagedialog.h>
#include <glibmm/ustring.h>
#include <glibmm/refptr.h>
#include <list>
#include <sstream>
#include <cassert>

using namespace std;

namespace GtkGUI
{

class BasicConfigDialog::Impl : public Gtk::Window
{
public:
	Impl();
	
	virtual ~Impl();
	
	static void showWindow();
private:
	virtual void on_hide();
	
	void loadConfig();
	
	void setCurrentListenPort(Glib::ustring const& port);
	
	void onApply();
	
	bool restoreOldConfig(Config const& config);
	
	void showError(Glib::ustring const& msg);
	
	void showWarning(Glib::ustring const& msg);
	
	static Impl* m_spInstance;
	Glib::ustring m_curListenPort;
	Gtk::Entry* m_pListenPort;
	Gtk::CheckButton* m_pTrayAnimation;
	Gtk::RadioButtonGroup m_pageCleanupGroup;
	Gtk::RadioButton* m_pPageCleanupOff;
	Gtk::RadioButton* m_pPageCleanupSafe;
	Gtk::RadioButton* m_pPageCleanupMax;
};


void
BasicConfigDialog::showWindow()
{
	Impl::showWindow();
}


BasicConfigDialog::Impl* BasicConfigDialog::Impl::m_spInstance = 0;

BasicConfigDialog::Impl::Impl()
:	m_pListenPort(0),
	m_pTrayAnimation(0),
	m_pPageCleanupOff(0),
	m_pPageCleanupSafe(0),
	m_pPageCleanupMax(0)
{
	set_title("Basic Configuration");
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	set_border_width(6);
	
	Gtk::VBox* top_vbox = manage(new Gtk::VBox);
	add(*top_vbox);
	top_vbox->set_spacing(3);
	
	Gtk::Alignment* port_align = manage(new Gtk::Alignment(0.5, 0.5, 0.0, 0.0));
	top_vbox->pack_start(*port_align);
	
	Gtk::HBox* port_hbox = manage(new Gtk::HBox);
	port_align->add(*port_hbox);
	
	port_hbox->set_spacing(10);
	port_hbox->pack_start(*manage(new Gtk::Label("Listen Port")));
	
	m_pListenPort = manage(new Gtk::Entry);
	port_hbox->pack_start(*m_pListenPort);
	m_pListenPort->set_size_request(50);
	
	Gtk::Frame* ads_frame = manage(new Gtk::Frame(" Ads "));
	top_vbox->pack_start(*ads_frame, Gtk::PACK_SHRINK);
	
	Gtk::VBox* ads_vbox = manage(new Gtk::VBox);
	ads_frame->add(*ads_vbox);
	ads_vbox->set_border_width(3);
	ads_vbox->set_spacing(3);
	
	m_pPageCleanupOff = manage(
		new Gtk::RadioButton(
			m_pageCleanupGroup,
			"Substitute ads with clickable replacements."
		)
	);
	ads_vbox->pack_start(*m_pPageCleanupOff, Gtk::PACK_SHRINK);
	
	m_pPageCleanupSafe = manage(
		new Gtk::RadioButton(
			m_pageCleanupGroup,
			"Remove ads completely, where it's safe."
		)
	);
	ads_vbox->pack_start(*m_pPageCleanupSafe, Gtk::PACK_SHRINK);
	
	m_pPageCleanupMax = manage(
		new Gtk::RadioButton(
			m_pageCleanupGroup,
			"Always remove ads completely."
		)
	);
	ads_vbox->pack_start(*m_pPageCleanupMax, Gtk::PACK_SHRINK);
	
	m_pTrayAnimation = manage(new Gtk::CheckButton("Tray icon animation"));
	top_vbox->add(*m_pTrayAnimation);
	
	Gtk::Alignment* bbox_align = manage(new Gtk::Alignment(0.5, 0.5, 0.0, 0.0));
	top_vbox->pack_start(*bbox_align, Gtk::PACK_SHRINK);
	bbox_align->set_border_width(2);
	
	Gtk::HButtonBox* bbox = manage(new Gtk::HButtonBox);
	bbox_align->add(*bbox);
	bbox->set_spacing(10);
	
	Gtk::Button *cancel_btn = manage(new Gtk::Button(Gtk::Stock::CANCEL));
	bbox->pack_start(*cancel_btn, Gtk::PACK_SHRINK);
	cancel_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &Impl::hide)
	);
	
	Gtk::Button *ok_btn = manage(new Gtk::Button(Gtk::Stock::OK));
	bbox->pack_start(*ok_btn, Gtk::PACK_SHRINK);
	ok_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &Impl::onApply)
	);
	
	loadConfig();
	cancel_btn->grab_focus();
	show_all_children();
	
	Application::instance()->destroySignal().connect(
		sigc::mem_fun(*this, &Impl::hide)
	);
}

BasicConfigDialog::Impl::~Impl()
{
	assert(m_spInstance);
	m_spInstance = 0;
}

void
BasicConfigDialog::Impl::showWindow()
{
	if (m_spInstance) {
		m_spInstance->present();
	} else {
		(m_spInstance = new Impl)->show();
	}
}

void
BasicConfigDialog::Impl::on_hide()
{
	delete this;
}

void
BasicConfigDialog::Impl::loadConfig()
{
	Config config(GlobalState::ReadAccessor()->config());
	list<SymbolicInetAddr> listen_addrs = config.getListenAddresses();
	if (listen_addrs.size() > 1) {
		setCurrentListenPort(Glib::ustring());
		showWarning(
			"Your current config has multiple listen addresses.\n"
			"You should be using the Advanced Configuration dialog."
		);
	} else if (listen_addrs.size() == 1) {
		setCurrentListenPort(
			StringUtils::fromNumber(listen_addrs.front().getPort())
		);
	} else {
		setCurrentListenPort(Glib::ustring());
	}
	
	switch (config.getPageCleanupLevel()) {
	case Config::CLEANUP_OFF:
		m_pPageCleanupOff->set_active();
		break;
	case Config::CLEANUP_SAFE:
		m_pPageCleanupSafe->set_active();
		break;
	case Config::CLEANUP_MAXIMUM:
		m_pPageCleanupMax->set_active();
		break;
	}
	
	m_pTrayAnimation->set_active(config.isTrayAnimationEnabled());	
}

void
BasicConfigDialog::Impl::setCurrentListenPort(Glib::ustring const& port)
{
	m_curListenPort = port;
	m_pListenPort->set_text(port);
}

void
BasicConfigDialog::Impl::onApply()
{
	bool const listen_port_changed = (m_pListenPort->get_text() != m_curListenPort);
	uint16_t listen_port = 0;
	if (listen_port_changed) {
		try {
			string port = StringUtils::trimStr(m_pListenPort->get_text().c_str());
			listen_port = boost::lexical_cast<uint16_t>(port);
		} catch (boost::bad_lexical_cast& e) {}
		if (listen_port == 0) {
			showError("Listen port must be an integer in range of [1, 65535]");
			m_pListenPort->grab_focus();
			return;
		}
	}
	
	Config const old_config(GlobalState::ReadAccessor()->config());
	Config new_config(old_config);
	if (listen_port_changed) {
		new_config.setListenAddresses(
			list<SymbolicInetAddr>(1, SymbolicInetAddr("127.0.0.1", listen_port))
		);
	}
	
	if (m_pPageCleanupMax->get_active()) {
		new_config.setPageCleanupLevel(Config::CLEANUP_MAXIMUM);
	} else if (m_pPageCleanupSafe->get_active()) {
		new_config.setPageCleanupLevel(Config::CLEANUP_SAFE);
	} else {
		new_config.setPageCleanupLevel(Config::CLEANUP_OFF);
	}
	
	new_config.setTrayAnimationEnabled(m_pTrayAnimation->get_active());
	
	Application& app = *Application::instance();
	
	if (!app.configFile().applyAndSave(new_config)) {
		app.showLogDialog();
		return;
	}
	
	hide();
}

void
BasicConfigDialog::Impl::showError(Glib::ustring const& msg)
{
	Gtk::MessageDialog dialog(
		*this, msg,
		false, /* use_markup */
		Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK,
		true /* modal */
	);
	dialog.run();
}

void
BasicConfigDialog::Impl::showWarning(Glib::ustring const& msg)
{
	Gtk::MessageDialog dialog(
		*this, msg,
		false, /* use_markup */
		Gtk::MESSAGE_WARNING, Gtk::BUTTONS_OK,
		true /* modal */
	);
	dialog.run();
}

} // namespace GtkGUI
