/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Daemon.h"
#include "NonCopyable.h"
#include "Conf.h"
#include "ConfError.h"
#include "ConfErrorHandler.h"
#include "ConfigFileStructure.h"
#include "UrlPatterns.h"
#include "UrlsFileStructure.h"
#include "ContentFilterList.h"
#include "AvailableContentFilters.h"
#include "AvailableFiltersOrdered.h"
#include "Forwarding.h"
#include "FilterGroupTag.h"
#include "FilterFileStructure.h"
#include "FilterJsLogger.h"
#include "RegexFilterDescriptor.h"
#include "GlobalState.h"
#include "ConnAcceptor.h"
#include "InetAddr.h"
#include "SymbolicInetAddr.h"
#include "DnsResolver.h"
#include "IntrusivePtr.h"
#include "AutoClosingSAP.h"
#include "RefCountableSAP.h"
#include "StringUtils.h"
#include "Color.h"
#include "ArraySize.h"
#include "Alarm.h"
#include "types.h"
#include "cache/ObjectStorage.h"
#ifdef ENABLE_BINRELOC
#include "binreloc/prefix.h"
#endif
#ifdef ENABLE_PROXYWATCHER
#include "ProxyWatcherThread.h"
#endif
#ifdef ENABLE_RECONFIGURE
#include "LocationUpdater.h"
#endif
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/Signal.h>
#include <ace/OS_NS_Thread.h> // for ACE_OS::sigwait()
#include <ace/OS_NS_signal.h> // for ACE_OS::sigprocmask()
#include <ace/FILE_IO.h>
#include <ace/FILE_Addr.h>
#include <ace/FILE_Connector.h>
#include <boost/program_options.hpp>
#include <boost/tokenizer.hpp>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <list>
#include <cstdlib>
#include <exception>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <pwd.h>
#include <grp.h>
#include <assert.h>

namespace po = boost::program_options;
using namespace std;

namespace
{

class ScopeSetEUID
{
public:
	ScopeSetEUID(uid_t euid);
	
	~ScopeSetEUID();
private:
	uid_t m_prevEUID;
};


class ScopeSetEGID
{
public:
	ScopeSetEGID(gid_t egid);
	
	~ScopeSetEGID();
private:
	uid_t m_prevEGID;
};


ScopeSetEUID::ScopeSetEUID(uid_t euid)
{
	m_prevEUID = geteuid();
	seteuid(euid);
}

ScopeSetEUID::~ScopeSetEUID()
{
	seteuid(m_prevEUID);
}

ScopeSetEGID::ScopeSetEGID(gid_t egid)
{
	m_prevEGID = getegid();
	setegid(egid);
}

ScopeSetEGID::~ScopeSetEGID()
{
	setegid(m_prevEGID);
}

} // anonymous namespace


class Daemon::ConfigErrorHandler : public ConfErrorHandler
{
public:
	ConfigErrorHandler(std::string const& fname)
	: m_fileName(fname), m_hasCriticalErrors(false) {}
	
	virtual bool handleError(ConfError const& err);
	
	bool hasCriticalErrors() const { return m_hasCriticalErrors; }
private:
	std::string m_fileName;
	bool m_hasCriticalErrors;
};


class FileOps
{
public:
	static bool readFile(std::string const& file_path,
		std::string& target, bool log_errors = true);
};


class Daemon::ConfigLoader
{
	DECLARE_NON_COPYABLE(ConfigLoader)
public:
	ConfigLoader(std::string const& file_path);
	
	~ConfigLoader();
	
	/**
	 * \brief Load and apply, but don't bind to listen addresses.
	 */
	bool loadAndApply();
	
	bool load(Config& new_config, ConfigFileStructure& new_structure);
private:
	std::string m_filePath;
};


class Daemon::ForwardingConfigLoader
{
	DECLARE_NON_COPYABLE(ForwardingConfigLoader)
public:
	ForwardingConfigLoader(std::string const& file_path);
	
	~ForwardingConfigLoader();
	
	bool loadAndApply(ObsoleteForwardingInfo const& fallback);
private:
	void applyFallback(ObsoleteForwardingInfo const& fallback);
	
	std::string m_filePath;
};


class Daemon::UrlPatternsLoader
{
	DECLARE_NON_COPYABLE(UrlPatternsLoader)
public:
	enum Which { URLS, URLS_LOCAL };
	
	UrlPatternsLoader(Which which, std::string const& file_path);
	
	~UrlPatternsLoader();
	
	bool loadAndApply();
private:
	std::string m_filePath;
	Which m_which;
};


class Daemon::ContentFiltersLoader
{
	DECLARE_NON_COPYABLE(ContentFiltersLoader)
public:
	ContentFiltersLoader(std::string const& filters_dir);
	
	~ContentFiltersLoader();
	
	void loadAndApply();
private:
	void loadFilter(std::string const& fname);
	
	std::vector<std::string> readFilterFileNames();
	
	void loadEnabledFilterList(
		ContentFilterList& filters,
		std::string const& enabled_file_path);
	
	std::string m_filtersDir;
	AvailableContentFilters m_filters;
};


#if defined(ENABLE_RECONFIGURE)

class Daemon::Reconfigurer
{
	DECLARE_NON_COPYABLE(Reconfigurer)
public:
	Reconfigurer(std::string const& confdir, uid_t bind_uid, gid_t bind_gid);
	
	~Reconfigurer();
	
	bool reconfigure();
private:
	bool loadConfig();
	
	SymbolicInetAddr checkAndFixListenAddr(SymbolicInetAddr const& addr);
	
	uint16_t tryBind(InetAddr const& addr);
	
	uint16_t tryBindWithFeedback(InetAddr const& addr);
	
	bool writeListenAddrToConfig(SymbolicInetAddr const& addr);
	
	bool updateNetworkProfiles(int listen_port);
	
	static uint16_t const m_sAlternativePorts[];
	std::string m_confDir;
	uid_t m_bindUID;
	gid_t m_bindGID;
	SymbolicInetAddr const m_defaultListenAddr;
	SymbolicInetAddr const m_badAddr;
	Config m_config;
	ConfigFileStructure m_configFile;
};

#endif // ENABLE_RECONFIGURE


/* ==================== Daemon::ConfigErrorHandler =======================*/

bool
Daemon::ConfigErrorHandler::handleError(ConfError const& err)
{
	cerr << '[' << m_fileName;
	if (err.getLine()) {
		cerr << ':' << err.getLine();
		if (err.getCol()) {
			cerr << ':' << err.getCol();
		}
	}
	cerr << "] ";
	if (err.getType() == ConfError::T_WARNING) {
		cerr << "Warning: ";
	} else {
		m_hasCriticalErrors = true;
	}
	cerr << err.getMessage();
	cerr << endl;
	return true;
}


/* ========================= Daemon::FileOps ===========================*/

bool
FileOps::readFile(std::string const& file_path,
	std::string& target, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(file_path.c_str());
	ACE_FILE_Connector connector;
	
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, O_RDONLY) == -1) {
		if (log_errors) {
			std::cerr << "Could not open " << file_path << std::endl;
		}
		return false;
	}
	
	std::string content;
	
	char buf[4096];
	ssize_t bytes_read = 0;
	while ((bytes_read = file.recv(buf, sizeof(buf))) > 0) {
		content.append(buf, bytes_read);
	}
	if (bytes_read != 0) {
		if (log_errors) {
			std::cerr << "Error reading " << file_path << std::endl;
		}
		return false;
	}
	
	target.swap(content);
	return true;
}

/* ====================== Daemon::ConfigLoader =========================*/

Daemon::ConfigLoader::ConfigLoader(std::string const& file_path)
:	m_filePath(file_path)
{
}

Daemon::ConfigLoader::~ConfigLoader()
{
}

bool
Daemon::ConfigLoader::loadAndApply()
{
	std::string text;
	if (!FileOps::readFile(m_filePath, text)) {
		return false;
	}
	
	Config new_config;
	ConfigFileStructure new_structure;
	ConfigErrorHandler eh("config");
	new_structure.load(text, new_config, eh);
	if (eh.hasCriticalErrors()) {
		return false;
	}
	
	GlobalState::WriteAccessor()->swapConfig(new_config);
	return true;
}

bool
Daemon::ConfigLoader::load(
	Config& new_config, ConfigFileStructure& new_structure)
{
	std::string text;
	if (!FileOps::readFile(m_filePath, text)) {
		return false;
	}
	
	ConfigErrorHandler eh("config");
	new_structure.load(text, new_config, eh);
	return true;
}


/* =================== Daemon::ForwardingConfigLoader ==================*/

Daemon::ForwardingConfigLoader::ForwardingConfigLoader(
	std::string const& file_path)
:	m_filePath(file_path)
{
}

Daemon::ForwardingConfigLoader::~ForwardingConfigLoader()
{
}

bool
Daemon::ForwardingConfigLoader::loadAndApply(
	ObsoleteForwardingInfo const& fallback)
{
	if (ACE_OS::access(m_filePath.c_str(), F_OK) == -1) {
		applyFallback(fallback);
		return true;
	}
	
	std::string text;
	if (!FileOps::readFile(m_filePath, text)) {
		return false;
	}
	
	std::istringstream strm(text);
	ConfigErrorHandler eh("forwarding.xml");
	
	Forwarding::Config config;
	if (!config.fromStream(strm, eh)) {
		return false;
	}
	
	if (config.options().empty()) {
		applyFallback(fallback);
		return true;
	}
	
	Forwarding::Resolver resolver(config);
	GlobalState::WriteAccessor()->swapForwardingResolver(resolver);
	return true;
}

void
Daemon::ForwardingConfigLoader::applyFallback(
	ObsoleteForwardingInfo const& fallback)
{
	Forwarding::Resolver resolver(fallback.toNewFormat());
	GlobalState::WriteAccessor()->swapForwardingResolver(resolver);
}


/* ==================== Daemon::UrlPatternsLoader =======================*/

Daemon::UrlPatternsLoader::UrlPatternsLoader(
	Which which, std::string const& file_path)
:	m_filePath(file_path),
	m_which(which)
{
}

Daemon::UrlPatternsLoader::~UrlPatternsLoader()
{
}

bool
Daemon::UrlPatternsLoader::loadAndApply()
{
	std::string text;
	if (!FileOps::readFile(m_filePath, text)) {
		return false;
	}
	
	UrlPatterns new_patterns;
	UrlsFileStructure new_structure;
	ConfigErrorHandler eh(m_which == URLS ? "urls" : "urls.local");
	new_structure.load(text, new_patterns, eh);
	
	GlobalState::WriteAccessor global_state;
	if (m_which == URLS) {
		global_state->swapStandardUrlPatterns(new_patterns);
	} else {
		global_state->swapLocalUrlPatterns(new_patterns);
	}
	
	return true;
}


/* ====================== Daemon::ContentFilterLoader =========================*/

Daemon::ContentFiltersLoader::ContentFiltersLoader(std::string const& filters_dir)
:	m_filtersDir(filters_dir)
{
}

Daemon::ContentFiltersLoader::~ContentFiltersLoader()
{
}

void
Daemon::ContentFiltersLoader::loadAndApply()
{
	std::vector<std::string> fnames(readFilterFileNames());
	std::vector<std::string>::const_iterator it(fnames.begin());
	std::vector<std::string>::const_iterator const end(fnames.end());
	for (; it != end; ++it) {
		loadFilter(*it);
	}
	
	AvailableFiltersOrdered ordered_filters(m_filters);
	GlobalState::WriteAccessor()->swapContentFilters(ordered_filters);
}

void
Daemon::ContentFiltersLoader::loadFilter(std::string const& fname)
{
	std::string const filter_file_path(m_filtersDir+'/'+fname);
	std::string const enabled_file_path(filter_file_path+".enabled");
	
	std::string text;
	if (!FileOps::readFile(filter_file_path, text)) {
		return;
	}
	
	FilterFileStructure new_structure;
	ContentFilterList new_filters;
	ConfigErrorHandler eh(fname);
	FilterGroupTag group_tag(fname);
	new_structure.load(text, new_filters, eh, group_tag);
	loadEnabledFilterList(new_filters, enabled_file_path);
	
	m_filters.add(new_filters);
}

std::vector<std::string>
Daemon::ContentFiltersLoader::readFilterFileNames()
{
	std::vector<std::string> fnames;
	
	ACE_Dirent dir_h;
	if (dir_h.open(m_filtersDir.c_str()) == -1) {
		std::cerr << "Could not open directory " << m_filtersDir << std::endl;
		return fnames;
	}
	
	for (dirent* ent; (ent = dir_h.read()); ) {
		std::string const fname(ent->d_name);
		if (fname.find('.') != std::string::npos) {
			// skip files with extensions
			continue;
		}
		fnames.push_back(fname);
	}
	
	return fnames;
}

void
Daemon::ContentFiltersLoader::loadEnabledFilterList(
	ContentFilterList& filters, std::string const& enabled_file_path)
{
	std::string text;
	if (!FileOps::readFile(enabled_file_path, text, false)) {
		return;
	}
	
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("\r\n");
	Tokenizer tokens(text, sep);
	std::set<std::string> enabled_set(tokens.begin(), tokens.end());
	bool const all_enabled = (enabled_set.find("*") != enabled_set.end());

	ContentFilterList::iterator it(filters.begin());
	ContentFilterList::iterator const end(filters.end());
	for (; it != end; ++it) {
		RegexFilterDescriptor& filter = *it;
		bool const enabled = all_enabled ||
			(enabled_set.find(filter.name()) != enabled_set.end());
		filter.setEnabled(enabled);
	}
}


/* ====================== Daemon::Reconfigurer =========================*/

#if defined(ENABLE_RECONFIGURE)

uint16_t const Daemon::Reconfigurer::m_sAlternativePorts[] = {
	3128, 8099, 8111, 8188, 0
	// 0 will cause the OS to select an available port 
};

Daemon::Reconfigurer::Reconfigurer(
	std::string const& confdir, uid_t bind_uid, gid_t bind_gid)
:	m_confDir(confdir),
	m_bindUID(bind_uid),
	m_bindGID(bind_gid),
	m_defaultListenAddr("127.0.0.1", 8080),
	m_badAddr(string(), -1)
{
}

Daemon::Reconfigurer::~Reconfigurer()
{
}

bool
Daemon::Reconfigurer::reconfigure()
{
	if (getuid() != 0 || geteuid() != 0) {
		std::cout << "Reconfiguring must be done by root" << std::endl;
		return false;
	}
	
	loadConfig(); // if it fails, it's not a problem
	
	SymbolicInetAddr user_listen_addr(string(), -1);
	
	list<SymbolicInetAddr> addrs(m_config.getListenAddresses());
	if (!addrs.empty()) {
		user_listen_addr = addrs.front();
	}
	if (addrs.size() > 1 ||
	    user_listen_addr.getHost() != m_defaultListenAddr.getHost()) {
		// We've got a power user here who modified settings by hand.
		// It's better to leave them as they are.
		return updateNetworkProfiles(user_listen_addr.getPort());
	}
	
	SymbolicInetAddr listen_addr(checkAndFixListenAddr(user_listen_addr));
	if (listen_addr == m_badAddr) {
		return false;
	}
	
	if (listen_addr != user_listen_addr) {
		if (!writeListenAddrToConfig(listen_addr)) {
			return false;
		}
	}
	
	return updateNetworkProfiles(listen_addr.getPort());
}

bool
Daemon::Reconfigurer::loadConfig()
{
	ConfigLoader loader(m_confDir+"/config");
	return loader.load(m_config, m_configFile);
}

SymbolicInetAddr
Daemon::Reconfigurer::checkAndFixListenAddr(SymbolicInetAddr const& addr)
{
	std::cout << "Checking listen address" << std::endl;
	
	SymbolicInetAddr new_addr(addr);
	if (new_addr.getPort() <= 0 || new_addr.getPort() >= (1<<16)) {
		new_addr = m_defaultListenAddr;
	}
	
	vector<InetAddr> resolved_addrs(DnsResolver::resolve(new_addr));
	if (resolved_addrs.empty() && new_addr != m_defaultListenAddr) {
		// This indicates a problem with host, not with port.
		std::cout << "Could not resolve listen address. "
			"Falling back to the default one." << std::endl;
		new_addr = m_defaultListenAddr;
		DnsResolver::resolve(new_addr).swap(resolved_addrs);
	}
	if (resolved_addrs.empty()) {
		std::cout << "Could not resolve listen address " << new_addr << std::endl;
		return m_badAddr;
	}
	
	InetAddr try_addr(resolved_addrs[0]);
	uint16_t port = tryBindWithFeedback(try_addr);
	if (port) {
		new_addr.setPort(port);
		return new_addr;
	}
	
	for (unsigned i = 0; i < ARRAY_SIZE(m_sAlternativePorts); ++i) {
		try_addr.set_port_number(m_sAlternativePorts[i]);
		port = tryBindWithFeedback(try_addr);
		if (port) {
			new_addr.setPort(port);
			return new_addr;
		}
	}
	
	return m_badAddr;
}

uint16_t
Daemon::Reconfigurer::tryBind(InetAddr const& addr)
{
	ScopeSetEGID gid_setter(m_bindGID);
	ScopeSetEUID uid_setter(m_bindUID);
	
	/*
	Why can't we just do the bind as root?
	Consider the following situation:
	Apache listens on INADDR_ANY:8080.
	We are trying to bind to 127.0.0.1:8080 and are using SO_REUSEADDR.
	In *BSD systems (including OSX), the following rule is applied:
	You are allowed to "steal" a bound address from another process
	if you have the same uids or you are root. That's why we
	temporarely drop our priviledges here.
	*/
	
	AutoClosingSAP<ACE_SOCK_Acceptor> acceptor;
	if (acceptor.open(addr, /* reuse_addr = */1) != -1) {
		InetAddr new_addr;
		if (acceptor.get_local_addr(new_addr) != -1) {
			return new_addr.get_port_number();
		}
	}
	
	return 0;
}

uint16_t
Daemon::Reconfigurer::tryBindWithFeedback(InetAddr const& addr)
{
	uint16_t const req_port = addr.get_port_number();
	if (req_port == 0) {
		std::cout << "Letting OS to select a port for us" << std::endl;
	} else {
		std::cout << "Trying port " << req_port << " ..." << std::endl;
	}
	
	uint16_t const res_port = tryBind(addr);
	if (res_port == 0) {
		std::cout << "Port is unavailable" << std::endl;
	} else {
		std::cout << "Port " << res_port << " is available" << std::endl;
	}
	
	return res_port;
}

bool
Daemon::Reconfigurer::writeListenAddrToConfig(SymbolicInetAddr const& addr)
{
	list<SymbolicInetAddr> addrs;
	addrs.push_back(addr);
	m_config.setListenAddresses(addrs);
	m_configFile.updateWith(m_config);
	
	std::string const fname(m_confDir+"/config");
	std::ofstream strm(fname.c_str());
	if (strm.fail()) {
		std::cout << "Could not open " << fname << " for writing." << std::endl;
		return false;
	}
	m_configFile.toStream(strm);
	if (strm.bad()) {
		std::cout << "Error writing to " << fname << std::endl;
		return false;
	}
	
	return true;
}

bool
Daemon::Reconfigurer::updateNetworkProfiles(int listen_port)
{
	std::cout << "Updating network profiles" << std::endl;
	try {
		LocationUpdater prefs;
		for (int i = 0; i < 2; ++i) {
			if (prefs.tryLock() || SCError() != kSCStatusPrefsBusy) {
				break;
			}
			sleep(1);
		}
		if (!prefs.isLocked()) {
			std::cout << "Unable to lock the preferences" << std::endl;
			return false;
		}
		
		prefs.updateLocations(listen_port);
		if (!prefs.applyChanges()) {
			std::cout << "Could not apply changes" << std::endl;
			return false;
		}
	} catch (std::exception& e) {
		std::cout << "Error updating network locations: " << e.what() << std::endl;
		return false;
	}
	
	return true;
}

#endif // ENABLE_RECONFIGURE


/*=========================== Daemon ==================================*/

bool
Daemon::init(std::string const& confdir, std::string const& cache_dir)
{
	if (!loadConfiguration(confdir)) {
		return false;
	}
	
	if (!cache_dir.empty()) {
		if (!initCache(cache_dir)) {
			return false;
		}
	}
	
	if (!bind()) {
		return false;
	}
	return true;
}

bool
Daemon::run(bool nodaemon, ACE_FILE_IO& pid_file)
{
	if (!nodaemon) {
		if (ACE::daemonize("/", false) == -1) {
			cerr << "Could not enter the daemon mode" << endl;
			return false;
		}
	}
	
	ACE_Sig_Action(SIG_IGN, SIGPIPE);
	
	ACE_Sig_Set termsigs;
	termsigs.sig_add(SIGTERM);
	termsigs.sig_add(SIGINT);
	ACE_Sig_Set oldsigs;
	ACE_OS::sigprocmask(SIG_BLOCK, termsigs, oldsigs);
	
	if (pid_file.get_handle() != ACE_INVALID_HANDLE) {
		ostringstream strm;
		strm << getpid() << '\n';
		string pid = strm.str();
		pid_file.truncate(0);
		pid_file.send_n(pid.c_str(), pid.size());
	}
	
#ifdef ENABLE_PROXYWATCHER
	ProxyWatcherThread proxy_wather;
#endif
	
	m_workerPool.activate();
	
	ACE_OS::sigwait(termsigs);
	
	if (pid_file.get_handle() != ACE_INVALID_HANDLE) {
		// It would be better to delete the pid file,
		// but we've already chrooted and dropped privileges.
		pid_file.truncate(0);
	}
	
	return true;
}

int
Daemon::killRunning(ACE_FILE_IO& pid_file)
{
	if (flock(pid_file.get_handle(), LOCK_EX|LOCK_NB) == 0) {
		std::cerr << "No instance is running" << std::endl;
		return EXIT_SUCCESS;
	}
	
	char buf[20];
	ssize_t size = pid_file.recv(buf, sizeof(buf));
	if (size == -1) {
		std::cerr << "Error reading pid file" << std::endl;
		return EXIT_FAILURE;
	}
	
	stringstream strm;
	strm.write(buf, size);
	pid_t pid = 0;
	strm >> pid;
	if (pid <= 0) {
		std::cerr << "Pid file contains an invalid process id" << std::endl;
		return EXIT_FAILURE;
	}
	if (kill(pid, 0) == -1) {
		switch (errno) {
		case EPERM:
			std::cerr << "No permission to signal the daemon process" << std::endl;
			break;
		case ESRCH:
			std::cerr << "Process referenced by pid not found" << std::endl;
			break;
		default:
			std::cerr << "Signalling the daemon process failed" << std::endl;
		}
		return EXIT_FAILURE;
	}
	
	kill(pid, SIGTERM);
	
	Alarm alrm;
	alrm(1);
	
	if (flock(pid_file.get_handle(), LOCK_EX) == 0) {
		return EXIT_SUCCESS;
	}
	
	kill(pid, SIGKILL);
	alrm(1);
	
	if (flock(pid_file.get_handle(), LOCK_EX) == 0) {
		pid_file.truncate(0);
		return EXIT_SUCCESS;
	}
	
	std::cerr << "Could not lock the pid file even after SIGKILL" << std::endl;
	return EXIT_FAILURE;
}

void
Daemon::initResolver()
{
	/*
	There is a problem with name resolution in a chroot'ed environment.
	On many Unix systems, the resolver will dlopen() some shared libraries
	and some config files before doing the actual name resolution.
	The bad news is that at least some of the files (/etc/hosts and
	/etc/resolv.conf in my case) are read on every name resolution,
	so we just have to have them in our chroot'ed environment.
	The good news is that shared libraries are only loaded on the first
	name resolution, which we do here.
	Note that resolving "localhost" doesn't cause loading of all the
	shared libraries needed by the resolver.
	*/
#if defined(__APPLE__)
	// Not necessary on OSX.
#else
	Alarm alrm;
	alrm(1); // Don't let it block for too long.
	gethostbyname("com.");
#endif
}

std::string
Daemon::normalizePath(std::string const& path)
{
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("/");
	Tokenizer tokenizer(path, sep);
	
	vector<string> elements;
	
	Tokenizer::iterator it = tokenizer.begin();
	Tokenizer::iterator const end = tokenizer.end();
	for (; it != end; ++it) {
		if (*it == "..") {
			if (elements.empty() || elements.back() == "..") {
				elements.push_back(*it);
			} else {
				elements.pop_back();
			}
		} else if (*it != ".") {
			elements.push_back(*it);
		}
	}
	
	string res;
	res.reserve(path.size());
	if (path[0] == '/') {
		res += '/';
	}
	res += StringUtils::join("/", elements.begin(), elements.end());
	
	return res;
}

bool
Daemon::loadConfiguration(std::string const& confdir)
{
	{
		ConfigLoader config_loader(confdir+"/config");
		if (!config_loader.loadAndApply()) {
			return false;
		}
	}
	
	{
		// !!! assumes config is already loaded !!!
		ObsoleteForwardingInfo const fwd_fallback(
			GlobalState::ReadAccessor()->config()
			.getObsoleteForwardingInfo()
		);
		ForwardingConfigLoader fwd_loader(confdir+"/forwarding.xml");
		if (!fwd_loader.loadAndApply(fwd_fallback)) {
			return false;
		}
	}
	
	{
		UrlPatternsLoader urls_loader(
			UrlPatternsLoader::URLS, confdir+"/urls");
		urls_loader.loadAndApply();
	}
	
	{
		UrlPatternsLoader urls_local_loader(
			UrlPatternsLoader::URLS_LOCAL, confdir+"/urls.local");
		urls_local_loader.loadAndApply();
	}
	
	{
		ContentFiltersLoader filters_loader(confdir+"/filters");
		filters_loader.loadAndApply();
	}
	
	return true;
}

bool
Daemon::initCache(std::string const& cache_dir)
{
	if (GlobalState::ReadAccessor()->config().getCacheSize() == 0) {
		return true;
	}
	
	if (!HttpCache::ObjectStorage::instance()->setCacheDir(cache_dir)) {
		if (access(cache_dir.c_str(), R_OK|W_OK|X_OK|F_OK) == 0) {
			fprintf(stderr, "Can't use %s as a cache directory for unknown reason.\n", cache_dir.c_str());
		} else if (errno == ENOENT || errno == ENOTDIR) {
			fprintf(stderr, "%s doesn't exist.\n", cache_dir.c_str());
		} else {
			fprintf(stderr, "%s doesn't have correct permissions.\n", cache_dir.c_str());
		}
		return false;
	}
	return true;
}

bool
Daemon::bind()
{
	typedef RefCountableSAP<ACE_SOCK_Acceptor> Acceptor;
	typedef IntrusivePtr<Acceptor> AcceptorPtr;
	
	bool ok = true;
	vector<AcceptorPtr> acceptors;
	list<SymbolicInetAddr> addrs(GlobalState::ReadAccessor()->config().getListenAddresses());
	for (; !addrs.empty(); addrs.pop_front()) {
		SymbolicInetAddr const& addr = addrs.front();
		vector<InetAddr> resolved_addrs(DnsResolver::resolve(addr));
		if (resolved_addrs.empty()) {
			ok = false;
			cerr << "Could not resolve listen address \"" << addr << '"' << endl;
			continue;
		}
		AcceptorPtr acceptor(new Acceptor);
		if (acceptor->open(resolved_addrs[0], true) == -1) {
			ok = false;
			cerr << "Could not bind to \"" << addr << '"' << endl;
			continue;
		}
		acceptors.push_back(acceptor);
	}
	if (!ok) {
		return false;
	}
	if (acceptors.empty()) {
		cerr << "No addresses to listen on!" << endl;
		return false;
	}
	
	vector<AcceptorPtr>::const_iterator it(acceptors.begin());
	vector<AcceptorPtr>::const_iterator const end(acceptors.end());
	for (; it != end; ++it) {
		m_workerPool.addAcceptor(*it);
	}
	
	return true;
}


/* =============================== main ================================*/

// ACE #define's main() to include its own initialization
int main(int argc, char **argv)
{
	{
		// Create references to Color and FilterJsLogger,
		// to prevent link-time errors.
		Color color(0, 0, 0);
		FilterJsLogger::setHandler(0);
	}
	
	bool nodaemon = false;
	bool kill_running = false;
	bool reconfigure = false;
	string confdir;
	string cache_dir;
	string chroot_dir;
	string effective_confdir;  // 
	string effective_cachedir; // relative to chroot_dir
#if defined(SYSCONFDIR)
	// binreloc may redefine SYSCONFDIR
	confdir = string(SYSCONFDIR) + "/bfilter";
#elif defined(__APPLE__)
	confdir = "/Library/Application Support/BFilter";
#endif
	string user;
	string group;
	string pidfile;
	try {
		po::options_description desc("Allowed options");
		desc.add_options()
			("help,h", "Print help message")
			("version,v", "Print version")
			("confdir,c", po::value<string>(&confdir), "Set custom config directory")
			("cachedir,C", po::value<string>(&cache_dir), "Set cache directory. "
				"May be relative to config directory."
			)
			("chroot,r", po::value<string>(&chroot_dir),
				"Set chroot directory. It must contain the config "
				"directory and may be specified relative to it."
			)
			("user,u", po::value<string>(&user), "Set unprivileged user")
			("group,g", po::value<string>(&group), "Set unprivileged group")
			("nodaemon,n", po::bool_switch(&nodaemon), "Disable background daemon mode")
			("pid,p", po::value<string>(&pidfile),
				"[without -k] Write process id to file\n"
				"[with -k]    Kill the running instance")
			("kill,k", po::bool_switch(&kill_running),
				"Kill the running instance. To be used with -p")
#if defined(ENABLE_RECONFIGURE)
			// This one is used by the installer
			("reconfigure", po::bool_switch(&reconfigure),
				"Find a suitable listen port and update network locations")
#endif
		;
		
		po::variables_map vm;
		po::store(po::parse_command_line(argc, argv, desc), vm);
		po::notify(vm);
		if (vm.count("help")) {
			std::cout << desc << std::endl;
			return EXIT_SUCCESS;
		}
		if (vm.count("version")) {
			std::cout << BFILTER_VERSION << std::endl;
			return EXIT_SUCCESS;
		}
		if (kill_running && pidfile.empty()) {
			std::cerr << "--kill requires --pid" << std::endl;
			return EXIT_FAILURE;
		}
	} catch (std::exception& e) {
		std::cerr << e.what() << std::endl;
		return EXIT_FAILURE;
	}
	
	Alarm::installNoOpHandler();
	
	AutoClosingSAP<ACE_FILE_IO> pid_file;
	if (!pidfile.empty()) {
		ACE_FILE_Addr addr(pidfile.c_str());
		ACE_FILE_Connector connector;
		if (connector.connect(pid_file, addr, 0, ACE_Addr::sap_any, 0, O_RDWR|O_CREAT) == -1) {
			std::cerr << "Could not open " << pidfile << " for writing" << std::endl;
			return EXIT_FAILURE;
		}
		if (kill_running) {
			return Daemon::killRunning(pid_file);
		}
		if (flock(pid_file.get_handle(), LOCK_EX|LOCK_NB) == -1) {
			std::cerr << "Another instance is running" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	if (confdir.c_str()[0] != '/') {
		std::cerr << "Config directory must be absolute" << std::endl;
		return EXIT_FAILURE;
	}
	
	confdir = Daemon::normalizePath(confdir);
	cache_dir = Daemon::normalizePath(cache_dir);
	effective_confdir = confdir;
	effective_cachedir = cache_dir;
	
	if (!chroot_dir.empty()) {
		if (chroot_dir.c_str()[0] == '/') {
			chroot_dir = Daemon::normalizePath(chroot_dir);
		} else {
			chroot_dir = Daemon::normalizePath(confdir+'/'+chroot_dir);
		}
		
		if (StringUtils::startsWith(confdir, chroot_dir)) {
			effective_confdir = confdir.substr(chroot_dir.size());
			// normalizePath() removes the trailing slash among other
			// things, so effective_confdir may now we empty, which is OK.
		} else {
			std::cerr << "Config directory must be within the chroot directory" << std::endl;
			return EXIT_FAILURE;
		}
		
		if (!cache_dir.empty()) {
			if (cache_dir.c_str()[0] == '/') {
				cache_dir = Daemon::normalizePath(cache_dir);
			} else {
				cache_dir = Daemon::normalizePath(confdir+'/'+cache_dir);
			}
			
			if (StringUtils::startsWith(cache_dir, chroot_dir)) {
				effective_cachedir = cache_dir.substr(cache_dir.size());
			} else {
				std::cerr << "Cache directory must be within the chroot directory" << std::endl;
				return EXIT_FAILURE;
			}
		}
	}
	
	Daemon::initResolver();
	
	struct passwd* user_r = 0;
	struct group* group_r = 0;
	
	if (!user.empty()) {
		user_r = getpwnam(user.c_str());
		if (!user_r) {
			std::cerr << "Could not drop privileges: user "
				<< user << " doesn't exist" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	if (!group.empty()) {
		group_r = getgrnam(group.c_str());
		if (!group_r) {
			std::cerr << "Could not drop privileges: group "
				<< group << " doesn't exist" << std::endl;
		}
	}
	
#if defined(ENABLE_RECONFIGURE)
	// This must come before dropping priviledges or chroot()ing.
	if (reconfigure) {
		uid_t const bind_uid = user_r ? user_r->pw_uid : geteuid();
		gid_t const bind_gid = group_r ? group_r->gr_gid : getegid();
		Daemon::Reconfigurer reconfigurer(confdir, bind_uid, bind_gid);
		if (reconfigurer.reconfigure()) {
			return EXIT_SUCCESS;
		} else {
			return EXIT_FAILURE;
		}
	}
#endif
	
	if (!chroot_dir.empty()) {
		if (chdir(chroot_dir.c_str())) {
			std::cerr << "Could not chdir to " << chroot_dir << std::endl;
			return EXIT_FAILURE;
		}
		if (chroot(chroot_dir.c_str())) {
			std::cerr << "Could not chroot to " << chroot_dir << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	if (group_r) {
		if (setgid(group_r->gr_gid)) {
			std::cerr << "Could not drop priviliges: setgid() failed" << std::endl;
			return EXIT_FAILURE;
		}
	}
	if (user_r) {
		if (setuid(user_r->pw_uid)) {
			std::cerr << "Could not drop privileges: setuid() failed" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	
	Daemon daemon;
	if (!daemon.init(effective_confdir, effective_cachedir)) {
		return EXIT_FAILURE;
	}
	if (!daemon.run(nodaemon, pid_file)) {
		return EXIT_FAILURE;
	}
	
	return EXIT_SUCCESS;
}
