/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef IPV4SUBNETPARSER_H_
#define IPV4SUBNETPARSER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "types.h"
#include <string>

class BString;

class IPv4SubnetParser
{
	DECLARE_NON_COPYABLE(IPv4SubnetParser)
public:
	IPv4SubnetParser();
	
	/**
	 * \brief Parses strings like "127.0.0/24" and "192.168"
	 * \return Returns true on success, false on failure.
	 */
	bool parse(char const* begin, char const* end);
	
	/**
	 * \brief Parses strings like "127.0.0/24" and "192.168"
	 * \return Returns true on success, false on failure.
	 */
	bool parse(std::string const& str);
	
	/**
	 * \brief Parses strings like "127.0.0/24" and "192.168"
	 * \return Returns true on success, false on failure.
	 */
	bool parse(BString const& str);
	
	/**
	 * \brief Get the network address in host byte order.
	 *
	 * That is, 127.0.0.0 would become 0x7F000000.
	 * Note that 0x7F000000 is a number, not a collection of bytes.
	 */
	uint32_t getNetAddr() const { return m_netAddr; }
	
	/**
	 * \brief Get the number of network bits.
	 *
	 * Examples:
	 * \li <tt>192.168.0/24 - 24 bits</tt>
	 * \li <tt>192.168/24   - 24 bits</tt>
	 * \li <tt>192.168      - 16 bits</tt>
	 * \li <tt>10           - 8 bits</tt>
	 */
	unsigned getNetBits() const { return m_netBits; }
	
	/**
	 * \returns Returns the error position or null if no error occured.
	 */
	char const* getErrorPos() const { return m_pErrorPos; }
private:
	char const* parseSubnet(char const* begin, char const* end);
	
	bool parseOctet(char const*& p, char const* end, uint8_t& octet);
	
	bool parseNetBits(char const*& p, char const* end, unsigned& netbits);
	
	uint32_t m_netAddr;
	unsigned m_netBits;
	char const* m_pErrorPos;
};

#endif
