/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Conf.h"
#include "Color.h"
#include "Forwarding.h"
#include <algorithm>
#include <assert.h>

using namespace std;

template<typename T>
inline void
Config::auto_ptr_swap(std::auto_ptr<T>& o1, std::auto_ptr<T>& o2)
{
	T* p1 = o1.release();
	T* p2 = o2.release();
	o1.reset(p2);
	o2.reset(p1);
}


Config::Config()
:	m_listenAddrs(),
	m_forwardingInfo(),
	m_isClientCompressionEnabled(false),
	m_isTrayAnimationEnabled(true),
	m_ptrBorderColor(new Color(0x67, 0x67, 0x67)),
	m_pageCleanupLevel(CLEANUP_OFF),
	m_maxScriptNestLevel(6),
	m_maxScriptFetchSize(60),
	m_maxScriptEvalSize(180),
	m_saveTrafficThreshold(15),
	m_reportClientIP(REPORT_IP_OFF),
	m_fixedClientIP(),
	m_allowedTunnelPorts(),
	m_cacheSize(5*1024*1024)
{
	// ports allowed by default
	bool const success = m_allowedTunnelPorts.fromString("443, 563");
	assert(success);
}

Config::Config(Config const& other)
:	m_listenAddrs(other.m_listenAddrs),
	m_forwardingInfo(other.m_forwardingInfo),
	m_isClientCompressionEnabled(other.m_isClientCompressionEnabled),
	m_isTrayAnimationEnabled(other.m_isTrayAnimationEnabled),
	m_ptrBorderColor(other.getBorderColor()),
	m_pageCleanupLevel(other.m_pageCleanupLevel),
	m_maxScriptNestLevel(other.m_maxScriptNestLevel),
	m_maxScriptFetchSize(other.m_maxScriptFetchSize),
	m_maxScriptEvalSize(other.m_maxScriptEvalSize),
	m_saveTrafficThreshold(other.m_saveTrafficThreshold),
	m_reportClientIP(other.m_reportClientIP),
	m_fixedClientIP(other.m_fixedClientIP),
	m_allowedTunnelPorts(other.m_allowedTunnelPorts),
	m_cacheSize(other.m_cacheSize)
{
}

Config::~Config()
{
}

Config&
Config::operator=(Config const& other)
{
	if (&other == this) {
		return *this;
	}
	Config(other).swap(*this);
	return *this;
}

void
Config::swap(Config& other)
{
	if (&other == this) {
		return;
	}
	m_listenAddrs.swap(other.m_listenAddrs);
	m_forwardingInfo.swap(other.m_forwardingInfo);
	std::swap(m_isClientCompressionEnabled, other.m_isClientCompressionEnabled);
	std::swap(m_isTrayAnimationEnabled, other.m_isTrayAnimationEnabled);
	auto_ptr_swap(m_ptrBorderColor, other.m_ptrBorderColor);
	std::swap(m_pageCleanupLevel, other.m_pageCleanupLevel);
	std::swap(m_maxScriptNestLevel, other.m_maxScriptNestLevel);
	std::swap(m_maxScriptFetchSize, other.m_maxScriptFetchSize);
	std::swap(m_maxScriptEvalSize, other.m_maxScriptEvalSize);
	std::swap(m_saveTrafficThreshold, other.m_saveTrafficThreshold);
	std::swap(m_reportClientIP, other.m_reportClientIP);
	m_fixedClientIP.swap(other.m_fixedClientIP);
	m_allowedTunnelPorts.swap(other.m_allowedTunnelPorts);
	std::swap(m_cacheSize, other.m_cacheSize);
}

auto_ptr<Color>
Config::getBorderColor() const
{
	if (!m_ptrBorderColor.get()) {
		return auto_ptr<Color>();
	}
	return auto_ptr<Color>(new Color(*m_ptrBorderColor));
}

void
Config::setBorderColor(auto_ptr<Color> color)
{
	m_ptrBorderColor = color;
}

bool
Config::isTunnelPortAllowed(unsigned port) const
{
	return m_allowedTunnelPorts.contains(port);
}


/* ======================== ObsoleteForwardingInfo ========================*/

ObsoleteForwardingInfo::ObsoleteForwardingInfo()
:	m_nextHopProxy(),
	m_proxyBypassList(),
	m_isNextHopProxyEnabled(false)
{
	// backward compatibility
	m_nextHopProxy.setType(ProxyDescriptor::HTTP);
}

ObsoleteForwardingInfo::ObsoleteForwardingInfo(ObsoleteForwardingInfo const& other)
:	m_nextHopProxy(other.m_nextHopProxy),
	m_proxyBypassList(other.m_proxyBypassList),
	m_isNextHopProxyEnabled(other.m_isNextHopProxyEnabled)
{
}

ObsoleteForwardingInfo::~ObsoleteForwardingInfo()
{
}

ObsoleteForwardingInfo&
ObsoleteForwardingInfo::operator=(ObsoleteForwardingInfo const& rhs)
{
	ObsoleteForwardingInfo(rhs).swap(*this);
	return *this;
}

void
ObsoleteForwardingInfo::swap(ObsoleteForwardingInfo& other)
{
	m_nextHopProxy.swap(other.m_nextHopProxy);
	m_proxyBypassList.swap(other.m_proxyBypassList);
	std::swap(m_isNextHopProxyEnabled, other.m_isNextHopProxyEnabled);
}

Forwarding::Config
ObsoleteForwardingInfo::toNewFormat() const
{
	Forwarding::Config fconfig;
	
#if !defined(ENABLE_PROXYWATCHER)
	// Note: On OSX with ProxyWatcher enabled, we may run into a situation
	// where a network profile named "Direct" or "Proxy" is overriden with
	// profiles we are creating.  Because with ProxyWatcher enabled, the
	// [forwarding] section was just ignored, we can just skip converting
	// it to a new format.
	
	if (!m_isNextHopProxyEnabled) {
		Forwarding::Option option(Forwarding::Utf8String("Direct"));
		option.setSelected(true);
		fconfig.options().push_back(option);
	} else {
		Forwarding::Option option(Forwarding::Utf8String("Proxy"));
		option.setSelected(true);
		option.proxyChain().push_back(m_nextHopProxy);
		
		typedef std::list<std::string> BypassList;
		BypassList::const_iterator it(m_proxyBypassList.begin());
		BypassList::const_iterator const end(m_proxyBypassList.end());
		for (; it != end; ++it) {
			Forwarding::BypassPatternConstPtr pattern;
			std::string const& pat_str = *it;
			char const first_char = pat_str.c_str()[0];
			char const last_char = pat_str.empty() ?
				'\0' : pat_str.c_str()[pat_str.size() - 1];
			
			if (!isASCII(pat_str)) {
				continue;
			}
			
			if (first_char == '.') {
				Forwarding::Utf8String host_mask("*"+pat_str);
				pattern.reset(new Forwarding::HostMaskMatchPattern(host_mask));
			} else if (last_char == '.') {
				Forwarding::Utf8String host_mask(pat_str+"*");
				pattern.reset(new Forwarding::HostMaskMatchPattern(host_mask));
			} else {
				Forwarding::Utf8String host_mask(pat_str);
				pattern.reset(new Forwarding::HostMaskMatchPattern(host_mask));
			}
			
			if (pattern && pattern->getMatcher()) {
				option.bypassList().push_back(pattern);
			}
		}
		
		fconfig.options().push_back(option);
	}
#endif // !defined(ENABLE_PROXYWATCHER)

	return fconfig;
}

bool
ObsoleteForwardingInfo::isASCII(std::string const& str)
{
	char const* p = str.c_str();
	char const* const end = p + str.size();
	for (; p != end; ++p) {
		unsigned char ch = *p;
		if (ch > 0x7F) {
			return false;
		}
	}
	
	return true;
}
