/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef DEBUG

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Debug.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "RequestTag.h"
#include "SplittableBuffer.h"
#include "AbstractCommand.h"
#include "AbstractDebugAgent.h"
#include "InterthreadCommandQueue.h"
#include "AtomicOps.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_Thread.h>
#include <iostream>
#include <sstream>
#include <string>

using namespace std;

AbstractDebugAgent* Debug::m_spAgent = 0;
int32_t Debug::m_requestIdGenerator = 0;


class Debug::ClientConnectionBeginCommand : public AbstractCommand
{
public:
	ClientConnectionBeginCommand();
	
	virtual ~ClientConnectionBeginCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
};


class Debug::ClientConnectionEndCommand : public AbstractCommand
{
public:
	ClientConnectionEndCommand();
	
	virtual ~ClientConnectionEndCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
};


class Debug::RegisterClientRequestCommand : public AbstractCommand
{
public:
	RegisterClientRequestCommand(int id, std::string url);
	
	virtual ~RegisterClientRequestCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
	int m_requestId;
	std::string m_requestUrl;
};


class Debug::HttpMessageBeginCommand : public AbstractCommand
{
public:
	HttpMessageBeginCommand(
		HttpMessageType type, int request_id,
		std::string const& headers);
	
	virtual ~HttpMessageBeginCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
	HttpMessageType m_messageType;
	int m_requestId;
	std::string m_headers;
};


class Debug::HttpMessageEndCommand : public AbstractCommand
{
public:
	HttpMessageEndCommand(HttpMessageType type, bool error);
	
	virtual ~HttpMessageEndCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
	HttpMessageType m_messageType;
	bool m_error;
};


class Debug::LogMessageCommand : public AbstractCommand
{
public:
	LogMessageCommand(std::string const& msg);
	
	virtual ~LogMessageCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
	std::string m_message;
};


class Debug::LogTrafficCommand : public AbstractCommand
{
public:
	LogTrafficCommand(SplittableBuffer const& traf, TrafficDirection dir);
	
	virtual ~LogTrafficCommand();
	
	virtual void operator()();
private:
	ACE_thread_t m_threadId;
	SplittableBuffer m_traffic;
	TrafficDirection m_trafficDirection;
};


/*============================= Debug =============================*/

void
Debug::clientConnectionBegin()
{
	if (!m_spAgent) {
		return;
	}
	
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new ClientConnectionBeginCommand()
		)
	);
}

void
Debug::clientConnectionEnd()
{
	if (!m_spAgent) {
		return;
	}
	
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new ClientConnectionEndCommand()
		)
	);
}

void
Debug::registerClientRequest(
	HttpRequestMetadata const& metadata, RequestTag const& req_tag)
{
	if (!m_spAgent) {
		return;
	}
	
	int const id = AtomicOps::add(&m_requestIdGenerator, 1);
	req_tag->setGroupId(id);
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new RegisterClientRequestCommand(
				id, metadata.requestLine().getURI().toString()
			)
		)
	);
}

void
Debug::httpMessageBegin(
	HttpMessageType type, HttpRequestMetadata const& metadata,
	RequestTag const& req_tag)
{
	if (!m_spAgent) {
		return;
	}
	
	ostringstream strm;
	metadata.requestLine().toStream(strm);
	metadata.headers().toStream(strm);
	
	httpMessageBegin(type, req_tag->getGroupId(), strm.str());
}

void
Debug::httpMessageBegin(
	HttpMessageType type, HttpResponseMetadata const& metadata,
	RequestTag const& req_tag)
{
	if (!m_spAgent) {
		return;
	}
	
	ostringstream strm;
	metadata.statusLine().toStream(strm);
	metadata.headers().toStream(strm);
	
	httpMessageBegin(type, req_tag->getGroupId(), strm.str());
}

void
Debug::httpMessageEnd(HttpMessageType type, bool error)
{
	if (!m_spAgent) {
		return;
	}
	
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new HttpMessageEndCommand(type, error)
		)
	);
}

void
Debug::logMessage(std::string const& msg)
{
	if (!m_spAgent) {
		std::cerr << msg << std::endl;
		return;
	}
	
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new LogMessageCommand(msg)
		)
	);
}

void
Debug::logTraffic(SplittableBuffer const& traf, TrafficDirection dir)
{
	if (!m_spAgent) {
		return;
	}
	
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new LogTrafficCommand(traf, dir)
		)
	);
}

void
Debug::httpMessageBegin(
	HttpMessageType type, int request_id, std::string const& headers)
{
	if (!m_spAgent) {
		return;
	}
	
	m_spAgent->getCommandQueue().push(
		InterthreadCommandQueue::CommandPtr(
			new HttpMessageBeginCommand(type, request_id, headers)
		)
	);
}


/*================= Debug::ClientConnectionBeginCommand =================*/

Debug::ClientConnectionBeginCommand::ClientConnectionBeginCommand()
:	m_threadId(ACE_OS::thr_self())
{
}

Debug::ClientConnectionBeginCommand::~ClientConnectionBeginCommand()
{
}

void
Debug::ClientConnectionBeginCommand::operator()()
{
	Debug::getAgent()->clientConnectionBegin(m_threadId);
}


/*================== Debug::ClientConnectionEndCommand ==================*/

Debug::ClientConnectionEndCommand::ClientConnectionEndCommand()
:	m_threadId(ACE_OS::thr_self())
{
}

Debug::ClientConnectionEndCommand::~ClientConnectionEndCommand()
{
}

void
Debug::ClientConnectionEndCommand::operator()()
{
	Debug::getAgent()->clientConnectionEnd(m_threadId);
}


/*================= Debug::RegisterClientRequestCommand =================*/

Debug::RegisterClientRequestCommand::RegisterClientRequestCommand(
	int id, std::string url)
:	m_threadId(ACE_OS::thr_self()),
	m_requestId(id),
	m_requestUrl(url)
{
}

Debug::RegisterClientRequestCommand::~RegisterClientRequestCommand()
{
}

void
Debug::RegisterClientRequestCommand::operator()()
{
	Debug::getAgent()->registerClientRequest(
		m_threadId, m_requestId, m_requestUrl
	);
}


/*==================== Debug::HttpMessageBeginCommand ====================*/

Debug::HttpMessageBeginCommand::HttpMessageBeginCommand(
	HttpMessageType type, int request_id,
	std::string const& headers)
:	m_threadId(ACE_OS::thr_self()),
	m_messageType(type),
	m_requestId(request_id),
	m_headers(headers)
{
}

Debug::HttpMessageBeginCommand::~HttpMessageBeginCommand()
{
}

void
Debug::HttpMessageBeginCommand::operator()()
{
	Debug::getAgent()->httpMessageBegin(
		m_threadId, m_requestId, m_messageType,  m_headers
	);
}


/*===================== Debug::HttpMessageEndCommand =====================*/

Debug::HttpMessageEndCommand::HttpMessageEndCommand(
	HttpMessageType type, bool error)
:	m_threadId(ACE_OS::thr_self()),
	m_messageType(type),
	m_error(error)
{
}

Debug::HttpMessageEndCommand::~HttpMessageEndCommand()
{
}

void
Debug::HttpMessageEndCommand::operator()()
{
	Debug::getAgent()->httpMessageEnd(m_threadId, m_messageType, m_error);
}


/*======================== Debug::LogMessageCommand =======================*/

Debug::LogMessageCommand::LogMessageCommand(std::string const& msg)
:	m_threadId(ACE_OS::thr_self()),
	m_message(msg)
{
}

Debug::LogMessageCommand::~LogMessageCommand()
{
}

void
Debug::LogMessageCommand::operator()()
{
	Debug::getAgent()->logMessage(m_threadId, m_message);
}


/*======================== Debug::LogTrafficCommand =======================*/

Debug::LogTrafficCommand::LogTrafficCommand(
	SplittableBuffer const& traf, TrafficDirection dir)
:	m_threadId(ACE_OS::thr_self()),
	m_traffic(traf),
	m_trafficDirection(dir)
{
}

Debug::LogTrafficCommand::~LogTrafficCommand()
{
}

void
Debug::LogTrafficCommand::operator()()
{
	Debug::getAgent()->logTraffic(m_threadId, m_traffic, m_trafficDirection);
}

#endif // DEBUG
