/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HASHVALUE_H_
#define HASHVALUE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HexEncoder.h"
#include <string>
#include <iosfwd>
#include <string.h>
#include <stddef.h>

template<int LEN>
class HashValue
{
public:
	// Member-wise copying is OK.
	
	enum { HASH_SIZE = LEN };
	
	HashValue();
	
	unsigned char const* get() const { return m_bytes; }
	
	void set(unsigned char const* bytes);
	
	int compare(HashValue const& rhs) const;
	
	bool operator==(HashValue const& rhs) const {
		return compare(rhs) == 0;
	}
	
	bool operator!=(HashValue const& rhs) const {
		return compare(rhs) != 0;
	}
	
	bool operator<(HashValue const& rhs) const {
		return compare(rhs) < 0;
	}
	
	bool operator>(HashValue const& rhs) const {
		return compare(rhs) > 0;
	}
	
	bool operator<=(HashValue const& rhs) const {
		return compare(rhs) <= 0;
	}
	
	bool operator>=(HashValue const& rhs) const {
		return compare(rhs) >= 0;
	}
	
	/**
	 * \brief Returns a human-readable hexadecimal representation.
	 *
	 * Alpha characters are in lower case.
	 */
	std::string hexEncode() const;
	
	/**
	 * \brief Writes the hexadecimal representation to a stream.
	 *
	 * Alpha characters are in lower case.
	 */
	void hexEncode(std::ostream& out) const;
private:
	unsigned char m_bytes[LEN];
};


template<int LEN>
HashValue<LEN>::HashValue()
{
	memset(m_bytes, 0, LEN);
}

template<int LEN>
inline void
HashValue<LEN>::set(unsigned char const* bytes)
{
	memcpy(m_bytes, bytes, LEN);
}

template<int LEN>
inline int
HashValue<LEN>::compare(HashValue const& rhs) const
{
	return memcmp(m_bytes, rhs.m_bytes, LEN);
}

template<int LEN>
inline std::string
HashValue<LEN>::hexEncode() const
{
	return HexEncoder::hexEncode(m_bytes, LEN);
}

template<int LEN>
inline void
HashValue<LEN>::hexEncode(std::ostream& out) const
{
	HexEncoder::hexEncode(m_bytes, LEN, out);
}

#endif
