#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import os.path as osp
import re
import unittest

from common import dict_conf, tmpdir, aster_version
from data   import study_export

from asrun.installation import aster_root
from asrun.run          import AsRunFactory
from asrun.profil       import ASTER_PROFIL, MODES
from asrun.maintenance  import get_version
from asrun.repart       import (ResourceManager, PureLocalResource,
                               ALLOCATED, NORESOURCE, OVERLIMIT)
from asrun.common.rcfile import parse_config
from asrun.common.sysutils import local_user, local_full_host


class TestObjects(unittest.TestCase):

    def test01_run(self):
        run = AsRunFactory()
        assert type(run.get_as_run_cmd()) is list and len(run.get_as_run_cmd()) > 0
        p1 = run.get_pid()
        p2 = run.get_pid()
        assert p1 != p2
        args = run.get_remote_args()
        assert args[0].find("remote_shell") > -1 and args[1].find("remote_copy") > -1
        # test GetGrav
        assert run.GetGrav('NO_TEST_RESU') == -1
        assert run.GetGrav('OK') == 0
        assert run.GetGrav('<A>_ALARM') == 1
        assert run.GetGrav('NOOK_TEST_RESU') == 2
        assert run.GetGrav('<S>_CPU_LIMIT') == 4
        assert run.GetGrav('NO_RESU_FILE') == 6
        assert run.GetGrav('<F>_ERROR') == 10
        # test get_version_path
        assert run.get_version_path(aster_version) == osp.join(aster_root, aster_version)
        assert run.get_version_path(aster_version, "/aster") == osp.join("/aster", aster_version)
        assert run.GetHostName() == local_full_host


    def test02a_profil(self):
        run = AsRunFactory()
        export = osp.join(tmpdir, "object_from_file.export")
        open(export, "w").write(study_export % dict_conf)
        pr_cnt = ASTER_PROFIL()
        pr_cnt.parse(study_export % dict_conf)
        pr_cnt.set_filename(osp.join(tmpdir, "object_from_content.export"))
        pr_file = ASTER_PROFIL(export)
        # the same + run object
        pr_run = ASTER_PROFIL(export, run)
        for prof in (pr_file, pr_cnt, pr_run):
            assert type(prof['nomjob']) in (list, tuple), prof.get_filename()
            assert prof['nomjob'][0] == "study", prof.get_filename()
            # undefined field
            assert type(prof['display']) in (list, tuple), prof.get_filename()
            assert prof['display'][0].strip() == "", prof.get_filename()
            vref = aster_version
            if prof.run is not None:
                vref = osp.join(aster_root, vref)
            assert prof.get_version_path() == vref, (prof.get_filename(), prof.get_version_path())
        pr_cnt.WriteExportTo(pr_cnt.get_filename())


    def test02b_profil_remote(self):
        run = AsRunFactory()
        export = osp.join(tmpdir, "object_from_file.export")
        open(export, "w").write(study_export % dict_conf)
        pr_run = ASTER_PROFIL(export, run)
        comm0 = pr_run.data[0]['path']
        pr_run.from_remote_server()
        # check difference
        comm1 = pr_run.data[0]['path']
        diff = comm1.replace(comm0, "")
        assert diff == "%s@%s:" % (local_user, local_full_host), (comm0, comm1, ">>>", diff)


    def test02c_mode(self):
        run = AsRunFactory()
        export = osp.join(tmpdir, "object_from_file.export")
        open(export, "w").write(study_export % dict_conf)
        pr_run = ASTER_PROFIL(export, run)
        pr_run.set_running_mode(MODES.BATCH)
        assert pr_run["mode"][0] == "batch", pr_run["mode"]
        # string assignement is used by the OM module
        pr_run.set_running_mode("interactif")
        assert pr_run["mode"][0] == "interactif", pr_run["mode"]


    def test02d_profil_win(self):
        run = AsRunFactory()
        prwin = ASTER_PROFIL(osp.join("datafiles", "with_win_CR.export"))
        assert prwin.args.get("memjeveux", "").isdigit()
        assert prwin['actions'][0] == "make_etude"
        assert len(prwin.Get("D", "comm")) > 0
        #path = prwin.Get("D", "comm")[0]["path"]
        #spl = split_path(path)
        #assert len(spl) == 9 and spl[0] == "C:"


    def test03_config(self):
        pass


    def test04_build(self):
        pass


    def test05_resource(self):
        run = AsRunFactory()
        rc = PureLocalResource(run)
        result = rc.CheckHosts(run, silent=True)
        assert result == (1, 1), "local host should be alone and ok : %s" % repr(result)
        hosts = rc.get_all_connected_hosts()
        info = rc.GetConfig(hosts[0])
        cputot = info["cpu"]
        assert rc.get_sum("cpu") == cputot, rc.get_sum("cpu")
        cpu_i = max(cputot / 2, 1)
        host, status = rc.Request(nomjob="job_1", cpu=cpu_i, run=run)
        assert status is ALLOCATED, status
        if cputot >= 2:
            host, status = rc.Request(nomjob="job_2", cpu=cpu_i, run=run)
            assert status is ALLOCATED, status
        host, status = rc.Request(nomjob="job_3", cpu=cpu_i, run=run)
        assert status is NORESOURCE, status
        host, status = rc.Request(nomjob="job_4", cpu=cputot*2, run=run)
        assert status is OVERLIMIT, status
        rc.Free("job_1")
        rc.Free("job_2")
        txt = rc.Load()
        assert type(txt) in (str, unicode)
        txt = rc.repr_history()
        assert type(txt) in (str, unicode)
        # create a new ResourceManager object from the previous one
        content = rc.hostinfo_repr()
        hostfile = osp.join(tmpdir, "hostfile_from_purelocal")
        open(hostfile, "w").write(content)
        host_infos = parse_config(hostfile)
        newrc = ResourceManager(host_infos)
        hosts = newrc.get_all_connected_hosts()
        assert len(hosts) == 1


    def test06_maint(self):
        run = AsRunFactory()
        vers = run['default_vers']
        assert vers is not None, "default_vers is None! Check aster configuration file"
        REPREF = run.get_version_path(vers)
        res = get_version(run, REPREF)
        assert type(res) in (list, tuple) and len(res) == 5, res


if __name__ == "__main__":
    unittest.main()



