------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--              A S I S _ U L . G L O B A L _ S T A T E . C G               --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                       Copyright (C) 2007, AdaCore                        --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not,  write  to the  Free Software Foundation,  51 Franklin  --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the interface routines for building and analyzing
--  the call graph

with Asis;

package ASIS_UL.Global_State.CG is

   procedure Add_CG_Info
     (Element : Asis.Element;
      At_SLOC : String_Loc := Nil_String_Loc);
   --  This procedure is supposed to be called during the element traversal
   --  as a part of pre-operation for each element being visited. It adds to
   --  the global structure the information corresponding to the given Element.
   --  It also updates the auxiliary data structures used for the call graph
   --  construction (scope stack)
   --
   --  For each piece of the stored call graph information this procedure
   --  should store the full "trace" of the information needed to build a full
   --  cal graph. That is, if we process a subprogram call, we store the
   --  declaration of the called subprogram, and we also check for the source
   --  containing the body, and if it is not in the set of sources to be
   --  processed by gnatcheck, we try to locate this source and add to the
   --  source table (???)
   --
   --  If At_SLOC is not equal Nil_String_Loc, it is used as the SLOC of a
   --  construct that is a reason to add some information in the call graph.
   --  (For a function call that is a consequence of computing the default
   --  expression for a record componet as a part of elaboration of an object
   --  declaration, the SLOC of the call is the SLOC of the object declaration,
   --  but not the SLOC of the call in the default component expression in the
   --  corresponding record definition). Otehrwise the SLOC of the
   --  corresponding information item in the call graph is the SLOC of the
   --  argument Element.

   procedure Complete_CG_Info (El : Asis.Element);
   --  This procedure is supposed to be called during the element traversal
   --  as a part of post-operation for each element being visited. It completes
   --  the information stored in the global structure for the given Element.
   --  It also updates the auxiliary data structures used for the call graph
   --  construction (scope stack)

   procedure Add_CG_Info_Pre_Op
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out String_Loc);

   procedure Complete_CG_Info_Post_Op
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out String_Loc);
   --  Wrapper procedures for Add_CG_Info and Add_CG_Info_Pre_Op, add State and
   --  control parameters, this makes it possible to use these procedures as
   --  actuals in Traverse_Element instantiation. The State parameter is used
   --  to pass the information of the SLOC of the information items stored
   --  in the call graph. If it is Nil_String_Loc, the SLOC of an information
   --  item is computed from the ELmenet from that this item is derived.
   --  Otherwise the State is used as SLOC of all the information items.
   --  (Consider getting the function calls from the default component
   --  expressions that are computed as a part of an object declaration
   --  elaboration)

   procedure Collect_CG_Info_From_Construct
     (Element : Asis.Element;
      At_SLOC : String_Loc := Nil_String_Loc);
   --  Collects the call graph information from an argument element by
   --  tarversing the Element structure and calling Add_CG_Info as
   --  pre-operation and Complete_CG_Info as post-operation.
   --  If At_SLOC is not nil, it is used as a SLOC for the information items
   --  stored in the call graph as the result of calling this procredure

   procedure Transitive_Closure;
   --  This procedure should be called when all the "local" call information is
   --  already stored in the global data structures. It performs the transitive
   --  closure of the call graph. After the call to this procedure, for
   --  each entity (call graph node) we have a full set of all calls.
   --
   --  Call graph transitive closure is a time-consuming operation, At the
   --  moment we use the workpile algorithm to compute a set of all the nodes
   --  called by the given node.

   function Is_Recursive_Node (N : GS_Node_Id) return Boolean;
   --  Check if N calls itself (directly or indirectly). It is an error to call
   --  this function if No (N), or if the transitive clousure of the global
   --  structure has not been performed yet.

   ---------------------------------------------------------------------
   --  Asses and update routines for callable node general properties --
   ---------------------------------------------------------------------

   --  All the routines declared in this section should be allied to callable
   --  entities only.

   function Body_Analyzed (N : GS_Node_Id) return Boolean;
   procedure Set_Body_Analyzed (N : GS_Node_Id; Val : Boolean := True);
   --  Reports and sets the flag indicating if the body of the callable entity
   --  has been analyzed. Raise Constraint_Error is No (N).

   function GS_Is_Renaming (N : GS_Node_Id) return Boolean;
   --  Tells if the node is a renaming of another node. Raises Constraint_Error
   --  is No (N).

   function GS_Is_Task_Type (N : GS_Node_Id) return Boolean;
   --  Tells if N represents a task node correspondong to a task type. Always
   --  returns False if N is a callable, but non-task node. Raises
   --  Constraint_Error is No (N).

   procedure Set_Is_Task_Type (N : GS_Node_Id; Val : Boolean := True);
   --  Sets the flag indicating that N represents a task type entity. Assumes
   --  that N is of A_Task kind. Raises Constraint_Error is No (N).

end ASIS_UL.Global_State.CG;
