------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                     G N A T C H E C K . O U T P U T                      --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                     Copyright (C) 2004-2007, AdaCore                     --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Handling; use Ada.Characters.Handling;
with Asis.Implementation;     use Asis.Implementation;
with Ada.Text_IO;             use Ada.Text_IO;

with GNAT.OS_Lib;             use GNAT.OS_Lib;

with ASIS_UL.Common;          use ASIS_UL.Common;
with ASIS_UL.Output;          use ASIS_UL.Output;

with Gnatvsn;

package body Gnatcheck.Output is

   Report_File : File_Type;
   --  The file to place the gnatcheck report into

   Report_File_Name : constant String :=
     ".." & Directory_Separator & "gnatcheck.out";
   --  The name of the report file. For the development period only

   Indent_String : constant String := "   ";

   ------------------------
   -- Print_Version_Info --
   ------------------------

   procedure Print_Version_Info is
   begin
      Info_No_EOL ("GNATCHECK (built with ");
      Info_No_EOL (To_String (Asis.Implementation.ASIS_Implementor_Version));
      Info        (")");

      Info_No_EOL ("Copyright 2004-");
      Info_No_EOL (Gnatvsn.Current_Year);
      Info        (", AdaCore.");
   end Print_Version_Info;

   ----------------
   -- Brief_Help --
   ----------------

   procedure Brief_Help is
   begin
      Info ("gnatcheck: the GNAT rule checking tool");
      Info ("usage: gnatcheck [options] {filename} {-files=filename} " &
            "[-cargs gcc_switches] [-rules rule_switches]");
      Info ("options:");
      Info (" -a             - process RTL units");
      Info (" -d             - debug mode");
      Info (" -dd            - progress indicator mode (for use in GPS)");
      Info (" -h             - print out the list of the currently " &
            "implemented rules");
      Info (" -l             - full source location chains in report file");
      Info (" -q             - quiet mode (do not report detections in "&
            "Stdout)");
      Info (" -v             - verbose mode");
      Info (" -s             - short form of the report file");
      Info (" -sn            - include only section n (n in 1 .. 3) in the " &
            "report file");
      Info ("");

      Info ("filename        - the name of the Ada source file " &
            "to be analyzed.");
      Info ("                  Wildcards are allowed");
      Info ("-files=filemane - the name of the text file containing a list");
      Info ("                  of Ada source files to analyze");
      Info ("");
      Info ("gcc_switches  - a list of switches that are " &
            "valid switches for gcc");
      Info ("");
      Info ("rule_switches          - a list of the following switches");
      Info ("   -from=filename      - read rule options from filename");
      Info ("   +ALL                - turn all the rules ON");
      Info ("   -ALL                - turn all the rules OFF");
      Info ("   +R<rule_id>[:param] - turn ON a given rule [with " &
            "given parameter]");
      Info ("   -R<rule_id>         - turn OFF a given rule");
      Info ("   -R<rule_id>:param   - turn OFF some of the checkS for a " &
            "given  rule,");
      Info ("                         depending on the specified parameter");
      Info ("where <rule_id> - ID of one of the currently implemented");
      Info ("                  rules, use '-h' for the full list");
      Info ("      param     - string representing parameter(s) of a given " &
            "rule, more then ");
      Info ("                  one parameter can be set separated by ','");
   end Brief_Help;

   -----------------------
   -- Close_Report_File --
   -----------------------

   procedure Close_Report_File is
   begin
      Close (Report_File);
   end Close_Report_File;

   ------------
   -- Report --
   ------------

   procedure Report
     (Message      : String;
      Indent_Level : Natural := 0)
   is
   begin
      for J in 1 .. Indent_Level loop
         Put (Report_File, Indent_String);
      end loop;

      Put_Line (Report_File, Message);

   end Report;

   ----------------
   -- Report_EOL --
   ----------------

   procedure Report_EOL is
   begin
      New_Line (Report_File);
   end Report_EOL;

   -------------------
   -- Report_No_EOL --
   -------------------

   procedure Report_No_EOL
     (Message      : String;
      Indent_Level : Natural := 0)
   is
   begin
      for J in 1 .. Indent_Level loop
         Put (Report_File, Indent_String);
      end loop;

      Put (Report_File, Message);

   end Report_No_EOL;

   ---------------------
   -- Set_Report_File --
   ---------------------

   procedure Set_Report_File is
   begin

      --  We are still in the temporary dir:

      if Is_Regular_File (Report_File_Name) then
         Open (Report_File, Out_File, Report_File_Name);
      else
         Create (Report_File, Out_File, Report_File_Name);
      end if;

   exception
      when Status_Error =>
         Error ("can not open the report file, the file may be in use");
         raise Fatal_Error;
      when others =>
         Error ("can not open the report file, check the file name");
         raise Fatal_Error;
   end Set_Report_File;

end Gnatcheck.Output;
