#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Tests the PackageKit compatibilty layer."""

import logging
import os.path
import shutil
import subprocess
import tempfile
import time
import unittest

from gi.repository import PackageKitGlib as pk

import aptdaemon.test

REPO_PATH = os.path.join(aptdaemon.test.get_tests_dir(), "repo")
DEBUG = True


class PackageKitTest(aptdaemon.test.AptDaemonTestCase):

    """Test the PackageKit compatibilty layer."""

    def setUp(self):
        """Setup a chroot, run the aptdaemon and a fake PolicyKit daemon."""
        # Setup chroot
        self.chroot = aptdaemon.test.Chroot()
        self.chroot.setup()
        self.addCleanup(self.chroot.remove)
        self.chroot.add_test_repository()
        self.start_session_aptd(self.chroot.path)
        # Start the fake PolikcyKit daemon
        self.start_fake_polkitd()
        time.sleep(0.5)

    def test_install(self):
        """Test installing a package."""
        self.chroot.add_test_repository()
        pkg_name = "silly-depend-base"

        client = pk.Client()

        # Resolve the id of the package
        res = client.resolve(pk.FilterEnum.NONE, [pkg_name], None,
                             lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        ids = []
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), pkg_name)
            ids.append(pkg.get_id())
            break
        else:
            self.fail("Failed to resolve %s" % pkg_name)

        # Simulate
        res = client.simulate_install_packages(ids, None,
                                               lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), "silly-base")
            self.assertEqual(pkg.get_info(),
                             pk.info_enum_from_string("installing"))
            break
        else:
            self.fail("Failed to get dependencies of %s" % pkg_name)

        # Install
        res = client.install_packages(True, ids, None,
                                      lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)

    def test_download(self):
        """Test downloading packages."""
        self.chroot.add_test_repository()
        pkg_filename = "silly-base_0.1-0update1_all.deb"
        pkg_id = "silly-base;0.1-0update1;all;"
        temp_dir = tempfile.mkdtemp(prefix="aptd-download-test-")

        client = pk.Client()
        res = client.download_packages([pkg_id], temp_dir,
                                       None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        if not os.path.exists(os.path.join(temp_dir, pkg_filename)):
            self.fail("Failed to download the package")

        shutil.rmtree(temp_dir)

    def test_get_updates(self):
        """Test getting updates."""
        self.chroot.add_test_repository()
        pkg = "silly-base_0.1-0_all.deb"
        self.chroot.install_debfile(os.path.join(REPO_PATH, pkg), True)

        client = pk.Client()

        res = client.get_updates(pk.FilterEnum.NONE, None,
                                 lambda p, t, d: True, None)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), "silly-base")
            self.assertEqual(pkg.get_version(), "0.1-0update1")
            self.assertEqual(pkg.get_info(),
                             pk.info_enum_from_string("normal"))
            break
        else:
            self.fail("Failed to detect upgrade")
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)

    def test_dependencies(self):
        """Test getting dependencies and dependants."""
        pkg_id_depend = "silly-depend-base;0.1-0;all;"
        pkg_id = "silly-base;0.1-0update1;all;"

        client = pk.Client()

        # Get depends
        res = client.get_depends(pk.FilterEnum.NONE, [pkg_id_depend], True,
                                 None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_id(), pkg_id)
            break
        else:
            self.fail("Failed to get dependencies of %s" % pkg_id_depend)

        # Get requires
        res = client.get_requires(pk.FilterEnum.NONE, [pkg_id], True,
                                  None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_id(), pkg_id_depend)
            break
        else:
            self.fail("Failed to get dependants of %s" % pkg_id)


def start_dbus_daemon():
    """Start a dbus system daemon.

    The PackageKit gir client doesn't support a changing dbus-server address
    after initialisation, so we need to keep a persistent dbus daemon for
    the whole test suite.
    """
    config_path = os.path.join(aptdaemon.test.get_tests_dir(), "dbus.conf")
    proc = subprocess.Popen(["dbus-daemon", "--nofork", "--print-address",
                             "--config-file", config_path],
                            stdout=subprocess.PIPE)
    address = proc.stdout.readline().strip()
    os.environ["DBUS_SYSTEM_BUS_ADDRESS"] = address
    return proc

def setup():
    proc = start_dbus_daemon()
    global DBUS_PID
    DBUS_PID = proc.pid

def tearDown():
    os.kill(DBUS_PID, 15)

if __name__ == "__main__":
    if DEBUG:
        logging.basicConfig(level=logging.DEBUG)
    dbus_proc = start_dbus_daemon()
    unittest.main()
    dbus_proc.kill()
    dbus_proc.wait()

# vim: ts=4 et sts=4
