#! /usr/bin/perl
# This is a library file for Apt-cacher to allow code
# common to Apt-cacher itself plus its supporting scripts
# (apt-cacher-report.pl and apt-cacher-cleanup.pl) to be
# maintained in one location.

# This function reads the given config file into the
# given hash ref. The key and value are separated by
# a '=' and will have all the leading and trailing
# spaces removed.

use strict;
use warnings;
use POSIX;
use Fcntl qw/:flock/;
use FreezeThaw qw(freeze thaw);
use HTTP::Response;
use URI;
use IO::Uncompress::AnyUncompress qw($AnyUncompressError);
use Module::Load::Conditional;
use File::Spec;
use Carp;
our $cfg;

sub read_config {

    (my $config_file) = @_;

    # set the default config variables
    my %config = (
		  # General
		  admin_email => 'root@localhost',
		  allowed_hosts => '',
		  allowed_hosts_6 => '',
		  allowed_ssl_locations => '',
		  allowed_ssl_ports => '443',
		  cache_dir => '/var/cache/apt-cacher',
		  clean_cache => 1,
		  curl_idle_timeout => 120,
		  daemon_port => 3142,
		  debug => 0,
		  denied_hosts => '',
		  denied_hosts_6 => '',
		  distinct_namespaces => 0,
		  expire_hours => 0,
		  fetch_timeout => 300, # five minutes from now
		  generate_reports => 1,
		  group => (sub { my $g = $); $g =~ s/\s.*$//; $g; })->(),
		  http_proxy => '',
		  http_proxy_auth => '',
		  limit => 0,
		  limit_global => 0,
		  log_dir => '/var/log/apt-cacher',
		  request_empty_lines => 5,
		  request_timeout => 30,
		  return_buffer_size => 1048576, # 1Mb
		  sha_algorithm => 1,
		  use_proxy => 0,
		  use_proxy_auth => 0,
		  user => $>,

		  # Regexps
		  checksum_files_regexp => '^(?:' . join('|',
							 qw(Packages(?:\.gz|\.bz2)?
							    Sources(?:\.gz|\.bz2)?
							    (?:In)?Release
							    Index
							  )
							) . ')$',
		  index_files_regexp => '^(?:' . join('|',
						      qw(Index
							 Packages(?:\.gz|\.bz2)?
							 Release(?:\.gpg)?
							 InRelease
							 Sources(?:\.gz|\.bz2)?
							 Contents-.+\.gz
							 pkglist.*\.bz2
							 release
							 release\..*
							 srclist.*\.bz2
							 Translation-.+(?:\.gz|\.bz2)
						       )
						     ) . ')$',
		  installer_files_regexp => '^(?:' . join('|',
							  qw(vmlinuz
							     linux
							     initrd\.gz
							     changelog
							     NEWS.Debian
							     2\d{3}-\d{2}-\d{2}-\d{4}\.\d{2}\.gz
							   )
							 ) . ')$',
		  package_files_regexp => '(?:' . join('|',
						       qw(\.deb
							  \.rpm
							  \.dsc
							  \.tar(?:\.gz|\.bz2)
							  \.diff\.gz
							  \.udeb
							  index\.db-.+\.gz
							  \.jigdo
							  \.template
							)
						      ) .')$',
		  soap_url_regexp => '^(?:http://)?bugs\.debian\.org(?::80)?/cgi-bin/soap.cgi$',
		 );

    foreach ($config_file, grep {!/~$/} glob((File::Spec->splitpath($config_file))[1].'conf.d/*')) {

	open my $fh, '<', $_ or die $!;
	local $/; # Slurp
	my $buf = $fh->getline;
	$buf=~s/\\\n#/\n#/mg; # fix broken multilines
	$buf=~s/\\\n//mg; # merge multilines
	
	for(split(/\n/, $buf))
	  {
	      next if(/^#/); # weed out whole comment lines immediately
	
	      s/#.*//; # kill off comments
	      s/^\s+//; # kill off leading spaces
	      s/\s+$//; # kill off trailing spaces
	
	      if ($_)
		{
		    my ($key, $value) = split(/\s*=\s*/); # split into key and value pair
		    $value = 0 unless ($value);
		    #print "key: $key, value: $value\n";
		    $config{$key} = $value;
		    #print "$config{$key}\n";
		}
	  }

	close $fh;
    }

    if ($config{logdir}) { # Recognise the old name
	$config{log_dir} = $config{logdir};
	delete $config{logdir};
    }

    return \%config;
}

sub cfg_split {
    my ($item) = @_;
    return $item ? grep {!/^$/} split(/\s*[,;]\s*/, $item) : undef;
}

# check directories exist and are writable
# Needs to run as root as parent directories may not be writable
sub check_install {
    # Die if we have not been configured correctly
    die "$0: No cache_dir directory!\n" if (!-d $cfg->{cache_dir});

    my $uid = $cfg->{user}=~/^\d+$/ ? $cfg->{user} : getpwnam($cfg->{user});
    my $gid = $cfg->{group}=~/^\d+$/ ? $cfg->{group} : getgrnam($cfg->{group});

    if (!defined ($uid || $gid)) {
	die "Unable to get user:group";
    }

    my @dir = ($cfg->{cache_dir}, $cfg->{log_dir}, "$cfg->{cache_dir}/private",
		     "$cfg->{cache_dir}/import", "$cfg->{cache_dir}/packages",
		     "$cfg->{cache_dir}/headers", "$cfg->{cache_dir}/temp");
    foreach my $dir (@dir) {
	if (!-d $dir) {
	    print "Info: $dir missing. Doing mkdir($dir, 0755)\n";
	    mkdir($dir, 0755) || die "Unable to create $dir: $!";
	}
	if ((stat($dir))[4] != $uid || (stat(_))[5] !=  $gid) {
	    print "Warning: $dir -- setting ownership to $uid:$gid\n";
	    chown ($uid, $gid, $dir) || die "Unable to set ownership for $dir: $!";
	}
    }
    for my $file ("$cfg->{log_dir}/access.log", "$cfg->{log_dir}/error.log") {
	if(!-e $file) {
	    print "Warning: $file missing. Creating.\n";
	    open(my $tmp, '>', $file) || die "Unable to create $file: $!";
	    close($tmp);
	    chown ($uid, $gid, $file) || die "Unable to set ownership for $file: $!";
	}
    }
    return;
}

# Convert a human-readable IPv4 address to raw form (4-byte string)
# Returns undef if the address is invalid
sub ipv4_normalise {
    my ($ip) = @_;
	return if $ip =~ /:/;
	my @in = split (/\./, $ip);
	return '' if $#in != 3;
	my $out = '';
	foreach my $num (@in)
	{
		return if $num !~ /^[[:digit:]]{1,3}$/o;
		$out .= pack ("C", $num);
	}
	return $out;
}

# Convert a human-readable IPv6 address to raw form (16-byte string)
# Returns undef if the address is invalid
sub ipv6_normalise {
    my ($ip) = @_;
	return "\0" x 16 if $ip eq '::';
	return if $ip =~ /^:[^:]/  || $ip =~ /[^:]:$/ || $ip =~ /::.*::/;
	my @in = split (/:/, $ip);
	return if $#in > 7;
	shift @in if $#in >= 1 && $in[0] eq '' && $in[1] eq ''; # handle ::1 etc.
	my $num;
	my $out = '';
	my $tail = '';
	while (defined ($num = shift @in) && $num ne '') # Until '::' found or end
	{
	    # Mapped IPv4
	    if ($num =~ /^(?:[[:digit:]]{1,3}\.){3}[[:digit:]]{1,3}$/) {
		$out .= ipv4_normalise($num);
	    } else {
		return if $num !~ /^[[:xdigit:]]{1,4}$/o;
		$out .= pack ("n", hex $num);
	    }
	}
	foreach my $num (@in) # After '::'
	{
	    # Mapped IPv4
	    if ($num =~ /^(?:[[:digit:]]{1,3}\.){3}[[:digit:]]{1,3}$/) {
		$tail .= ipv4_normalise($num);
		last;
	    }
	    else {
	        return if $num !~ /^[[:xdigit:]]{1,4}$/o;
		$tail .= pack ("n", hex $num);
	    }
	}
	my $l = length ($out.$tail);
	return $out.("\0" x (16 - $l)).$tail if $l < 16;
	return $out.$tail if $l == 16;
	return;
}

# Make a netmask from a CIDR network-part length and the IP address length
sub make_mask {
	my ($mask, $bits) = @_;
	return if $mask < 0 || $mask > $bits;
	my $m = ("\xFF" x ($mask / 8));
	$m .= chr ((-1 << (8 - $mask % 8)) & 255) if $mask % 8;
	return $m . ("\0" x ($bits / 8 - length ($m)));
}

# Arg is ref to flattened hash. Returns hash ref
sub hashify {
    my ($href) = @_;
    return unless $$href;
    if ($$href =~ /^FrT;/) {
	# New format: FreezeThaw
	return (thaw($$href))[0];
    } elsif ($$href =~ /. ./) {
    	# Old format: join
	return {split(/ /, $$href)};
    } else {
	return;
    }
}

# returns HTTP::Response
sub read_header {
    my ($file) = @_;
    my $r;
    if ($file && open(my $fh, '<', $file)) {
	local $/; # Slurp
	$r = HTTP::Response->parse(<$fh>);
	close($fh);
	chomp_message($r);
    }
    return $r;
}

# HTTP::Response->parse is leaving \r on the end of the message!
sub chomp_message {
    my ($r) = @_;

    for ($r->message) {
	local $/ = "\r";
	redo if chomp;
	$r->message($_);
    }
    return $r;
}

# Returns valid namespace from URL/path array
sub get_namespace {
    my ($pathref) = @_;

    if ($cfg->{distinct_namespaces}) {
	# Use path_map, if defined
	if (defined $cfg->{_path_map}{@{$pathref}[0]}) {
	    return @{$pathref}[0];
	}
	# Search from the end so we will always have something at $i-1
	for (my $i = $#{$pathref}; $i; $i--) {
	    last if $i == 1; # STOP! Don't want the hostname
	    return @{$pathref}[$i-1] if @{$pathref}[$i] =~ /^(?:pool|dists)$/;
	}
    }
    return;
}

# Returns URI object of url used to fetch file
sub get_original_url {
    my ($filename) = @_;
    my $uri;

    # Try cached headers first
    if (my $response = read_header("$cfg->{cache_dir}/headers/$filename")) {
	$uri = URI->new($response->header('X-AptCacher-URL'));
    }
    # Old complete file
    if (!$uri && -f (my $complete_file = "$cfg->{cache_dir}/private/$filename.complete")) {
	open(my $cfh, '<', $complete_file) || die "Failed to open $complete_file:$!";
	$uri = URI->new(<$cfh>);
	close($cfh);
    }
    return $uri;
}

# Stores data flattened for use in tied hashes
sub extract_sums {
   my ($name, $hashref) = @_;

   seek($name,0,0) if fileno($name);
   my $raw = IO::Uncompress::AnyUncompress->new($name)
     or die "Decompression failed: $AnyUncompressError\n";

   # If arg is fd get filename
   $name = readlink('/proc/self/fd/'.fileno($name)) if fileno($name);

   my ($indexbase) = ($name =~ /([^\/]*_)(?:Index|Release)(?: \(deleted\))?$/);
   $indexbase = '' unless $indexbase; # Empty by default (for Sources)

   my ($skip,%data);
   while(<$raw>) {
       last if $AnyUncompressError;
       chomp;
       # This flag prevents us bothering with the History section of diff_Index files
       if (/^SHA1-(?:Current|History)/) {
	   $skip = 1;
       }
       elsif (/^SHA1-Patches:/) {
	   $skip = 0;
       }
       elsif (/^\s(\w{32}|\w{40}|\w{64})\s+(\d+)\s(\S+)$/) { # diff_Index/Release/Sources
	   next if $skip;
	   my $hexdigest=$1;
	   my $size=$2;
	   my $file=$indexbase.$3;

	   $file=~s!/!_!g; # substitute any separators in indexed filename

	   if ($name =~ /Index$/) {
	       $file.=".gz";
	   }
	   elsif ($name =~ /_Sources(?:\.gz|\.bz2)?$/) {
	       # Prepend namespace, if required
	       $file = File::Spec->catfile(get_namespace([split(/_/, (File::Spec->splitpath($name))[2])]), $file);
	   }
	   $data{$file}{size} = $size;
	   for (my $len = length($hexdigest)) { # Select algorithm based on hex length
	       $len == 32 # md5
		 && do { $data{$file}{md5}=$hexdigest; last; };
	       $len == 40 # sha1
		 && do { $data{$file}{sha1}=$hexdigest; last; };
	       $len == 64 # sha256
		 && do { $data{$file}{sha256}=$hexdigest; last; };
	       warn "Unrecognised algorithm length: $len. Ignoring.";
	   }
       }
       elsif(/^MD5sum:\s+(.*)$/) { # Packages
	   $data{md5}=$1;
       }
       elsif(/^SHA1:\s+(.*)$/) {
	   $data{sha1}=$1;
       }
       elsif(/^SHA256:\s+(.*)$/) {
	   $data{sha256}=$1;
       }
       elsif(/^Size:\s+(.*)$/) {
	   $data{size}=$1;
       }
       elsif(/^Filename:\s+.*?([^\/]+)$/) { # Non-greedy quantifier essential
	   # Prepend namespace, if required
	   $data{file} = File::Spec->catfile(get_namespace([split(/_/, (File::Spec->splitpath($name))[2])]), $1);
       }

       if(/^$/|| $raw->eof()) { # End of record/file
	   if (exists $data{file}) {
	       # From Packages. Convert to hash of hashes with filename as key
	       foreach (qw(size md5 sha1 sha256)) {
		   $data{$data{file}}{$_} = $data{$_};
		   delete $data{$_};
	       }
	       delete $data{file};
	   }

	   foreach (keys %data) {
	       $hashref->{$_} = freeze($data{$_});
	   }
	   %data = (); # Reset
       }
   };
   if ($AnyUncompressError) {
       warn "$name Read failed: $AnyUncompressError. Aborting read\n";
       return;
   }
   return 1;
}

{ # Scoping block
    my $glock;

    sub set_global_lock {
	my ($msg)=@_;

	my $glockfile="$cfg->{cache_dir}/private/glock";

	$msg='Unspecified' if !$msg;

	debug_message("Global lock: \u$msg") if defined (&debug_message);

	#may need to create it if the file got lost
	open($glock, ((-f $glockfile) ? '<' : '>'), $glockfile) || die "Unable to open lockfile: $!";
	if (!flock($glock, LOCK_EX)) {
	    debug_message("Unable to lock $glockfile: $!") if defined (&debug_message);
	    die "Unable to lock $glockfile: $!";
	}
	return defined($glock);
    }

    sub release_global_lock {
	unless ($glock->opened) {
	    carp('Attmept to free lock not held');
	    return;
	}
	flock($glock, LOCK_UN) || die "Unable to release lock: $!";
	close $glock || die "Unable to close lock: $!";
	debug_message("Release global lock") if defined (&debug_message);
	return;
    }
}


sub setup_ownership {
    my $uid=$cfg->{user};
    my $gid=$cfg->{group};

    if($cfg->{chroot}) {
	if($uid || $gid) {
	    # open them now, before it is too late
	    # FIXME: reopening won't work, but the lose of file handles needs to be
	    # made reproducible first
	    open_log_files();
	}
	chroot $cfg->{chroot} || die "Unable to chroot: $1";
	chdir $cfg->{chroot};
    }

    if($gid) {
	if($gid=~/^\d+$/) {
	    my $name=getgrgid($gid);
	    die "Unknown group ID: $gid (exiting)\n" if !$name;
	}
	else {
	    $gid=getgrnam($gid);
	    die "No such group (exiting)\n" if !defined($gid);
	}
	setgid($gid) || barf("setgid failed: $!");;
	$) =~ /^$gid\b/ && $( =~ /^$gid\b/ || barf("Unable to change group id");
    }

    if($uid) {
	if($uid=~/^\d+$/) {
	    my $name=getpwuid($uid);
	    die "Unknown user ID: $uid (exiting)\n" if !$name;
	}
	else {
	    $uid=getpwnam($uid);
	    die "No such user (exiting)\n" if !defined($uid);
	}
	setuid($uid) || barf("setuid failed: $!");
	$> == $uid && $< == $uid || barf("Unable to change user id");
    }
    return;
}

sub barf {
	my $errs = shift;
	die "--- $0: Fatal: $errs\n";
}

sub is_index_file {
    my ($file) = @_;
    return ($file =~ /$cfg->{index_files_regexp}/);
}

sub is_package_file {
    my ($file) = @_;
    return ($file =~ /$cfg->{package_files_regexp}/);
}

sub is_installer_file {
    my ($file) = @_;
    return ($file =~ /$cfg->{installer_files_regexp}/);
}

sub is_checksum_import_file {
    my ($file) = @_;
    return ($file =~ /$cfg->{checksum_files_regexp}/);
}

sub load_checksum {
    return unless $cfg->{checksum};
    if (Module::Load::Conditional::check_install(module => 'BerkeleyDB')) {
	require('apt-cacher-lib-cs.pl');
    }
    else {
	warn "Checksum disabled as BerkeleyDB not found. Install libberkeleydb-perl\n";
	$cfg->{checksum}=0;
    }
    return;
}

######### HOOKS ###########
#
# arg: file to be scanned and added to DB
sub import_sums {
   return 1;
}

# purpose: ?create?, lock the DB file and establish DB connection
sub db {
   return 1;
}

# args: filehandle and DB handle
sub check_sum {
   return 1;
}

1;
