//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2009-2011 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <errno.h>

#include "common.h" 
#include "utils.h"

#ifdef USE_L2CAP

#ifdef USE_BLUEZ
#include <bluetooth/bluetooth.h>
#include <bluetooth/l2cap.h>
#endif

#ifdef USE_BT_FBSD
#include <bluetooth.h> 
#include <sdp.h> 
#include <err.h>
#endif

extern char tmp[MAXMAXLEN];

//
// Support SDP
//

#ifdef USE_BLUEZ
 
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

static sdp_session_t *session = NULL;
static sdp_record_t  *record  = NULL; 

static int portfd       = 0;
static int sportfd      = 0;

void sdpRegisterL2cap(int port)
{
    uint8_t svc_uuid_int[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xab, 0xcd };
    const char *service_name = "anyRemote/l2cap";
    const char *svc_dsc = "Bluetooth remote control";
    const char *service_prov = "anyRemote";
    uint8_t l2cap_port = port;

    uuid_t root_uuid, l2cap_uuid, svc_uuid, svc_class_uuid;
    sdp_list_t *l2cap_list = 0, 
               *root_list = 0,
               *proto_list = 0, 
               *access_proto_list = 0,
               *svc_class_list = 0,
               *profile_list = 0;
    sdp_data_t *channel = 0;
    sdp_profile_desc_t profile;
    record = sdp_record_alloc();

    // set the general service ID
    sdp_uuid128_create( &svc_uuid, &svc_uuid_int );
    sdp_set_service_id( record, svc_uuid );

    // set the service class
    sdp_uuid16_create(&svc_class_uuid, SERIAL_PORT_SVCLASS_ID);
    svc_class_list = sdp_list_append(0, &svc_class_uuid);
    sdp_set_service_classes(record, svc_class_list);

    // set the Bluetooth profile information
    sdp_uuid16_create(&profile.uuid, SERIAL_PORT_PROFILE_ID);
    profile.version = 0x0100;
    profile_list = sdp_list_append(0, &profile);
    sdp_set_profile_descs(record, profile_list);

    // make the service record publicly browsable
    sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
    root_list = sdp_list_append(0, &root_uuid);
    sdp_set_browse_groups( record, root_list );

    // set l2cap information
    sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
    l2cap_list = sdp_list_append( 0, &l2cap_uuid );
    channel = sdp_data_alloc(SDP_UINT8, &l2cap_port);
    sdp_list_append(l2cap_list, channel );
    proto_list = sdp_list_append( 0, l2cap_list );

    access_proto_list = sdp_list_append( 0, proto_list );
    sdp_set_access_protos( record, access_proto_list );

    // set the name, provider, and description
    sdp_set_info_attr(record, service_name, service_prov, svc_dsc);

    // connect to the local SDP server, register the service record, 
    // and disconnect
    session = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, SDP_RETRY_IF_BUSY);
    if ( (!(session && record)) || sdp_record_register(session, record, 0) == -1) {
        logger("ERR", "can not register SDP service");
    }

    // cleanup
    sdp_data_free( channel );
    sdp_list_free( l2cap_list, 0 );
    sdp_list_free( proto_list, 0 );
    sdp_list_free( root_list, 0 );
    sdp_list_free( access_proto_list, 0 );
    sdp_list_free( svc_class_list, 0 );
    sdp_list_free( profile_list, 0 );
}
#endif

#ifdef USE_BT_FBSD
static void			*session = NULL;
static uint32_t		 record;


void sdpRegisterL2cap(int port)
{
	errx(1, "Not yet supported");
}
#endif

void sdpDeregisterL2cap()
{
    #ifdef USE_BLUEZ
    if (session != NULL) {
        sdp_record_unregister(session, record);
        session = NULL;
    }
    #endif
    #ifdef USE_BT_FBSD
    if (session != NULL) {
	    sdp_unregister_service(session, record);
	    sdp_close(session);
	    session = NULL;
    }
    #endif
}

//
// Support L2CAP sockets
//

int openL2capPort(int port) 
{
	#ifdef USE_BLUEZ
        struct sockaddr_l2 l2_addr;
	#endif
	#ifdef USE_BT_FBSD

	#endif
	
	struct sockaddr*   socketaddr = NULL;

	int sz;

	if (sportfd) { 
		logger("ERR", "L2CAP socket was already opened");
		return 1;
	}

	if ((sportfd = socket(PF_BLUETOOTH, SOCK_SEQPACKET|SOCK_CLOEXEC, BTPROTO_L2CAP)) < 0) {
		logger("ERR", "opening socket");
		printf("ERROR: opening socket\n");
	
		return -1;
	}
	
	#ifdef USE_BLUEZ
	memset((void *) &l2_addr, 0, sizeof(l2_addr));
	sz = sizeof(l2_addr);
	
	// bind socket to the specified port of the first available local bluetooth adapter
        l2_addr.l2_family = AF_BLUETOOTH;
        l2_addr.l2_bdaddr = *BDADDR_ANY;
        l2_addr.l2_psm = htobs(0x1001); //port);

        sdpRegisterL2cap(port);
	sprintf(tmp, "registered L2CAP on port %i", port);
	logger("INF", tmp);
	socketaddr=(struct sockaddr *)&l2_addr;
	#endif
	
	#ifdef USE_BT_FBSD

	#endif

	if (bind(sportfd, (struct sockaddr *) socketaddr, sz) < 0) {
		logger("ERR", "on binding");
		printf("ERROR: on binding %d->%s\n", errno, strerror(errno));
		return -1;
	}
	return 1;
}

int closeL2capPort(int final) 
{
	logger("INF", "closeL2capPort");

	if (portfd) { 
		close(portfd);
		portfd = 0;
	}
	if (sportfd) { 
		close(sportfd);
		sportfd = 0;
	}
	
	if (finale) {
		sdpDeregisterL2cap();
	}
	return 1;
}

int listenAndAcceptL2capConn() 
{
	int cnt;
	char buf[1024] = { 0 };

	#ifdef USE_BLUEZ
	struct sockaddr_l2 l2rem_addr;
	socklen_t opt = sizeof(l2rem_addr);
	#endif

	#ifdef USE_BT_FBSD

	#endif

	logger("INF", "listenAndAcceptL2capConn");
	cnt = 0;

	while (1) {	       
		listen(sportfd,5);
		 
		portfd = accept(sportfd, (struct sockaddr *)&l2rem_addr, &opt);
		    	 
		if (portfd == -1 && errno == EAGAIN) {
			
			if (cnt >= 60) {    // Print to log every minute
				logger("INF", "listenAndAcceptL2capConn: waiting for connection");
				cnt = 0;
			}
			fflush(stdout);
	
			sleep(1);
			cnt++;
	
			continue;
		}	 
		 
		if (portfd < 0) {
			logger("ERR", "on accept");
			printf("ERROR: on accept %d\n", errno);
			return -1;
		}   
	       
	        ba2str(&l2rem_addr.l2_bdaddr, buf);
		sprintf(tmp, "listenAndAcceptL2capConn: accepted from %s", buf);
		logger("INF", tmp);

		break;
	}
	return 1;
}

int writeL2capConn(char* command, int count)
{
	if (portfd <= 0) return -1;
	logger("DBG", "writeL2capConn");

	// send command
	if (count > 0) {
        
		memset(tmp, 0, MAXMAXLEN);
        	strcat(tmp, "writeL2capConn ");
                
                int logSz = (count > 256 ? 255 : count);
                
                // it is possible to get binary data here
                memcpy(tmp, command, logSz); // Do not dump long commands
                tmp[logSz] = '\0';
        	logger("DBG", tmp);
		
		sprintf(tmp, "writeL2capConn %d bytes", count);
		logger("INF", tmp);

        	int n = write(portfd,command,count);        
        	if (n < 0) {
        		logger("ERR", "error writing to socket");
        		return EXIT_NOK;
        	}
        	//tcdrain(portfd);
	}
	return EXIT_OK;
}

int writeBytesL2capConn(char* command) 
{
	//logger("DBG", "writeL2capSocketConn");

	// send command
	if (command && command[0]) {
		
		char byteStr[MAXCKPDLEN];
		memset(byteStr,0,MAXCKPDLEN);
		strncpy(byteStr,command,MAXCKPDLEN-1);

		sprintf(tmp, "writeL2capSocketConn >%s<", byteStr);
		logger("DBG", tmp);

		char* bStr = strtok(byteStr,",");
		while (bStr != NULL) {
			
			//sprintf(tmp, "Next byte is >%s<", bStr);
			//logger("DBG", tmp);

			char bStripped[4];
		
			while (*bStr == ' ') {
				bStr++;
			}
			int i = 0;
			while (*bStr != ' ' && i < 3) {  // 0 < ... < 256
				bStripped[i] = *bStr;
				bStr++;
				i++;
			}
			bStripped[i] = '\0';

			sprintf(tmp, "Next byte is >%s<", bStripped);
			logger("DBG", tmp);
		
			unsigned char byte2write[2];
			byte2write[0] = (unsigned char) atoi(bStripped);
			byte2write[1] = '\0';
		
			if ( write(portfd, byte2write, 1) < 0) {
				logger("ERR", "error writing to socket");
				return EXIT_NOK;
			}
			bStr = strtok(NULL,",");
		}
	}
	//logger("DBG", "writeL2capSocketConn EXIT");
	return EXIT_OK;
}
#endif
