/* Copyright (C) 2002, 2003, 2004, 2005 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#ifndef __ERROR_HPP
#define __ERROR_HPP

#include <boost/shared_ptr.hpp>
#include <exception>
#include <stdarg.h>
#include <sstream>

class Error;

///
typedef boost::shared_ptr< Error > ErrorPtr;

/** An exception class for exception-handling.
    You can use the ERRORMACRO for throwing exceptions:
    \code

    void test( int i ) throw (Error)
    {
      ERRORMACRO( i > 0, Error, ,
                  "Parameter for method test must be greater than zero (but "
                  "was " << i << ")." );
    }
    
    int main(void)
    {
      try {
        test( -5 );
      } catch ( Error &e ) {
        cerr << e.what() << endl;
      };
      return 0;
    }
    \endcode
    Mind that the macro uses a variable with the name \c _e. Make sure, that
    you don't use this variable-name in any of the macro-arguments!
    @date Fri Apr 12 2002
    @author Jan Wedekind (wedesoft@users.sourceforge.net) */
class Error: public std::exception {
 public:
  /// Constructor.
  Error(void) {}
  ///
  virtual ~Error(void) throw() {}
  /// Copy constructor.
  Error( const Error &aError );
  ///
  template< typename T >
  std::ostream &operator<<( const T &t ) { msgStream << t; return msgStream; }
  /** Interface for manipulators.
      Manipulators such as \c std::endl and \c std::hex use these
      functions in constructs like "Error e; e << std::endl".
      For more information, see the iomanip header. */
  std::ostream &operator<<( std::ostream& (*__pf)( std::ostream&) )
    { (*__pf)( msgStream ); return msgStream; }
  /// Returns error message (not thread-safe).
  virtual const char *what(void) const throw();
 protected:
  /// String containing the error message.
  std::ostringstream msgStream;
  /** Temporary string to do null-termination.
      The method \c what() requires a null-terminated string. */
  static std::string message;
};

#define ERRORMACRO( condition, class, params, message )   \
  if ( !( condition ) ) {                                 \
    class _e params;                                      \
    _e << message;                                         \
    throw _e;                                              \
  };

#endif
