// ---------------------------------------------------------------------------
// - OutputString.cpp                                                        -
// - standard object library - output string stream class implementation     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Ascii.hpp"
#include "Vector.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "OutputString.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a new default output string stream. 

  OutputString::OutputString (void)  {
    d_buffer.reset ();
  }

  // create a new output string with a value

  OutputString::OutputString (const String& value)  {
    d_buffer.reset ();
    d_buffer.add (value);
  }

  // return the class name

  String OutputString::repr (void) const {
    return "OutputString";
  }

  // flush this output string buffer
  
  void OutputString::flush (void) {
    d_buffer.reset ();
  }

  // write one character to the output stream
  
  void OutputString::write (const char value) {
    wrlock ();
    try {
      d_buffer.add (value);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a character string to the output stream
  
  void OutputString::write (const char* value) {
    wrlock ();
    try {
      long size = Ascii::strlen (value);
      if (size == 0) return;
      char c = nilc;
      while ((c = *value++) != nilc) d_buffer.add (c);
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the length of the output string

  long OutputString::length (void) const {
    rdlock ();
    try {
      long result = d_buffer.length ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // return a string in this output string and flush the buffer

  String OutputString::tostring (void) {
    rdlock ();
    try {
      String result = d_buffer.tostring ();
      d_buffer.reset ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_FLUSH    = zone.intern ("flush");
  static const long QUARK_TOSTRING = zone.intern ("to-string");

  // create a new object in a generic way

  Object* OutputString::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new OutputString;

    // check for 1 argument
    if (argc == 1) {
      String sval = argv->getstring (0);
      return new OutputString (sval);
    }
    throw Exception ("argument-error", 
		     "invalid arguments with with output string"); 
  }

  // return true if the given quark is defined

  bool OutputString::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Output::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* OutputString::apply (Runnable* robj, Nameset* nset, const long quark,
			       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_TOSTRING) return new String (tostring ());
      if (quark == QUARK_FLUSH) {
	flush ();
	return nilp;
      }
    }
    // call the output method
    return Output::apply (robj, nset, quark, argv);
  }
}
