// ---------------------------------------------------------------------------
// - Class.hpp                                                               -
// - afnix engine - class class definition                                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_CLASS_HPP
#define  AFNIX_CLASS_HPP

#ifndef  AFNIX_LOCALSET_HPP
#include "Localset.hpp"
#endif

#ifndef  AFNIX_QARRAY_HPP
#include "Qarray.hpp"
#endif

namespace afnix {

  /// The Class class is the class used to model afnix class object. The afnix
  /// model is based on a single inheritance mechanism. A class binds only
  /// a local set - which somehow is equivalent to c++ static data member.
  /// When an instance of that class is created, it is the instance initial
  /// method which is responsible to define the local data members.
  /// @author amaury darsch

  class Class : public virtual Object {
  private:
    /// the inference class
    Class* p_infer;
    /// the inference const flag
    bool   d_ctinf;
    /// the deferent class
    Class* p_defer;
    /// the deferent const flag
    bool   d_ctdef;
    /// the class local set
    Localset* p_cset;
    /// the list of data members
    Qarray d_mdata;

  public:
    /// create a new class 
    Class (void);

    /// destroy this class class
    ~Class (void);

    /// @return the class name
    String repr (void) const;

    /// make this class a shared object
    void mksho (void);

    /// add a new data member by quark
    /// @param quark the quark to add
    void add (const long quark);

    /// @return the array of data members
    const Qarray& getmdata (void) const;

    /// @return an object in the class localset but do not evaluate
    Object* find (const long quark) const;

    /// set the class inferent object
    /// @param object the class object to set
    /// @param flag   the class const flag
    Object* setinfer (Object* object, const bool flag);

    /// set the class inferent class
    /// @param infer the class to set
    /// @param flag  the class const flag
    Object* setinfer (Class* infer, const bool flag);

    /// set the class deferent object
    /// @param object the class object to set
    /// @param flag   the class const flag
    Object* setdefer (Object* object, const bool flag);

    /// set the class deferent class
    /// @param defer the class to set
    /// @param flag  the class const flag
    Object* setdefer (Class* defer, const bool flag);

  private:
    // make the copy constructor private
    Class (const Class&);
    // make the assignment operator private
    Class& operator = (const Class&);

  public:
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// operate this class with another object
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);

    /// set a constant object by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to define as const
    /// @param object the object to set
    Object* cdef (Runnable* robj, Nameset* nset, const long quark, 
		  Object* object);

    /// set an object by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to define as trans
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, const long quark, 
		  Object* object);

    /// evaluate an object member by quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset
    /// @param quark the member quark to evaluate
    Object* eval (Runnable* robj, Nameset* nset, const long quark);

    /// apply this object by arguments
    /// @param robj the current runnable
    /// @param nset the current nameset 
    /// @param args the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, Cons* args);

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param args  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Cons* args);
  };
}

#endif
