#include "rpggame.h"

namespace rpgio
{
	#define GAME_VERSION 13
	#define GAME_MAGICZ "RPGS"

	/**
		SAVING STUFF
			STRINGS - use writestring(stream *, char *)
			VECTORS - write an int corresponding to the number of elements, then write the elements
			POINTERS - convert to reference, if this is not possible don't save it
			TIME - subtract lastmillis before writing
			VEC - use writevec macro, you need to write all 3 coordinates independantly

		LOADING STUFF
			STRINGS - use readstring(stream *)
			VECTORS - read the next int, and use that as the basis of knowing how many elements to load
			POINTERS - read the reference and convert it back to a pointer of the intended object. If this can't be done reliably don't save it
			TIME - add lastmillis onto it
			VEC - use readvec macro, you need to read all 3 coordinates independantly

		REMINDERS
			ORDER - you must read and write items in the same order
			LITTLE ENDIAN - pullil add getlil for single values and lilswap for arrays and objects are going to be your bestest friends
			POINTERS - they change each run, remember that, for convention, use -1 for players
			ARGUMENTS - as a argument to a function, ORDER IS NOT RESPECTED, use with extreme care

		ORDER
			they are to coincide with the order of the item structs, and before the map functions, the order in which crap is stored there
	*/

	#define readvec(v) \
		v.x = f->getlil<float>(); \
		v.y = f->getlil<float>(); \
		v.z = f->getlil<float>(); \
		if(DEBUG_IO) \
			conoutf("DEBUG: Read vec (%f, %f, %f) from file", v.x, v.y, v.z);

	#define writevec(v) \
		f->putlil(v.x); \
		f->putlil(v.y); \
		f->putlil(v.z); \
		if(DEBUG_IO) \
			conoutf("DEBUG: Wrote vec (%f, %f, %f) to file", v.x, v.y, v.z);

	struct saveheader
	{
		char magic[4];
		int version;
	};

	bool abort = false;
	mapinfo *lastmap = NULL;

	enum
	{
		SAVE_SCRIPT = 0, // script bases
		SAVE_EFFECT, // pretty effect bases
		SAVE_STATUS, // status effect bases
		SAVE_ITEM,   // item bases
		SAVE_AMMO,   // ammo groups
		SAVE_CHAR,   // character bases
		SAVE_FACT,   // factions
		SAVE_VAR,    // game variables
		SAVE_CAMERA, // cutscenes, see rpgcamera.spp
		SAVE_MAP,    // map statuses
		SAVE_TIP,    // tips
		SAVE_OBJECT, // object base
		SAVE_MSCRIPT, // map scripts
		SAVE_RECIPE
	};

	rpgent *entfromnum(int num)
	{
		if(num == -2)
			return NULL;
		if(num == -1)
			return game::player1;
		else if(num >= lastmap->objs.length())
		{
			conoutf("WARNING: invalid entity num (%i), possible corruption", num);
			return NULL;
		}
		return lastmap->objs[num];
	}

	const char *readstring(stream *f)
	{
		int len = f->getlil<int>();
		char *s = NULL;
		if(len)
		{
			s = new char[len + 1];
			f->read(s, len + 1);
			if(DEBUG_IO)
				conoutf(CON_DEBUG, "DEBUG: Read \"%s\" from file (%i)", s, len);
		}

		return s;
	}

	script *readscript(stream *f)
	{
		script *scr = new script();

		int numdiags = f->getlil<int>();
		loopi(numdiags)
		{
			if(DEBUG_IO)
				conoutf(CON_DEBUG, "DEBUG: reading script->dialogue[%i] from file", i);
			scr->dialogue.add(new rpgchat(readstring(f), readstring(f)));
		}
		loopi(SCR_MAX)
		{
			if(DEBUG_IO)
				conoutf(CON_DEBUG, "DEBUG: reading script->scripts[%i] from file", i);
			scr->scripts[i] = readstring(f);
		}

		return scr;
	}

	effect *readeffect(stream *f)
	{
		effect *eff = new effect();

		eff->flags = f->getlil<int>();
		eff->decal = f->getlil<int>();

		eff->mdl = readstring(f);
		eff->projpart = f->getlil<int>();
		eff->projpartcol = f->getlil<int>();
		eff->projpartsize = f->getlil<float>();

		eff->trailpart = f->getlil<int>();
		eff->trailpartcol = f->getlil<int>();
		eff->trailpartfade = f->getlil<int>();
		eff->trailpartgrav = f->getlil<int>();
		eff->trailpartsize = f->getlil<float>();

		eff->traillightradius = f->getlil<int>();
		readvec(eff->traillightcol);

		eff->deathpart = f->getlil<int>();
		eff->deathpartcol = f->getlil<int>();
		eff->deathpartfade = f->getlil<int>();
		eff->deathpartgrav = f->getlil<int>();
		eff->deathpartsize = f->getlil<float>();

		eff->deathlightflags = f->getlil<int>();
		eff->deathlightfade = f->getlil<int>();
		eff->deathlightradius = f->getlil<int>();
		eff->deathlightinitradius = f->getlil<int>();
		readvec(eff->deathlightcol);
		readvec(eff->deathlightinitcol);

		return eff;
	}

	statusgroup *readstatus(stream *f)
	{
		statusgroup *group = new statusgroup();

		int numeffects = f->getlil<int>();
		loopi(numeffects)
		{
			status *stat = NULL;
			int type = f->getlil<int>();
			switch(type)
			{
				case STATUS_POLYMORPH:
				{
					status_polymorph *poly = new status_polymorph();
					poly->mdl = readstring(f);
					stat = poly;

					break;
				}
				case STATUS_LIGHT:
				{
					status_light *light = new status_light();
					readvec(light->colour);
					light->radius = f->getlil<int>();
					stat = light;

					break;
				}
				default:
				{
					status_generic *gen = new status_generic();
					gen->strength = f->getlil<int>();
					stat = gen;

					break;
				}
			}
			stat->type = type;

			group->effects.add(stat);
		}
		group->friendly = f->getchar();
		group->icon = readstring(f);
		group->name = readstring(f);
		group->description = readstring(f);

		return group;
	}

	item_base *readitem(stream *f)
	{
		item_base *item = new item_base();
		delete[] item->mdl;

		item->name = readstring(f);
		item->icon = readstring(f);
		item->description = readstring(f);
		item->mdl = readstring(f);

		item->script = f->getlil<int>();
		item->type = f->getlil<int>();
		item->flags = f->getlil<int>();
		item->worth = f->getlil<int>();
		item->weight = f->getlil<int>();
		item->cursebase = f->getlil<int>();

		int numuses = f->getlil<int>();
		loopi(numuses)
		{
			use *usable = NULL;
			int type = f->getlil<int>();
			switch(type)
			{
				case USE_WEAPON:
				{
					if(!usable) usable = new use_weapon(0);
					use_weapon *weap = (use_weapon *) usable;

					weap->range = f->getlil<int>();
					weap->angle = f->getlil<int>();
					weap->lifetime = f->getlil<int>();
					weap->effect = f->getlil<int>();
					weap->cost = f->getlil<int>();
					weap->flags = f->getlil<int>();
					weap->element = f->getlil<int>();
					weap->ammo = f->getlil<int>();
					weap->target = f->getlil<int>();
					weap->radius = f->getlil<int>();
					weap->kickback = f->getlil<int>();
					weap->charge = f->getlil<int>();
					weap->basecharge = f->getlil<float>();
					weap->mincharge = f->getlil<float>();
					weap->maxcharge = f->getlil<float>();
					weap->elasticity = f->getlil<float>();
					weap->speed = f->getlil<float>();
				}

				case USE_ARMOUR:
				{
					if(!usable) usable = new use_armour(0);
					use_armour *arm = (use_armour *) usable;

					f->read(&arm->reqs, sizeof(statreq));
					lilswap(&arm->reqs.attrs[0], STAT_MAX + SKILL_MAX);

					arm->slots = f->getlil<int>();
					arm->skill = f->getlil<int>();
				}
				case USE_CONSUME:
					if(!usable) usable = new use(0);

					usable->script = f->getlil<int>();
					usable->status = f->getlil<int>();
					usable->duration = f->getlil<int>();
					usable->cooldown = f->getlil<int>();
					usable->mul = f->getlil<float>();

					break;
			}

			item->uses.add(usable);
		}
		return item;
	}

	ammotype *readammo(stream *f)
	{
		ammotype *ammo = new ammotype();

		ammo->name = readstring(f);
		int num = f->getlil<int>();
		loopi(num)
			ammo->items.add(f->getlil<int>());

		return ammo;
	}

	char_base *readchar(stream *f, char_base *base = NULL) //also used to read character stats
	{
		if(!base) base = new char_base;

		delete[] base->name;
		delete[] base->mdl;

		base->name = readstring(f);
		base->mdl = readstring(f);

		f->read(&base->base, sizeof(stats));
		lilswap(&base->base.experience, 3);
		lilswap(&base->base.baseattrs[0], STAT_MAX * 2 + SKILL_MAX * 2);

		lilswap(&base->base.bonusthresh, ATTACK_MAX * 2);
		lilswap(&base->base.bonushealth, 6);
		lilswap(&base->base.bonushregen, 2);

		lilswap(&base->base.deltathresh, ATTACK_MAX * 2);
		lilswap(&base->base.deltahealth, 6);
		lilswap(&base->base.deltahregen, 2);

		int items = f->getlil<int>();
		loopi(items)
		{
			int b = f->getlil<int>();
			int q = f->getlil<int>();
			base->inventory.add(new invstack(b, q));
		}

		base->script = f->getlil<int>();
		base->faction = f->getlil<int>();

		return base;
	}

	faction *readfaction(stream *f)
	{
		faction *fact = new faction();

		fact->name = readstring(f);
		fact->logo = readstring(f);

		int num = f->getlil<int>();
		loopi(num)
		fact->relations.add(f->getlil<short>());

		return fact;
	}

	object_base *readobject(stream *f, object_base *obj = NULL) //also used to read object stats
	{
		if(!obj) obj = new object_base();
		delete[] obj->mdl;

		obj->mdl = readstring(f);
		obj->name = readstring(f);

		obj->flags = f->getlil<int>();
		obj->weight = f->getlil<int>();
		obj->script = f->getlil<int>();

		int inv = f->getlil<int>();
		loopi(inv)
		{
			int base = f->getlil<int>();
			int quantity = f->getlil<int>();
			obj->inventory.add(new invstack(base, quantity));
		}

		return obj;
	}

	extern void readcutscene(stream *f);
	extern void readcamera(stream *f);

	// many entities will probably have effects, and these effects may be confered to them by something that is saved after them.
	// this means that the reference they want may not be loaded
	// so we defer setting the references until they are all loaded
	struct reference
	{
		int ind;
		mapinfo *map;
		rpgent *&ref;
		reference(int i, mapinfo *m, rpgent *&r) : ind(i), map(m), ref(r) {}
		~reference() {}
	};
	vector<reference> updates;

	rpgent *readent(stream *f, rpgent *ent = NULL)
	{
		int type = f->getlil<int>();
		if(ent)
			type = ent->type();

		switch(type)
		{
			case ENT_CHAR:
			{
				if(!ent) ent = new rpgchar();
				rpgchar *loading = (rpgchar *) ent;

				readchar(f, loading);

				int equiplen = f->getlil<int>();
				loopi(equiplen)
				{
					int base = f->getlil<int>();
					int use = f->getlil<int>();
					loading->equipped.add(equipment(base, use));
				}

				loading->health = f->getlil<float>();
				loading->mana = f->getlil<float>();
				loading->lastaction = f->getlil<int>() + lastmillis;

				int num = f->getlil<int>();
				loopi(num)
					loading->route.add(f->getlil<ushort>());

				break;
			}
			case ENT_ITEM:
			{
				if(!ent) ent = new rpgitem();
				rpgitem *loading = (rpgitem *) ent;

				loading->base = f->getlil<int>();
				loading->quantity = f->getlil<int>();

				break;
			}
			case ENT_OBJECT:
			{
				if(!ent) ent = new rpgobject();
				rpgobject *loading = (rpgobject *) ent;

				readobject(f, loading);
				loading->lasttrigger = f->getlil<int>() + lastmillis;

				break;
			}
			default:
				conoutf(CON_ERROR, "ERROR: unknown entity type %i", type);
				abort = true;
				return NULL;
		}

		int numeffs = f->getlil<int>();
		loopi(numeffs)
		{
			victimeffect *eff = new victimeffect();
			updates.add(reference(f->getlil<int>(), lastmap, eff->owner));
			eff->startmillis = f->getlil<int>() + lastmillis;
			eff->duration = f->getlil<int>();
			eff->group = f->getlil<int>();
			eff->mul = f->getlil<float>();

			ent->seffects.add(eff);
		}

		ent->eyeheight = f->getlil<float>();
		readvec(ent->o);
		ent->newpos = ent->o;
		readvec(ent->vel);
		readvec(ent->falling);

		ent->yaw = f->getlil<float>();
		ent->pitch = f->getlil<float>();
		ent->roll = f->getlil<float>();

		ent->timeinair = f->getlil<int>();

		ent->state = f->getchar();
		ent->editstate = f->getchar();

		ent->lastjump = f->getlil<int>() + lastmillis;

		return ent;
	}

	mapscript *readmapscript(stream *f)
	{
		mapscript *scr = new mapscript();
		loopi(MSCR_MAX) scr->scripts[i] = readstring(f);

		return scr;
	}

	recipe *readrecipe(stream *f)
	{
		recipe *loading = new recipe();
		loopi(3)
		{
			vector<invstack> &cont = (i ? (i == 1 ? loading->catalysts : loading->product) : loading->ingredients);
			int num = f->getlil<int>();

			loopj(num)
			{
				int base = f->getlil<int>();
				int qty = f->getlil<int>();
				cont.add(invstack(base, qty));
			}
		}
		return loading;
	}

	void readmap(stream *f)
	{
		const char *name = readstring(f);
		mapinfo *loading = game::accessmap(name);
		lastmap = loading;

		loading->name = name;

		loading->loaded = f->getchar();

		int numobjs = f->getlil<int>(),
		    numactions = f->getlil<int>(),
		    numprojs = f->getlil<int>(),
		    numaeffects = f->getlil<int>(),
		    numblips = f->getlil<int>();

		loopi(numobjs)
			loading->objs.add(readent(f));

		loopvrev(updates)
		{
			if(updates[i].map == lastmap)
				updates[i].ref = entfromnum(updates[i].ind);
		}

		loopi(numactions)
		{
			action *act = NULL;
			int type = f->getlil<int>();

			switch(type)
			{
				case ACTION_TELEPORT:
				{
					int ent = f->getlil<int>();
					if(!entfromnum(ent))
					{//how'd that happen?
						conoutf("WARNING: loaded teleport loadaction for invalid ent? ignoring");
						f->getlil<int>();
						continue;
					}
					act = new action_teleport(entfromnum(ent), f->getlil<int>());
					break;
				}
				case ACTION_SPAWN:
				{
					int tag = f->getlil<int>(),
					    ent = f->getlil<int>(),
					    id = f->getlil<int>(),
					    amount = f->getlil<int>(),
					    qty = f->getlil<int>();
					act = new action_spawn(tag, ent, id, amount, qty);
					break;
				}
				case ACTION_SCRIPT:
				{
					act = new action_script(readstring(f));
					break;
				}
			}

			loading->loadactions.add(act);
		}

		loopi(numprojs)
		{
			projectile *p = new projectile();

			p->owner = (rpgchar *) entfromnum(f->getlil<int>());
			if(p->owner->type() != ENT_CHAR)
				p->owner = NULL;

			p->item = f->getlil<int>();
			p->use = f->getlil<int>();
			p->ammo = f->getlil<int>();
			p->ause = f->getlil<int>();

			readvec(p->o); readvec(p->dir); readvec(p->emitpos);
			p->lastemit = f->getlil<int>() + lastmillis;
			p->gravity = f->getlil<int>();
			p->deleted = f->getchar();

			p->pflags = f->getlil<int>();
			p->time = f->getlil<int>();
			p->dist = f->getlil<int>();

			p->fx = f->getlil<int>();
			p->radius = f->getlil<int>();
			p->elasticity = f->getlil<float>();
			p->charge = f->getlil<float>();

			loading->projs.add(p);
		}

		loopi(numaeffects)
		{
			areaeffect *aeff = new areaeffect();

			readvec(aeff->o);
			aeff->owner = entfromnum(f->getlil<int>());
			aeff->startmillis = f->getlil<int>() + lastmillis;
			aeff->lastemit = f->getlil<int>() + lastmillis;
			aeff->duration = f->getlil<int>();
			aeff->group = f->getlil<int>();
			aeff->radius = f->getlil<int>();
			aeff->fx = f->getlil<int>();
			aeff->mul = f->getlil<float>();

			loading->aeffects.add(aeff);
		}

		loopi(numblips)
		{
			///FIXME finalize blip structure and write me
		}

		lastmap = NULL;
	}

	void loadgame(const char *name)
	{
		defformatstring(file)("data/rpg/saves/%s.sgz", name);
		stream *f = opengzfile(file, "rb");

		if(!f)
		{
			conoutf(CON_ERROR, "ERROR: unable to read file: %s", file);
			return;
		}

		saveheader hdr;
		f->read(&hdr, sizeof(saveheader));
		lilswap(&hdr.version);

		if(hdr.version != GAME_VERSION || strncmp(hdr.magic, GAME_MAGICZ, 4))
		{
			conoutf(CON_ERROR, "ERROR: Unsupported version or corrupt save: %i (%i) - %4.4s (%s)", hdr.version, GAME_VERSION, hdr.magic, GAME_MAGICZ);
			delete f;
			return;
		}
		if(DEBUG_IO)
			conoutf(CON_DEBUG, "DEBUG: supported save: %i  %4.4s", hdr.version, hdr.magic);

		game::cleangame();
		game::mapdata = new hashtable<const char *, mapinfo>; //FIXME don't do this here
		localconnect();

		abort = false;
		const char *curmap = readstring(f);
		if(!curmap)
		{
			delete f;
			conoutf(CON_ERROR, "ERROR: no game/map in progress, aborting");
			localdisconnect();
			return;
		}

		lastmap = game::mapdata->access(curmap);
		readent(f, game::player1);
		lastmap = NULL;
		readcamera(f);

		short type = f->getchar();
		while(!f->end())
		{
			switch(type)
			{
				case SAVE_SCRIPT:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read script token %i", type);
					game::scripts.add(readscript(f));
					break;
				case SAVE_EFFECT:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read effect token %i", type);
					game::effects.add(readeffect(f));
					break;
				case SAVE_STATUS:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read status token %i", type);
					game::statuses.add(readstatus(f));
					break;
				case SAVE_ITEM:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read itembase token %i", type);
					game::items.add(readitem(f));
					break;
				case SAVE_AMMO:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read ammo token %i", type);
					game::ammotypes.add(readammo(f));
					break;
				case SAVE_CHAR:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read char token %i", type);
					game::chars.add(readchar(f));
					break;
				case SAVE_FACT:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read faction token %i", type);
					game::factions.add(readfaction(f));
					break;
				case SAVE_VAR:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read variable token %i", type);
					game::variables.add(f->getlil<int>());
					break;
				case SAVE_TIP:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read tip token %i", type);
					game::tips.add(readstring(f));
					break;
				case SAVE_MAP:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read mapinfo token %i", type);
					readmap(f);
					break;
				case SAVE_CAMERA:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read camera token %i", type);
					readcutscene(f);
					break;
				case SAVE_OBJECT:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: Read object token %i", type);
					game::objects.add(readobject(f));
					break;
				case SAVE_MSCRIPT:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: read map script token %i", type);
					game::mapscripts.add(readmapscript(f));
					break;
				case SAVE_RECIPE:
					if(DEBUG_IO)
						conoutf(CON_DEBUG, "DEBUG: read recipe token %i", type);
					game::recipes.add(readrecipe(f));
					break;
				default:
					conoutf(CON_ERROR, "ERROR: invalid savetype %i", type);
					abort = true;
					break;
			}
			if(abort)
			{
				localdisconnect();
				delete f;
				return;
			}

			type = f->getchar();
		}

		updates.shrink(0);
		delete f;

		game::transfer = true;
		load_world(curmap);

		return;
	}
	COMMAND(loadgame, "s");

	int enttonum(rpgent *ent)
	{
		if(ent == NULL)
			return -2;
		if(ent == game::player1)
			return -1;

		int i = lastmap->objs.find(game::player1);
		if(i != -1)
		{
			int num = lastmap->objs.find(ent);
			if(num > i)
				num--;
			return num;
		}
		else
			return lastmap->objs.find(ent);
	}

	void writestring(stream *f, const char *s)
	{
		int len = s ? strlen(s) : 0;
		f->putlil(len);

		if(!len) return;

		len++;
		f->write(s, len);
		if(DEBUG_IO)
			conoutf(CON_DEBUG, "DEBUG: Wrote \"%s\" to file (%i)", s, len);
	}

	void writescript(stream *f, script *scr)
	{
		f->putlil(scr->dialogue.length());
		loopv(scr->dialogue)
		{
			if(DEBUG_IO)
				conoutf(CON_DEBUG, "DEBUG: Writing script->dialogue[%i] to file", i);
			rpgchat *chat = scr->dialogue[i];
			writestring(f, chat->talk);
			writestring(f, chat->script);
		}

		loopi(SCR_MAX)
		{
			if(DEBUG_IO)
				conoutf(CON_DEBUG, "DEBUG: Writing script->scripts[%i] to file", i);
			writestring(f, scr->scripts[i]);
		}
	}

	void writeeffect(stream *f, effect *eff)
	{
		f->putlil(eff->flags);
		f->putlil(eff->decal);

		writestring(f, eff->mdl);
		f->putlil(eff->projpart);
		f->putlil(eff->projpartcol);
		f->putlil(eff->projpartsize);

		f->putlil(eff->trailpart);
		f->putlil(eff->trailpartcol);
		f->putlil(eff->trailpartfade);
		f->putlil(eff->trailpartgrav);
		f->putlil(eff->trailpartsize);

		f->putlil(eff->traillightradius);
		writevec(eff->traillightcol);

		f->putlil(eff->deathpart);
		f->putlil(eff->deathpartcol);
		f->putlil(eff->deathpartfade);
		f->putlil(eff->deathpartgrav);
		f->putlil(eff->deathpartsize);

		f->putlil(eff->deathlightflags);
		f->putlil(eff->deathlightfade);
		f->putlil(eff->deathlightradius);
		f->putlil(eff->deathlightinitradius);
		writevec(eff->deathlightcol);
		writevec(eff->deathlightinitcol);
	}

	void writestatus(stream *f, statusgroup *group)
	{
		f->putlil(group->effects.length());
		loopv(group->effects)
		{
			f->putlil(group->effects[i]->type);
			switch(group->effects[i]->type)
			{
				case STATUS_POLYMORPH:
				{
					status_polymorph *poly = (status_polymorph *) group->effects[i];
					writestring(f, poly->mdl);

					break;
				}
				case STATUS_LIGHT:
				{
					status_light *light = (status_light *) group->effects[i];
					writevec(light->colour);
					f->putlil(light->radius);

					break;
				}
				default:
				{
					status_generic *gen = (status_generic *) group->effects[i];
					f->putlil(gen->strength);
					break;
				}
			}
		}

		f->putchar(group->friendly);
		writestring(f, group->icon);
		writestring(f, group->name);
		writestring(f, group->description);
	}

	void writeitem(stream *f, item_base *item)
	{
		writestring(f, item->name);
		writestring(f, item->icon);
		writestring(f, item->description);
		writestring(f, item->mdl);

		f->putlil(item->script);
		f->putlil(item->type);
		f->putlil(item->flags);
		f->putlil(item->worth);
		f->putlil(item->weight);

		f->putlil(item->cursebase);

		f->putlil(item->uses.length());
		loopv(item->uses)
		{
			f->putlil(item->uses[i]->type);
			switch(item->uses[i]->type)
			{
				case USE_WEAPON:
				{
					use_weapon *use = (use_weapon *) item->uses[i];
					f->putlil(use->range);
					f->putlil(use->angle);
					f->putlil(use->lifetime);
					f->putlil(use->effect);
					f->putlil(use->cost);
					f->putlil(use->flags);
					f->putlil(use->element);
					f->putlil(use->ammo);
					f->putlil(use->target);
					f->putlil(use->radius);
					f->putlil(use->kickback);
					f->putlil(use->charge);
					f->putlil(use->basecharge);
					f->putlil(use->mincharge);
					f->putlil(use->maxcharge);
					f->putlil(use->elasticity);
					f->putlil(use->speed);
				}
				case USE_ARMOUR:
				{
					use_armour *use = (use_weapon *) item->uses[i];

					statreq tmp = use->reqs;
					lilswap(&tmp.attrs[0], STAT_MAX + SKILL_MAX);
					f->write(&tmp, sizeof(tmp));

					f->putlil(use->slots);
					f->putlil(use->skill);
				}
				case USE_CONSUME:
				{
					//note type is stored first
					f->putlil(item->uses[i]->script);
					f->putlil(item->uses[i]->status);
					f->putlil(item->uses[i]->duration);
					f->putlil(item->uses[i]->cooldown);
					f->putlil(item->uses[i]->mul);
					break;
				}
			}
		}
	}

	void writeammo(stream *f, ammotype *ammo)
	{
		writestring(f, ammo->name);
		f->putlil(ammo->items.length());
		loopv(ammo->items)
			f->putlil(ammo->items[i]);
	}

	void writechar(stream *f, char_base *base)
	{
		writestring(f, base->name);
		writestring(f, base->mdl);

		stats tmp = base->base;

		lilswap(&tmp.experience, 3);
		lilswap(&tmp.baseattrs[0], STAT_MAX * 2 + SKILL_MAX * 2);

		lilswap(&tmp.bonusthresh, ATTACK_MAX * 2);
		lilswap(&tmp.bonushealth, 6);
		lilswap(&tmp.bonushregen, 2);

		lilswap(&tmp.deltathresh, ATTACK_MAX * 2);
		lilswap(&tmp.deltahealth, 6);
		lilswap(&tmp.deltahregen, 2);

		f->write(&tmp, sizeof(stats));

		f->putlil(base->inventory.length());
		loopv(base->inventory)
		{
			f->putlil(base->inventory[i]->base);
			f->putlil(base->inventory[i]->quantity);
		}

		f->putlil(base->script);
		f->putlil(base->faction);
	}

	void writefaction(stream *f, faction *fact)
	{
		writestring(f, fact->name);
		writestring(f, fact->logo);

		f->putlil(fact->relations.length());
		loopv(fact->relations)
			f->putlil(fact->relations[i]);
	}

	void writeobject(stream *f, object_base *obj)
	{
		writestring(f, obj->mdl);
		writestring(f, obj->name);

		f->putlil(obj->flags);
		f->putlil(obj->weight);
		f->putlil(obj->script);

		f->putlil(obj->inventory.length());
		loopv(obj->inventory)
		{
			f->putlil(obj->inventory[i]->base);
			f->putlil(obj->inventory[i]->quantity);
		}
	}

	extern void writecutscenes(stream *f, int token);
	extern void writecamera(stream *f);

	void writeent(stream *f, rpgent *d)
	{
		f->putlil(d->type());

		switch(d->type())
		{
			case ENT_CHAR:
			{
				rpgchar *saving = (rpgchar *) d;

				writechar(f, saving);

				f->putlil(saving->equipped.length());
				loopv(saving->equipped)
				{
					f->putlil(saving->equipped[i].base);
					f->putlil(saving->equipped[i].use);
				}

				f->putlil(saving->health);
				f->putlil(saving->mana);
				f->putlil(saving->lastaction - lastmillis);

				f->putlil(saving->route.length());
				loopv(saving->route)
					f->putlil(saving->route[i]);

				break;
			}
			case ENT_ITEM:
			{
				rpgitem *saving = (rpgitem *) d;

				f->putlil(saving->base);
				f->putlil(saving->quantity);

				break;
			}
			case ENT_OBJECT:
			{
				rpgobject *saving = (rpgobject *) d;

				writeobject(f, saving);
				f->putlil(saving->lasttrigger - lastmillis);

				break;
			}
			default:
				conoutf(CON_ERROR, "ERROR: unsupported ent type %i, aborting", d->type());
				return;
		}

		f->putlil(d->seffects.length());
		loopv(d->seffects)
		{
			f->putlil(enttonum(d->seffects[i]->owner));
			f->putlil(d->seffects[i]->startmillis - lastmillis);
			f->putlil(d->seffects[i]->duration);
			f->putlil(d->seffects[i]->group);
			f->putlil(d->seffects[i]->mul);
		}

		f->putlil(d->eyeheight);
		writevec(d->o);
		writevec(d->vel);
		writevec(d->falling);

		f->putlil(d->yaw);
		f->putlil(d->pitch);
		f->putlil(d->roll);

		f->putlil(d->timeinair);

		f->putchar(d->state);
		f->putchar(d->editstate);

		f->putlil(d->lastjump - lastmillis);
	}

	void writemapscript(stream *f, mapscript *scr)
	{
		loopi(MSCR_MAX) writestring(f, scr->scripts[i]);
	}

	void writerecipe(stream *f, recipe *r)
	{
		loopi(3)
		{
			vector<invstack> &cont = (i ? (i == 1 ? r->catalysts : r->product) : r->ingredients);
			f->putlil(cont.length());

			loopvj(cont)
			{
				f->putlil(cont[j].base);
				f->putlil(cont[j].quantity);
			}
		}
	}

	void writemap(stream *f, mapinfo *saving)
	{
		lastmap = saving;
		int player = saving->objs.find(game::player1);
		if(player >= 0)
			saving->objs.removeobj(game::player1);

		writestring(f, saving->name);
		f->putchar(saving->loaded);

		f->putlil(saving->objs.length());
		f->putlil(saving->loadactions.length());
		f->putlil(saving->projs.length());
		f->putlil(saving->aeffects.length());
		f->putlil(saving->blips.length());

		loopv(saving->objs)
		{
			writeent(f, saving->objs[i]);
		}

		loopv(saving->loadactions)
		{
			f->putlil(saving->loadactions[i]->type());

			switch(saving->loadactions[i]->type())
			{
				case ACTION_TELEPORT:
				{
					action_teleport *act = (action_teleport *) saving->loadactions[i];
					f->putlil(enttonum(act->ent));
					f->putlil(act->dest);

					break;
				}
				case ACTION_SPAWN:
				{
					action_spawn *spw = (action_spawn *) saving->loadactions[i];
					f->putlil(spw->tag);
					f->putlil(spw->ent);
					f->putlil(spw->id);
					f->putlil(spw->amount);
					f->putlil(spw->qty);

					break;
				}
				case ACTION_SCRIPT:
					writestring(f, ((action_script *) saving->loadactions[i])->script);
					break;
			}
		}

		loopv(saving->projs)
		{
			projectile *p = saving->projs[i];
			f->putlil(enttonum(p->owner));
			f->putlil(p->item);
			f->putlil(p->use);
			f->putlil(p->ammo);
			f->putlil(p->ause);
			writevec(p->o); writevec(p->dir); writevec(p->emitpos);
			f->putlil(p->lastemit - lastmillis);
			f->putlil(p->gravity);
			f->putchar(p->deleted);

			f->putlil(p->pflags);
			f->putlil(p->time);
			f->putlil(p->dist);

			f->putlil(p->fx);
			f->putlil(p->radius);

			f->putlil(p->elasticity);
			f->putlil(p->charge);
		}

		loopv(saving->aeffects)
		{
			areaeffect *aeff = saving->aeffects[i];

			f->putlil(enttonum(aeff->owner));
			writevec(aeff->o);
			f->putlil(aeff->startmillis - lastmillis);
			f->putlil(aeff->lastemit - lastmillis);
			f->putlil(aeff->duration);
			f->putlil(aeff->group);
			f->putlil(aeff->radius);
			f->putlil(aeff->fx);
			f->putlil(aeff->mul);
		}

		loopv(saving->blips)
		{
			///FIXME finalize blip structure and write me
		}

		if(player >= 0)
			saving->objs.add(game::player1);
		lastmap = NULL;
	}

	void savegame(const char *name)
	{
		if(!game::mapdata || !game::curmap)
		{
			conoutf(CON_ERROR, "ERROR: No game in progress, can't save");
			return;
		}
		else if(!game::cansave())
		{
			conoutf("You may not save at this time");
			return;
		}

		defformatstring(file)("data/rpg/saves/%s.png", name);
		scaledscreenshot(file, 2, 256, 256);
		copystring(file + strlen(file) - 4, ".sgz");
		stream *f = opengzfile(file, "wb");

		if(DEBUG_IO)
			conoutf(CON_DEBUG, "DEBUG: succesffully opened %s with modes wb", file);

		if(!f)
		{
			conoutf("ERROR: Unable to write to: %s", file);
			return;
		}

		saveheader hdr;
		memcpy(hdr.magic, GAME_MAGICZ, 4);
		hdr.version = GAME_VERSION;

		lilswap(&hdr.version);
		f->write(&hdr, sizeof(saveheader));

		lastmap = game::curmap;
		writestring(f, game::curmap->name);
		writeent(f, game::player1);
		writecamera(f);

		loopv(game::scripts)
		{
			f->putchar(SAVE_SCRIPT);
			writescript(f, game::scripts[i]);
		}
		loopv(game::effects)
		{
			f->putchar(SAVE_EFFECT);
			writeeffect(f, game::effects[i]);
		}
		loopv(game::statuses)
		{
			f->putchar(SAVE_STATUS);
			writestatus(f, game::statuses[i]);
		}
		loopv(game::items)
		{
			f->putchar(SAVE_ITEM);
			writeitem(f, game::items[i]);
		}
		loopv(game::ammotypes)
		{
			f->putchar(SAVE_AMMO);
			writeammo(f, game::ammotypes[i]);
		}
		loopv(game::chars)
		{
			f->putchar(SAVE_CHAR);
			writechar(f, game::chars[i]);
		}
		loopv(game::factions)
		{
			f->putchar(SAVE_FACT);
			writefaction(f, game::factions[i]);
		}
		loopv(game::variables)
		{
			f->putchar(SAVE_VAR);
			f->putlil(game::variables[i]);
		}
		loopv(game::tips)
		{
			f->putchar(SAVE_TIP);
			writestring(f, game::tips[i]);
		}
		loopv(game::objects)
		{
			f->putchar(SAVE_OBJECT);
			writeobject(f, game::objects[i]);
		}
		loopv(game::mapscripts)
		{
			f->putchar(SAVE_MSCRIPT);
			writemapscript(f, game::mapscripts[i]);
		}
		loopv(game::recipes)
		{
			f->putchar(SAVE_RECIPE);
			writerecipe(f, game::recipes[i]);
		}
		writecutscenes(f, SAVE_CAMERA);
		enumerate(*game::mapdata, mapinfo, map,
			f->putchar(SAVE_MAP);
			writemap(f, &map);
		);


		delete f;
		conoutf("Game saved successfully to %s", file);
	}
	COMMAND(savegame, "s");
}
