/* InvariantListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import be.ugent.caagt.swirl.dnd.LocalTransferHandler;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.help.CSH;
import javax.swing.AbstractAction;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;

import org.grinvin.gui.dnd.HasAccessControl;
import org.grinvin.gui.dnd.InvariantDropHandler;
import org.grinvin.gui.dnd.InvariantFactoryDropHandler;
import org.grinvin.graphs.GraphURIType;
import org.grinvin.gui.WatermarkPainter;
import org.grinvin.help.InvariantListHelpManager;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.list.invariants.DefaultInvariantList;
import org.grinvin.list.HasSelectableValues;
import org.grinvin.list.invariants.InvariantList;
import org.grinvin.list.invariants.InvariantListModel;
import org.grinvin.gui.dnd.SelectionDragHandler;


/**
 * List component with entries of type {@link Invariant}. Provides drag support
 * for invariants. Dropping is not allowed.
 */
public class InvariantListComponent extends JList implements AcceptsInvariant, HasAccessControl, HasSelectableValues {
    
    // shared instance
    private static final ListCellRenderer RENDERER = new Renderer();
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.worksheet.resources";
    
    // shared watermark painter
    private static final WatermarkPainter WATERMARK_PAINTER = new WatermarkPainter(
            20,
            ResourceBundle.getBundle(BUNDLE_NAME).getString("InvariantList.emptytext"),
            new Color(245, 237, 237)
            );
    
    // shared transfer handler
    protected static final LocalTransferHandler TRANSFER_HANDLER;
    
    static {
        TRANSFER_HANDLER = new LocalTransferHandler();
        TRANSFER_HANDLER.addDropHandler(InvariantDropHandler.getInstance());
        TRANSFER_HANDLER.addDropHandler(InvariantFactoryDropHandler.getInstance());
        TRANSFER_HANDLER.setDragHandler(new SelectionDragHandler(Invariant.class));
    }
    
    public InvariantListComponent(InvariantListModel model) {
        this(model, LocalTransferHandler.COPY_OR_MOVE, LocalTransferHandler.COPY_OR_MOVE);
    }
    
    /**
     * Creates a new instance of InvariantListComponent
     */
    public InvariantListComponent(InvariantListModel model, int dragOperations, int dropOperations) {
        super(model);
        
        this.dragOperations = dragOperations;
        this.dropOperations = dropOperations;
        
        setTransferHandler(TRANSFER_HANDLER);
        setDragEnabled(true);
        setCellRenderer(RENDERER);
        setOpaque(false); // we paint our own background
        setPaintWatermark(true);
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                if(!isReadOnly())
                    deleteSelectedElements();
            }
        });
        
        CSH.addManager(new InvariantListHelpManager(this));
    }
    
    //
    private int dragOperations;
    
    //
    private int dropOperations;
    
    // paint the watermark?
    private boolean paintWatermark;
    
    /**
     * Configure whether to paint the watermark or not. If no watermark
     * is painted, the list background is completely transparent.
     * Note that this behavious is different from a {@link GraphCellListComponent}
     * which has no watermark.
     */
    public void setPaintWatermark(boolean paintWatermark) {
        this.paintWatermark = paintWatermark;
    }
    
    /**
     * Adds a watermark to the list.
     */
    @Override
    protected void paintComponent(Graphics g) {
        if(paintWatermark){
            g.setColor(getBackground());
            g.fillRect(0, 0, getWidth(), getHeight());
            WATERMARK_PAINTER.paint(this, g);
        }
        super.paintComponent(g);
    }
    
    //
    private InvariantList getInvariantList() {
        return (InvariantList)super.getModel();
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // gather all elements to be deleted
        InvariantList list = new DefaultInvariantList();
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                list.add(getInvariantList().get(i));
        
        // and remove them
        getInvariantList().removeAll(list);
        //TODO: make sure changed messages are sent
    }
    
    public int getDragOperations() {
        return dragOperations;
    }
    
    public int getDropOperations() {
        return dropOperations;
    }
    
    //
    private boolean isReadOnly = false;
    
    //
    public void setIsReadOnly(boolean isReadOnly) {
        this.isReadOnly = isReadOnly;
    }
    
    //
    public boolean isReadOnly() {
        return isReadOnly;
    }

    public Reason addInvariant(Invariant invariant) {
        return getInvariantList().add(invariant) ? Reason.SUCCESS : Reason.DUPLICATE;
    }

    public Reason acceptsInvariant(Invariant invariant) {
        return getInvariantList().contains(invariant) ? Reason.DUPLICATE : Reason.SUCCESS;
    }

    public AcceptsInvariant.Reason acceptsInvariant(Class<? extends InvariantValue> clazz) {
        return Reason.SUCCESS;
    }
    
    //
    private static class Renderer extends DefaultListCellRenderer {
        
        Renderer() {
            // avoid creation of access type
        }
        
        @Override
        public Component getListCellRendererComponent
                (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            setOpaque(isSelected);
            Color bg = getBackground();
            setBackground(new Color(bg.getRed(), bg.getGreen(), bg.getBlue(), 128));
            if (value instanceof Invariant) {
                Invariant invariant = (Invariant)value;
                setText(invariant.getName());
                setIcon(GraphURIType.INVARIANT.getSmallIcon());
                if (InvariantManager.getInstance().getInvariantComputerFor(invariant) == null)
                    setForeground(Color.RED);
            } else
                throw new IllegalArgumentException("ListCellRenderer can only be used with lists of Invariant");
            return this;
        }
    }
    
}
