/* GraphBundleView.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphs;

import java.util.Collection;
import java.util.Set;

import org.grinvin.gui.icons.GraphIconFactory;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.util.InternationalizedProperties;

/**
 * Presents a read-only view of a graph bundle. Bundles contain
 * information about a single (abstract) graph: its abstract representation,
 * zero or more embeddings, name, description, etc. A single graph bundle typically
 * corresponds to a single file on disk.
 *
 * <p>
 * This interface only provides read access to the graph. Use the interface
 * {@link GraphBundle} if you also need write access.
 * </p>
 */
public interface GraphBundleView {
    
    /**
     * Abstract graph representation for this graph bundle.
     */
    public GraphView getGraph();
    
    /**
     * Number of embeddings stored in this bundle.
     */
    public int getEmbeddingCount();
    
    /**
     * Return the default embedding for this graph, or {@code null} if no such
     * embedding is stored in the bundle. The peer of this embedding is the
     * graph returned by {@link #getGraph}.
     */
    public EmbeddingView getEmbedding();
    
    /**
     * Return the embedding with given index. Index 0 corresponds to the
     * default embedding. The peer of this embedding is the
     * graph returned by {@link #getGraph}.
     */
    public EmbeddingView getEmbedding(int index);
    
    /**
     * Number of annotations stored in this bundle.
     */
    public int getAnnotationCount();

    /**
     * Return the default annotation for this graph, or {@code null} if no such
     * annotation is stored in the bundle. The peer of this annotation is the
     * graph returned by {@link #getGraph}.
     */
    public AnnotationView getAnnotation();
    
    /**
     * Return the annotation with the given index. The peer of this annotation
     * is the graph returned by {@link #getGraph}.
     */
    public AnnotationView getAnnotation(int index);
    
    /**
     * Return the internationalized properties-object for this bundle. Currently
     * the following property keys should be supported:
     * <ul>
     * <li><tt>graph.name</tt> Name of this graph.</li>
     * <li><tt>graph.description</tt> Textual description of this graph.</li>
     * <li><tt>embedding_<i>i</i>.description</tt> Textual description of the
     * embedding with index <i>i</i>, with <i>i</i> &gt; 0.</li>
     * </ul>
     */
    public InternationalizedProperties getProperties();
    
    /**
     * Return the name of this graph.
     * @return  {@code getProperties().getProperty("graph.name")}
     */
    public String getName();
    
    /**
     * Return a textual description of this graph.
     * @return  {@code getProperties().getProperty("graph.description")}
     */
    public String getDescription();
    
    /**
     * Return the value of the graph for the given invariant.
     * @param invariant the URI of the invariant
     * @return The value of the computed invariant or null if unavailable
     */
    public InvariantValue getInvariantValue(Invariant invariant);
    
    /**
     * Return the cached value of the graph for the given invariant, if available.
     * @param invariant the URI of the invariant
     * @return The value of the computed invariant or null if unavailable
     */
    
    public InvariantValue getCachedInvariantValue(Invariant invariant);
    
    /**
     * Return the collection of cached values of the invariants of the graph.
     * @return The current collection of invariant values
     */
    public Collection<InvariantValue> getInvariantValues();
    
    /**
     * Return the set of invariants currently registered with this bundle.
     */
    public Set<Invariant> getInvariants();
    
    /**
     * Return the preferred {@link GraphIconFactory} to be used when
     * displaying this graph as an icon.
     */
    public GraphIconFactory getGraphIconFactory();
    
    /**
     * Return a (cached) boolean adjacency matrix for the corresponding
     * graph. Row and column
     * indices correspond to vertex indices. Entries are {@code true}
     * when the corresponding vertices are adjacent (and different) and {@code false}
     * otherwise.<p>
     * <b>Important:</b> The return value should be considered read only and is valid
     * only as long as the graph in the bundle is not changed.
     */
    public boolean[][] booleanAdjacencyMatrix();
    
    
    /**
     * Returns a (cached) adjacency list representation for the corresponding graph.
     * The resulting array
     * contains an array of neighbour indices for every vertex. The length of each array is exactly
     * the valency of the corresponding vertex.<p>
     * <b>Important:</b> The return value should be considered read only and is valid
     * only as long as the graph in the bundle is not changed.
     */
    public int[][] adjacencyList();
    
    /**
     * Returns a (cached) distance matrix for the corresponding graph. Row and column
     * indices correspond to vertex indices. Entries contains the distance
     * between corresponding vertices or 0 when vertices are equal or
     * belong to different components of the graph.
     */
    public int[][] distanceMatrix();

    /**
     * Returns a (cached) eccentricity list for the corresponding graph.
     * The resulting array contains the eccentricity for every vertex. <p>
     * <b>Important:</b> The return value should be considered read only and is valid
     * only as long as the graph in the bundle is not changed.
     */
    public int[] eccentricityList();
    
}
