package org.inria.bmajwatcher.server.data;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.inria.biomaj.exe.bank.BankFactory;
import org.inria.biomaj.exe.bank.BiomajBank;
import org.inria.biomaj.session.bank.Bank;
import org.inria.biomaj.session.bank.BiomajSQLQuerier;
import org.inria.biomaj.session.bank.Configuration;
import org.inria.biomaj.session.bank.ProductionDirectory;
import org.inria.biomaj.session.bank.Session;
import org.inria.biomaj.session.process.BiomajProcess;
import org.inria.biomaj.session.process.MetaProcess;
import org.inria.biomaj.singleton.BiomajInformation;
import org.inria.biomaj.sql.SQLConnection;
import org.inria.biomaj.sql.SQLConnectionFactory;
import org.inria.biomaj.utils.BiomajConst;
import org.inria.biomaj.utils.BiomajException;
import org.inria.biomaj.utils.BiomajUtils;
import org.inria.bmajwatcher.server.JobSerializer;


public class BankRetriever {
	
	public static final String RELEASE = "_release";
	public static final String SESSION_COUNT = "_sessionCount";
	public static final String LAST_SESSION = "_lastSession";
	public static final String DURATION = "_duration";
	public static final String PROD_DIR = "_prodDir";
	public static final String PROD_DIRS = "_prodDirs";
	public static final String FILE_DOWNLOAD_COUNT = "_downloadCount";
	public static final String BANDWIDTH = "_bandwidth";
	public static final String DOWNLOAD_SIZE = "_downloadSize";
	public static final String BANK_SIZE = "_bankSize";
	public static final String LAST_LOG = "_logFile";
	public static final String PREPROCESSES = "_preProcesses";
	public static final String POSTPROCESSES = "_postProcesses";
	public static final String URL = "_url";
	
	
	private static BankRetriever bankRetriever = new BankRetriever();
	
	private int error = 0;
	private int total = 0;
	private int updating = 0;
	private Map<Integer, List<String>> records;
	
	private final Lock lock = new ReentrantLock();
	private static Logger log = Logger.getLogger(BankRetriever.class);
	
	public static BankRetriever getInstance() {
		return bankRetriever;
	}
	
	
	
	
	public String getBankInfo(String bankName, String parentId) {
		int id = 1;
		StringBuilder sb = new StringBuilder(1000);
		sb.append("<response><status>0</status><data>");
		String toRetrieve = "0";
		
		if (parentId.equals("null")) { // Generate the whole structure
			
			records = new HashMap<Integer, List<String>>();
			int bankId = BiomajSQLQuerier.getBankId(bankName);
			if (bankId >= 0) {
				/*
				 * Browse configurations
				 */
				List<Map<String, String>> configs = BiomajSQLQuerier.getConfigurations(bankId);
				List<String> confRecords = new ArrayList<String>();
				records.put(0, confRecords);
				for (Map<String, String> config : configs) {
					List<String> confDetails = new ArrayList<String>();
					
					String tmp = "";
					tmp += "<record id=\"" + id++ + "\" ";
					tmp += "parent=\"" + 0 + "\" ";
					tmp += "value=\"Configuration [" + config.get(BiomajSQLQuerier.DATE) + "]\" />";
					
					confRecords.add(tmp);
					int currentConfig = id - 1;
					
//					tmp = "";
//					tmp += "<record id=\"" + id++ + "\" ";
//					tmp += "parent=\"" + currentConfig + "\" ";
//					tmp += "value=\"Date : " + config.get(BiomajSQLQuerier.DATE) + "\" />";
//					confDetails.add(tmp);
					
					tmp = "";
					tmp += "<record id=\"" + id++ + "\" ";
					tmp += "parent=\"" + currentConfig + "\" ";
					tmp += "value=\"File : " + config.get(BiomajSQLQuerier.FILE) + "\" />";
					confDetails.add(tmp);
					
					records.put(currentConfig, confDetails);
					
					/*
					 * Browse updates
					 */
					List<Map<String, String>> updates = BiomajSQLQuerier.getConfigUpdates(Long.valueOf(config.get(BiomajSQLQuerier.CONFIGURATION_ID)));
					for (Map<String, String> update : updates) {
						tmp = "";
						tmp += "<record id=\"" + id++ + "\" ";
						tmp += "parent=\"" + currentConfig + "\" ";
						tmp += "value=\"Update [Release = " + update.get(BiomajSQLQuerier.UPDATE_RELEASE) + "]\" />";
						confDetails.add(tmp);
						
						int currentUpdate = id - 1;
						List<String> updateRecords = new ArrayList<String>();
						records.put(currentUpdate, updateRecords);
						
						tmp = "";
						tmp += "<record id=\"" + id++ + "\" ";
						tmp += "parent=\"" + currentUpdate + "\" ";
						tmp += "value=\"Updated ? " + update.get(BiomajSQLQuerier.UPDATED) + "\" />";
						updateRecords.add(tmp);
						
						tmp = "";
						tmp += "<record id=\"" + id++ + "\" ";
						tmp += "parent=\"" + currentUpdate + "\" ";
						tmp += "value=\"Start : " + update.get(BiomajSQLQuerier.UPDATE_START) + "\" />";
						updateRecords.add(tmp);
						
						tmp = "";
						tmp += "<record id=\"" + id++ + "\" ";
						tmp += "parent=\"" + currentUpdate + "\" ";
						tmp += "value=\"End : " + update.get(BiomajSQLQuerier.UPDATE_END) + "\" />";
						updateRecords.add(tmp);
						
						tmp += "<record id=\"" + id++ + "\" ";
						tmp += "parent=\"" + currentUpdate + "\" ";
						tmp += "value=\"Production directory : " + update.get(BiomajSQLQuerier.PRODUCTION_DIR_PATH) + " (" + update.get(BiomajSQLQuerier.SIZE_RELEASE) + ")\" />";
						updateRecords.add(tmp);
						
						/*
						 * Browse sessions
						 */
						List<Map<String, String>> sessions = BiomajSQLQuerier.getUpdateSessions(Integer.valueOf(update.get(BiomajSQLQuerier.UPDATE_ID)));
						for (Map<String, String> session : sessions) {
							tmp = "";
							tmp += "<record id=\"" + id++ + "\" ";
							tmp += "parent=\"" + currentUpdate + "\" ";
							tmp += "value=\"Session [Status = " + session.get(BiomajSQLQuerier.SESSION_STATUS) + "]\" />";
							updateRecords.add(tmp);
							
							int currentSession = id - 1;
							List<String> sessionRecords = new ArrayList<String>();
							records.put(currentSession, sessionRecords);
							
							tmp = "";
							tmp += "<record id=\"" + id++ + "\" ";
							tmp += "parent=\"" + currentSession + "\" ";
							tmp += "value=\"Start : " + session.get(BiomajSQLQuerier.SESSION_START) + "\" />";
							sessionRecords.add(tmp);
							
							tmp = "";
							tmp += "<record id=\"" + id++ + "\" ";
							tmp += "parent=\"" + currentSession + "\" ";
							tmp += "value=\"End : " + session.get(BiomajSQLQuerier.SESSION_END) + "\" />";
							sessionRecords.add(tmp);
							
							tmp = "";
							tmp += "<record id=\"" + id++ + "\" ";
							tmp += "parent=\"" + currentSession + "\" ";
							tmp += "value=\"Log : " + session.get(BiomajSQLQuerier.LOG_FILE) + "\" />";
							sessionRecords.add(tmp);
							
							/*
							 * Browse sessionTasks
							 */
							List<Map<String, String>> tasks = BiomajSQLQuerier.getSessionTasks(Long.valueOf(session.get(BiomajSQLQuerier.SESSION_ID)));
							for (Map<String, String> task : tasks) {
								String type = task.get(BiomajSQLQuerier.TASK_TYPE);
								tmp = "";
								tmp += "<record id=\"" + id++ + "\" ";
								tmp += "parent=\"" + currentSession + "\" ";
								tmp += "value=\"Task : " + type + " [Status = " + task.get(BiomajSQLQuerier.TASK_STATUS) + "]\" />";
								sessionRecords.add(tmp);
								
								int currentTask = id - 1;
								List<String> taskRecords = new ArrayList<String>();
								records.put(currentTask, taskRecords);
								
								tmp = "";
								tmp += "<record id=\"" + id++ + "\" ";
								tmp += "parent=\"" + currentTask + "\" ";
								tmp += "value=\"Start : " + task.get(BiomajSQLQuerier.TASK_START) + "\" />";
								taskRecords.add(tmp);
								
								tmp = "";
								tmp += "<record id=\"" + id++ + "\" ";
								tmp += "parent=\"" + currentTask + "\" ";
								tmp += "value=\"End : " + task.get(BiomajSQLQuerier.TASK_END) + "\" />";
								taskRecords.add(tmp);
								
								
								if (type.equalsIgnoreCase("release")) {
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Value : " + task.get(BiomajSQLQuerier.VALUE) + "\" />";
									taskRecords.add(tmp);
									
								} else if (type.equalsIgnoreCase("check")) {
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Files to extract : " + task.get(BiomajSQLQuerier.NB_EXTRACT) + "\" />";
									taskRecords.add(tmp);
									
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Local online files : " + task.get(BiomajSQLQuerier.NB_LOCAL_ONLINE_FILES) + "\" />";
									taskRecords.add(tmp);
									
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Local offline files : " + task.get(BiomajSQLQuerier.NB_LOCAL_OFFLINE_FILES) + "\" />";
									taskRecords.add(tmp);
									
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Files to download : " + task.get(BiomajSQLQuerier.NB_DOWNLOADED_FILES) + "\" />";
									taskRecords.add(tmp);
									
								} else if (type.equalsIgnoreCase("download")) {
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Download speed : " + task.get(BiomajSQLQuerier.BANDWIDTH) + " MB/s\" />";
									taskRecords.add(tmp);
									
								} else if (type.equalsIgnoreCase("addLocalFiles")) {
									
								} else if (type.equalsIgnoreCase("makeRelease")) {
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Moved files : " + task.get(BiomajSQLQuerier.NB_FILES_MOVED) + "\" />";
									taskRecords.add(tmp);
									
									tmp = "";
									tmp += "<record id=\"" + id++ + "\" ";
									tmp += "parent=\"" + currentTask + "\" ";
									tmp += "value=\"Copied files : " + task.get(BiomajSQLQuerier.NB_FILES_COPIED) + "\" />";
									taskRecords.add(tmp);
									
								} else if (type.equalsIgnoreCase("deployment")) {
									
								} else if (type.equalsIgnoreCase("preprocess") || type.equalsIgnoreCase("postprocess")) {
									
									/*
									 * Browse metaprocesses
									 */
									
									List<Map<String, String>> metas = BiomajSQLQuerier.getTaskMetaprocesses(Integer.valueOf(task.get(BiomajSQLQuerier.TASK_ID)));
									for (Map<String, String> meta : metas) {
										tmp = "";
										tmp += "<record id=\"" + id++ + "\" ";
										tmp += "parent=\"" + currentTask + "\" ";
										tmp += "value=\"Metaprocess [" + meta.get(BiomajSQLQuerier.META_NAME) + "]" +
												" [" + meta.get(BiomajSQLQuerier.META_STATUS) + "]\" />";
										taskRecords.add(tmp);
										
										int currentMeta = id - 1;
										List<String> metaRecords = new ArrayList<String>();
										records.put(currentMeta, metaRecords);
										
										tmp = "";
										tmp += "<record id=\"" + id++ + "\" ";
										tmp += "parent=\"" + currentMeta + "\" ";
										tmp += "value=\"Start : " + meta.get(BiomajSQLQuerier.META_START) + "\" />";
										metaRecords.add(tmp);
										
										tmp = "";
										tmp += "<record id=\"" + id++ + "\" ";
										tmp += "parent=\"" + currentMeta + "\" ";
										tmp += "value=\"End : " + meta.get(BiomajSQLQuerier.META_END) + "\" />";
										metaRecords.add(tmp);
										
										List<Map<String, String>> processes = BiomajSQLQuerier.getMetaprocessProcesses(meta.get(BiomajSQLQuerier.META_ID));
										for (Map<String, String> process : processes) {
											tmp = "";
											tmp += "<record id=\"" + id++ + "\" ";
											tmp += "parent=\"" + currentMeta + "\" ";
											tmp += "value=\"Process [" + process.get(BiomajSQLQuerier.PROC_NAME) + "]\" />";
											metaRecords.add(tmp);
											
											int currentPs = id - 1;
											List<String> psRecords = new ArrayList<String>();
											records.put(currentPs, psRecords);
											
											tmp = "";
											tmp += "<record id=\"" + id++ + "\" ";
											tmp += "parent=\"" + currentPs + "\" ";
											tmp += "value=\"Exe : " + process.get(BiomajSQLQuerier.PROC_EXE) + "\" />";
											psRecords.add(tmp);
											
											tmp = "";
											tmp += "<record id=\"" + id++ + "\" ";
											tmp += "parent=\"" + currentPs + "\" ";
											tmp += "value=\"Args : " + process.get(BiomajSQLQuerier.PROC_ARGS) + "\" />";
											psRecords.add(tmp);
											
											tmp = "";
											tmp += "<record id=\"" + id++ + "\" ";
											tmp += "parent=\"" + currentPs + "\" ";
											tmp += "value=\"Start : " + process.get(BiomajSQLQuerier.PROC_START) + "\" />";
											psRecords.add(tmp);
											
											tmp = "";
											tmp += "<record id=\"" + id++ + "\" ";
											tmp += "parent=\"" + currentPs + "\" ";
											tmp += "value=\"End : " + process.get(BiomajSQLQuerier.PROC_END) + "\" />";
											psRecords.add(tmp);
										}
									}
								}
							}
						}
					}
				}
				
			}
		} else
			toRetrieve = parentId;
		
		if (records.containsKey(Integer.valueOf(toRetrieve))) {
			for (String s : records.get(Integer.valueOf(toRetrieve))) {
				sb.append(s);
			}
		}
		sb.append("</data></response>");
		
		return sb.toString();
	}
	
	/**
	 * Returns a string that contains the bank list in XML format.
	 * 
	 * @param biomajRoot biomaj root directory
	 * @return
	 */
	public String getBankList(String biomajRoot) {
		lock.lock();
		
		int _error = 0;
		int _total = 0;
		int _updating = 0;

		String query = "SELECT session,path,ref_idbank,name FROM productionDirectory,bank WHERE state='available' and ref_idbank=idbank " +
				"ORDER BY ref_idbank";
		
		SQLConnection connection = SQLConnectionFactory.getConnection();
		Statement stat = connection.getStatement();
		ResultSet rs = connection.executeQuery(query, stat);
		Map<String, Map<String, String>> data = new HashMap<String, Map<String,String>>();
		Map<String, Map<Long, String>> bankPaths = new HashMap<String, Map<Long, String>>();
		try {
			while (rs.next()) {
				Map<String, String> bank = new HashMap<String, String>(); // Current bank
				String bankName = rs.getString(BiomajSQLQuerier.BANK_NAME);
				bank.put(BiomajSQLQuerier.BANK_NAME, bankName);
				bank.put(BiomajSQLQuerier.BANK_ID, String.valueOf(rs.getInt(3)));
				
				if (!bankPaths.containsKey(bankName)) {
					// Get status and session end
					long confId = BiomajSQLQuerier.getLatestConfigurationWithSession(rs.getInt(3), false);
					query = "SELECT status,endTime FROM session where ref_idupdateBank IN " +
							"(SELECT idupdateBank FROM updateBank WHERE ref_idconfiguration=" + confId + ") and endTime=" +
							"(SELECT max(endTime) FROM session where ref_idupdateBank IN " +
							"(SELECT idupdateBank FROM updateBank WHERE ref_idconfiguration=" + confId + "))";
					Statement stat2 = connection.getStatement();
					ResultSet rs2 = connection.executeQuery(query, stat2);
					rs2.next();
					bank.put(BiomajSQLQuerier.SESSION_STATUS, rs2.getBoolean(BiomajSQLQuerier.SESSION_STATUS) ? "OK" : "KO");
					bank.put(BiomajSQLQuerier.SESSION_END, BiomajUtils.dateToString(
							new Date(rs2.getTimestamp(BiomajSQLQuerier.SESSION_END).getTime()), Locale.US));
					SQLConnectionFactory.closeConnection(stat2);
					
					// Stores the path for later processing
					Map<Long, String> pd = new HashMap<Long, String>();
					pd.put(rs.getLong(BiomajSQLQuerier.DIR_SESSION), rs.getString(BiomajSQLQuerier.DIR_PATH));
					bankPaths.put(bankName, pd);
					
					data.put(bankName, bank);
				} else {
					bankPaths.get(bankName).put(rs.getLong(BiomajSQLQuerier.DIR_SESSION), rs.getString(BiomajSQLQuerier.DIR_PATH));
				}
			}
			SQLConnectionFactory.closeConnection(stat);
		} catch (SQLException ex) {
			SQLConnectionFactory.closeConnection(stat);
			log.error(ex);
			return "";
		}
		
		StringBuilder xmlOutput = new StringBuilder();
		xmlOutput.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
				+ "<response>" +
						"<status>0</status>" +
						"<startRow>0</startRow>" +
						"<endRow>" + data.size() + "</endRow>" +
						"<totalRows>" + data.size() + "</totalRows>" +
						"<data>\n");
		
		String tmpDir = "";
		try {
			tmpDir = BiomajInformation.getInstance().getProperty(BiomajInformation.TMPDIR);
			if (!tmpDir.startsWith("/") && biomajRoot != null)
				tmpDir = biomajRoot + "/" + tmpDir.substring(1);
			else if (biomajRoot == null)
				log.warn("Could not find tmp directory");
		} catch (BiomajException e) {
			tmpDir = biomajRoot + "/tmp";
			log.error(e);
		}
		
		Map<String, String> releases = getBankReleases(bankPaths);
		
		for (String bankName : data.keySet()) {
			_total++;
			String formats = "";
			String dbType = "";
			try {
				BiomajBank b = new BankFactory().createBank(bankName, false);
				Properties props = b.getPropertiesFromBankFile();
				if (props.containsKey(BiomajConst.dbFormatsProperty))
					formats = props.getProperty(BiomajConst.dbFormatsProperty);
				if (props.containsKey(BiomajConst.typeProperty))
					dbType = props.getProperty(BiomajConst.typeProperty);
			} catch (BiomajException e1) {
				e1.printStackTrace();
			}
			xmlOutput.append("<record id=\"" + data.get(bankName).get(BiomajSQLQuerier.BANK_ID) + "\" " +
					"name=\"" + bankName + "\" " +
					"formats=\"" + formats + "\" " +
					"type=\"" + dbType + "\" status=\"");
			
			if (new File(tmpDir + "/" + bankName + ".lock").exists()) {
				xmlOutput.append("Updating\" ");
				_updating++;
			} else {
				String s = data.get(bankName).get(BiomajSQLQuerier.SESSION_STATUS);
				xmlOutput.append(s + "\" ");
				if (s.equalsIgnoreCase("KO"))
					_error++;
			}
			
			String image = "sched.png";
			if (isScheduled(data.get(bankName).get(BiomajSQLQuerier.BANK_NAME)))
				xmlOutput.append("isSched=\"" + image + "\" ");
			else
				xmlOutput.append("isSched=\"\" ");
			
			xmlOutput.append("release=\"" + releases.get(bankName) + "\" session=\"" + data.get(bankName).get(BiomajSQLQuerier.SESSION_END) + "\" />");
		}
		xmlOutput.append("</data></response>");
		
		error = _error;
		total = _total;
		updating = _updating;
		
		lock.unlock();
		
		return xmlOutput.toString();
		
	}
	
	
	/**
	 * Returns the for each bank its latest release from the provided production directories.
	 * 
	 * @param banks
	 * @return
	 */
	private Map<String, String> getBankReleases(Map<String, Map<Long, String>> banks) {
		
		Map<String, String> res = new HashMap<String, String>();
		SQLConnection connection = SQLConnectionFactory.getConnection();
		Statement stat = connection.getStatement();
		
		for (String bankName : banks.keySet()) {
			
			long max = -1;
			// Get latest session
			for (Long session : banks.get(bankName).keySet()) {
				max = max > session ? max : session;
			}
			
			String query = "SELECT updateRelease FROM updateBank WHERE idupdateBank=" +
					"(SELECT ref_idupdateBank FROM session where idsession=" + max + ");";
			ResultSet rs = connection.executeQuery(query, stat);
			String release = "???";
			try {
				if (rs.next())
					release = rs.getString(1);
			} catch (SQLException e) {
				log.error(e);
			}
			
			res.put(bankName, release);
		}
		SQLConnectionFactory.closeConnection(stat);
		
		return res;
	}

	
	public Map<String, String> getBankDetail(String bankName) throws BiomajException {
		
		Map<String, String> info = BiomajSQLQuerier.getBankInfo(bankName);
		
		
		Configuration config = new Configuration(info);
		info.put(URL, config.getProtocol() + "://" + config.getUrl() + "/"+config.getRemoteDirectory());
		
		List<ProductionDirectory> lPd = BiomajSQLQuerier.getAvailableProductionDirectories(bankName);

		if (lPd.size() == 0)
			return info;

		long goodSession = 0 ;
		Date t = null;
		for (ProductionDirectory pd : lPd) {
			if (t == null)
				goodSession = pd.getSession();
			else if (t.getTime()<pd.getCreationDate().getTime())
				goodSession = pd.getSession();
		}

		
		Bank b = new Bank();
		
		Map<String, String> update = BiomajSQLQuerier.getUpdateFromId(goodSession);
		if (update != null) {
			b.setLoadMessages(false);
			b.fill(update, true);
			b.setConfig(config);
			fillBank(b, info);
		}
		
		/*
		// Future release 
		b = new Bank();
		update = BiomajSQLQuerier.getLatestUpdate(bankName, true);
		if (update != null) {
			b.fillSql(update, true);
			if (b.getEnd() == null) {
				b.setConfig(config);
				printBank(b, info);
			}
		}*/
		
		if (lPd.size() > 0) {
			StringBuilder strb = new StringBuilder();
			Collections.sort(lPd);
			strb.append("<ul>");
			for (ProductionDirectory pd : lPd) {
				strb.append("<li>" + BiomajUtils.dateToString(pd.getCreationDate(), Locale.getDefault())+ " " + pd.getPath() + "</li>");
			}
			strb.append("</ul>");
			info.put(PROD_DIRS, strb.toString());
		}
		
		return info;
	}
	
	
	private void fillBank(Bank b, Map<String, String> info) throws BiomajException {
		
		info.put(RELEASE, b.getWorkflowInfoRelease());
		info.put(SESSION_COUNT, Integer.toString(b.getListOldSession().size()));
		if (b.getEnd()!=null) {
			info.put(LAST_SESSION, BiomajUtils.dateToString(b.getEnd(), Locale.getDefault()));
			info.put(DURATION, BiomajUtils.timeToString(b.getEnd().getTime()-b.getStart().getTime()));
		}
		if (b.getWorkflowInfoProductionDir() != null)
			info.put(PROD_DIR, b.getWorkflowInfoProductionDir());

		info.put(FILE_DOWNLOAD_COUNT, String.valueOf(b.getNbFilesDownloaded()));
		info.put(BANDWIDTH, Float.toString(b.getBandWidth()));
		info.put(DOWNLOAD_SIZE, BiomajUtils.sizeToString(b.getWorkflowInfoSizeDownload()));
		info.put(BANK_SIZE, BiomajUtils.sizeToString(b.getWorkflowInfoSizeRelease()));
		
		if (b.getListOldSession().size() > 0)
			info.put(LAST_LOG, b.getListOldSession().get(b.getListOldSession().size()-1).getLogfile());

		Collection<MetaProcess> lMpPre = b.getAvailableMetaProcessSql(Session.PREPROCESS);
		StringBuilder strb = new StringBuilder();
		for (MetaProcess mpPre : lMpPre) {
			strb.append("BLOCK="+ mpPre.getBlock() + ";META=" + mpPre.getName() +";");
			for (BiomajProcess bp : mpPre.getListProcess())
				strb.append("PS=" + bp.getKeyName() + "_:_" + bp.getExe() + ";");
		}
		info.put(PREPROCESSES, strb.toString());
		
		Collection<MetaProcess> lMp = b.getAvailableMetaProcessSql(Session.POSTPROCESS);
		strb = new StringBuilder();
		for (MetaProcess mp : lMp) {
			strb.append("BLOCK="+ mp.getBlock() + ";META=" + mp.getName() +";");
			for (BiomajProcess bp : mp.getListProcess())
				strb.append("PS=" + bp.getKeyName() + "_:_" + bp.getExe() + ";");
		}
		info.put(POSTPROCESSES, strb.toString());
	}
	
	/**
	 * Checks in the jobs.xml file if the specified bank is scheduled.
	 * 
	 * @param bankName
	 * @return
	 */
	private boolean isScheduled(String bankName) {
		// We should use some kind of xml parser to properly search for the bank
		// but it is so much easier this way (and faster ?)
		
		try {
			BufferedReader br = new BufferedReader(new FileReader(JobSerializer.getJobsFileName()));
			String line = null;
			try {
				while ((line = br.readLine()) != null) {
					if (line.contains("name=\"" + bankName + "\""))
						return true;
				}

				br.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		
		return false;
	}
	
	public int getError() {
		lock.lock(); // wait for error count to be processed
		int err = error;
		lock.unlock();
		return err;
		
	}
	
	public int getUpdating() {
		lock.lock(); // wait for updating count to be processed
		int upd = updating;
		lock.unlock();
		return upd;
	}
	
	public int getTotal() {
		lock.lock(); // wait for total count to be processed
		int tot = total;
		lock.unlock();
		return tot;
	}
	
}
