/*
 * Convert X-Y-Z coordinate information in '<prefix>%c%c%c' format into NID
 */
#include "basil_mysql.h"
#include <limits.h>

#define HOSTLIST_BASE 36

static void hostlist_parse_int_to_array(int in, int *out, int dims)
{
	for ( ; --dims >= 0; in /= HOSTLIST_BASE)
		out[dims] = in % HOSTLIST_BASE;
}

int main(int ac, char **av)
{
	const char query[] =	"SELECT processor_id, "
				"	processor_type, "
				"	processor_status, "
				"	alloc_mode "
				"FROM  processor "
				"WHERE x_coord = ? "
				"AND   y_coord = ? "
				"AND   z_coord = ? ";

	enum query_params {
			PAR_X,
			PAR_Y,
			PAR_Z,
			PARAM_COUNT	/* sentinel */
	};
	MYSQL_BIND	bind_parm[PARAM_COUNT];
	int		params[PARAM_COUNT];

	enum query_columns {
			COL_NID,
			COL_TYPE,
			COL_STATE,
			COL_ALLOC,
			COLUMN_COUNT	/* sentinel */
	};
	MYSQL_BIND	bind_cols[COLUMN_COUNT];
	unsigned int	node_id;
	char 		c_data[COL_ALLOC - COL_TYPE + 1][BASIL_STRING_SIZE];
	my_bool		is_null[COLUMN_COUNT];
	my_bool		error[COLUMN_COUNT];

	MYSQL		*handle;
	MYSQL_STMT	*stmt;

	int		rc = -1, i;

	handle = cray_connect_sdb();
	if (handle == NULL)
		errx(1, "can not connect to XTAdmin database on the SDB");

	memset(bind_parm, 0, sizeof(bind_parm));
	for (i = 0; i < PARAM_COUNT; i ++) {
		bind_parm[i].buffer_type = MYSQL_TYPE_LONG;
		bind_parm[i].buffer	 = (char *)&params[i];
	}

	memset(bind_cols, 0, sizeof(bind_cols));
	for (i = 0; i < COLUMN_COUNT; i ++) {
		bind_cols[i].is_null = &is_null[i];
		bind_cols[i].error   = &error[i];
	}
	bind_cols[COL_NID].buffer_type	= MYSQL_TYPE_LONG;
	bind_parm[COL_NID].is_unsigned	= true;
	bind_cols[COL_NID].buffer	= (char *)&node_id;

	for (i = COL_TYPE; i <= COL_ALLOC; i++) {
		bind_cols[i].buffer_type   = MYSQL_TYPE_STRING;
		bind_cols[i].buffer	   = c_data[i - COL_TYPE];
		bind_cols[i].buffer_length = BASIL_STRING_SIZE;
	}
	stmt = prepare_stmt(handle, query, bind_parm, PARAM_COUNT,
					   bind_cols, COLUMN_COUNT);
	if (stmt == NULL)
		goto err;

	for (i = 0; i < ac - 1; i++) {
		char *p = *++av, *q;

		if (sscanf(p, "%d %d %d", params, params + 1, params + 2) == PARAM_COUNT) {
			/* Space-separated X/Y/Z digits */
		} else if (strlen(p) == PARAM_COUNT) {
			long val = strtol(p, &q, HOSTLIST_BASE);

			if (val < 0 || val == LONG_MIN || val == LONG_MAX || *q) {
				warnx("'%s' has wrong XYZ encoding", *av);
				continue;
			}
			hostlist_parse_int_to_array(val, params, PARAM_COUNT);
		} else {
			warnx("'%s' neither in 'X Y Z' nor in %dD base-%d format.",
			      *av, PARAM_COUNT, HOSTLIST_BASE);
			continue;
		}

		if (exec_stmt(stmt, query, bind_cols, COLUMN_COUNT) < 0)
			goto err_close;

		if (i == 0)
			printf("  X   Y   Z    NID         TYPE    STATUS        MODE\n");

		printf("%3d %3d %3d", params[0], params[1], params[2]);

		if (mysql_stmt_fetch(stmt) == 0) {
			printf(" %6u   %10s %9s %11s\n", node_id,
				c_data[0], c_data[1], c_data[2]);
		} else {
			printf("      ?\n");
		}
	}

	rc = 0;
err_close:
	if (mysql_stmt_close(stmt))
		warnx("error closing statement: %s",
			mysql_stmt_error(stmt));
err:
	mysql_close(handle);

	return rc ? EXIT_FAILURE : EXIT_SUCCESS;
}
