// FILTERDLL.CPP
// =============
//
//   WU_FTP log file format DLL for use with Serv-U v2.2a and higher.
//   Enables Serv-U users to create log files in WU_FTP file format.
//   The logging settings are read from a file named WULOG.INI.
//   Syntax for that file is (example):
//
//      ********** WULOG.INI ********
//
//              [LOG]
//              FileName=c:\serv-u\wulog.txt
//
//      *****************************
//
//             Author: Rob Beckers
//               Date: 11-JUL-97
//      Last revision: 11-JUL-97
//        Revision nr: 1
//
//   Revision history:
//

#include <windows.h>
#include <stdio.h>
#include <iostream>                     // needed for STL strings
#include <string>                       // use standard template lib strings, not Borland's
#include <string.h>
#include <time.h>
#pragma hdrstop                         // to make Borland compiler stop including pre-compiled header files
#include "list.h"                       // used here for simple list management
#include "wulog.h"

// Global variable definitions
LSTUINFO UserList;                      // list of current user info structures
char LogFile[MAXPATH];                  // full path/name of file to log to

int operator==(const RUserInfoStr& ArgA,const RUserInfoStr& ArgB)
// ******
// Comparison operator for user info structures.
// Is used by the simple list routines.
// ******
{
  // equality means the session ID's are the same
  return(ArgA.SessionID==ArgB.SessionID);
}

void LoadIniFile(HINSTANCE hInstance)
// ******
// Load .ini file variables
// ******
{
  char IniFile[MAXPATH];                // full path of .ini file

  // only initialize once
  static bool Once=FALSE;
  if (Once) return;                     // already initialized
  Once=true;                            // this is first time, remember 

  // construct full path of .ini file
  GetModuleFileName(hInstance,IniFile,MAXPATH);
  char* pEnd=strrchr(IniFile,'\\');	// strip .dll file name
  if (pEnd) pEnd[1]='\0';
  strcat(IniFile,INIFILE);              // create full path .ini name

  // get log file name
  GetPrivateProfileString(INISECTION,INILOGFILE,"",LogFile,MAXPATH,IniFile);
}

void LogLine(RFTPEventStr* pEventStruc)
// ******
// Log a line to the logfile.
//
// Parameter: pInfoStruc = pointer to event information
// ******
{
  // some bullet proofing
  if (LogFile[0]=='\0') return;         // only if we have a logfile
  RUserInfoStr UserInfo;
  UserInfo.SessionID=pEventStruc->SessionID;
  if (!UserList.Find(UserInfo)) return; // only if we have info on this user

  // create time stamp
  char CurTime[40];
  time_t Count=time(NULL);
  strcpy(CurTime,ctime(&Count));
  CurTime[24]='\0';                     // get rid of newline at end

  // log transfer to file
  FILE* hFile=fopen(LogFile,"a+t");     // append to file
  if (!hFile) return;                   // problem writing to file
  char Direction='o';                   // direction flag
  if (pEventStruc->Event==EVNT_EndUp) Direction='i';
  char Access='a';                      // access flag
  char User[512];                       // user name or anonymous 'password'
  if (stricmp(pEventStruc->User,"anonymous")) {
    Access='r';
    strcpy(User,pEventStruc->User);
  }
  else strcpy(User,UserInfo.AnonPass.c_str());
  fprintf(hFile,"%s %lu %s %lu %s b _ %c %c %s FTP 0 *\n",
    CurTime,pEventStruc->Duration/1000L,UserInfo.IPName.c_str(),pEventStruc->Size,
    pEventStruc->AuxOne,Direction,Access,User);
  fclose(hFile);  
}

WORD CALLBACK HandleEventHook(RFTPEventStr* pEventStruc)
// ******
// Event (hook) handler.
// ******
{
  RUserInfoStr UserInfo;

  // disect event
  switch (pEventStruc->Event) {

    // new user connects to server
    case EVNT_Connect:
      UserInfo.SessionID=pEventStruc->SessionID;
      UserInfo.IPName=pEventStruc->ClientIP;
      UserList.Add(UserInfo);           // add new user to list of users
      break;

    // user disconnects from server
    case EVNT_Close:
      UserInfo.SessionID=pEventStruc->SessionID;
      UserList.Remove(UserInfo);        // remove user info from list
      break;

    // received symbolic IP name info
    case EVNT_IPName:
      UserInfo.SessionID=pEventStruc->SessionID;
      if (UserList.Find(UserInfo)) {    // add IP name to structure in list
        UserList.GetCurItemPoint()->IPName=pEventStruc->AuxOne;
      }
      break;

    // catch anonymous' passwords
    case EVNT_Login:
      if (!stricmp(pEventStruc->User,"anonymous")) {
        UserInfo.SessionID=pEventStruc->SessionID;
        if (UserList.Find(UserInfo)) {  // add 'password' to structure in list
          UserList.GetCurItemPoint()->AnonPass=pEventStruc->AuxOne;
        }
      }
      break;

    // log file transfers
    case EVNT_EndUp:
    case EVNT_EndDown:
      LogLine(pEventStruc);             // log to file
      break;
  }

  return(REVNT_None);
}

BOOL WINAPI DllEntryPoint(HINSTANCE hInstance,DWORD fdwReason,PVOID /*pvReserved*/)
// ******
// Window's DLL entry point.
// Used for initializing list of names-to-block.
// ******
{
  switch (fdwReason) {
    case DLL_PROCESS_ATTACH:
    case DLL_THREAD_ATTACH:
      LoadIniFile(hInstance);           // load .ini file info
      break;
  }

  return(TRUE);                         // signal success to Windows
}