/* Preference handling for the Mac interface to Xconq.
   Copyright (C) 1992-1998 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

#include "conq.h"
extern int ok_to_exit;
#include "kpublic.h"
#include "macconq.h"
extern void recalc_spacing(void);
extern void invert_map(Map *map, int x, int y, int r);

#include <errors.h>
#include <time.h>

#ifdef __MWERKS__
#include <Sound.h>
#endif

#ifdef THINK_C
#define TETextBox TextBox
#endif

#define keyReplyErr 'errn'

#define PREFSIZE 2048		/* Size of prefs buffer in lisp format. Must be bigger than the default */
						/* BUFSIZE = 256 to make room for all the new preferences */

static Boolean missed_any_parameters(AppleEvent *message);

static pascal Boolean filter_warning_alert(DialogPtr dialog, EventRecord *evt, short *itemhit);

/* Global variables. */

/* True if new maps should show the cell grid. */

int default_draw_grid = TRUE;

/* True if new maps should display names and numbers. */

int default_draw_names = TRUE;

/* (should use these eventually instead) */

int default_draw_unit_names = FALSE;
int default_draw_unit_numbers = FALSE;
int default_draw_feature_names = FALSE;

/* True if should display latitude and longitude of locations. */
int default_draw_meridians = TRUE;

/* Spacing, in minutes, of latitude/longitude lines. */
int default_meridian_interval = 300;

int
open_preferences()
{
	short vref;
	long dirid;
	OSErr err;
	Str255 filename;

	GetIndString(filename, sFilenames, siPreferences);
	/* (should do a gestalt check first) */
	err = FindFolder(kOnSystemDisk, kPreferencesFolderType, kDontCreateFolder,
				   &vref, &dirid);
	if (err != noErr) {
		SysEnvRec se;

		err = SysEnvirons(1, &se);
		if (err == noErr) {
			vref = se.sysVRefNum;
		} else {
			vref = 0;
		}
		dirid = 0;
	}
	err = HCreate(vref, dirid, filename, XconqSignature, 'pref');
	if (err == dupFNErr)
	  err = noErr;
	if (err == noErr) {
		/* Ensure that the resource fork exists. */
		HCreateResFile(vref, dirid, filename);
		prefs_refnum = HOpenResFile(vref, dirid, filename, fsRdWrPerm);
		if (prefs_refnum == -1)
			err = -1;
		if (ResError())
			err = ResError();
	}
	return err;
}

void
close_preferences()
{
	CloseResFile(prefs_refnum);
	prefs_refnum = -1;
}

void
get_preferences()
{
	int startlineno = 0, endlineno = 0;
	Obj *uispec;
	OSErr err;
	Handle prefs;

	err = open_preferences();
	if (err == noErr) {
		UseResFile(prefs_refnum);
		prefs = GetResource('XCpf', 128);
		if (prefs != nil) {
			HLock(prefs);
			uispec = read_form_from_string(copy_string(*prefs), &startlineno, &endlineno, NULL);
			interp_mac_ui_data(uispec);
			HUnlock(prefs);
		}
		/* Optional loading of colors from XCol resource */
		if (use_RGB_values)
		  read_xcol();
		close_preferences();
	}
}

void
save_preferences()
{
	int cur_refnum;
	char prefbuf[PREFSIZE];	/* Defined as 2048 */
	OSErr err;
	Handle oldprefs, prefs;

	err = open_preferences();
	if (err == noErr) {
		sprintlisp(prefbuf, find_at_key(dside->uidata, "mac"), PREFSIZE);	/* Defined as 2048 */
		/* Copy the string into a handle that will be used for the resource. */
		prefs = NewHandle(strlen(prefbuf) + 1);
		HLock(prefs);
		strcpy(*prefs, prefbuf);
		HUnlock(prefs);
		cur_refnum = CurResFile();
		UseResFile(prefs_refnum);
		oldprefs = GetResource('XCpf', 128);
		if (oldprefs != nil) {
			RmveResource(oldprefs);
			UpdateResFile(prefs_refnum);
		}
		AddResource(prefs, 'XCpf', 128, "\pXconq Preferences");
		/* Optional loading of colors from XCol resource */
		if (use_RGB_values)
		  write_xcol();
		close_preferences();
		UseResFile(cur_refnum);
	}
}

void
interp_mac_ui_data(Obj *uispec)
{
	int numval;
	char *strval = NULL;
	char *name;
	Obj *rest, *bdg;

	for_all_list(uispec, rest) {
		bdg = car(rest);
		if (!consp(bdg)) {
			/* Don't complain out loud normally, confusing to users
			   because preferences are under Xconq and not user control. */
			Dprintf("Syntax error in preference binding?\n");
			continue;
		}
		if (symbolp(car(bdg))) {
			name = c_string(car(bdg));
			strval = NULL;
			numval = 0;
			if (numberp(cadr(bdg))) {
				numval = c_number(cadr(bdg));
			} else if (stringp(cadr(bdg))) {
				strval = c_string(cadr(bdg));
			} else {
				Dprintf("Preference property `%s' not a number or string, setting to zero\n",
						name);
			}
			if (strcmp(name, "want-statistics") == 0) {
				statistics_wanted = numval;
			} else if (strcmp(name, "checkpoint-interval") == 0) {
				checkpoint_interval = numval;
			} else if (strcmp(name, "default-draw-grid") == 0) {
				default_draw_grid = numval;
			} else if (strcmp(name, "default-draw-names") == 0) {
				default_draw_names = numval;
			} else if (strcmp(name, "default-draw-meridians") == 0) {
				default_draw_meridians = numval;
			} else if (strcmp(name, "default-meridian-interval") == 0) {
				default_meridian_interval = numval;
			} else if (strcmp(name, "color-unit-images") == 0) {
				pref_color_unit_images = numval;
			} else if (strcmp(name, "solid-color-terrain") == 0) {
				pref_solid_color_terrain = numval;
			} else if (strcmp(name, "small-font-size") == 0) {
				small_font_size = numval;
			} else if (strcmp(name, "large-font-size") == 0) {
				large_font_size = numval;
			} else if (strcmp(name, "small-font-id") == 0) {
				small_font_id = numval;
			} else if (strcmp(name, "large-font-id") == 0) {
				large_font_id = numval;
			} else if (strcmp(name, "sound") == 0)
				playsounds = numval;
			else if (strcmp(name, "sidecolors") == 0)
			  default_sidecolors = numval;
			else if (strcmp(name, "iconmasks") == 0)
			  default_iconmasks = numval;
			else if (strcmp(name, "boxmasks") == 0)
			  default_boxmasks = numval;
			else if (strcmp(name, "textmasks") == 0)
			  default_textmasks = numval;
			else if (strcmp(name, "featureborders") == 0)
			  default_featureborders = numval;
			else if (strcmp(name, "featurenames") == 0)
			  default_featurenames = numval;
			else if (strcmp(name, "shorelines") == 0)
			  default_shorelines = numval;
			else if (strcmp(name, "simple-borders") == 0)
			  default_simple_borders = numval;
			else if (strcmp(name, "draw-ai") == 0)
			  default_draw_ai = numval;
			else if (strcmp(name, "depth-alert") == 0)
			  alert_256_colors = numval;
			else if (strcmp(name, "optimize-fonts") == 0)
			  default_optimize_fonts = numval;
			else if (strcmp(name, "draw-emblems") == 0)
			  default_draw_emblems = numval;
			else if (strcmp(name, "draw-topline") == 0)
			  default_draw_topline = numval;
			else if (strcmp(name, "draw-topunit") == 0)
			  default_draw_topunit = numval;
			else if (strcmp(name, "draw-scrollbars") == 0)
			  default_draw_scrollbars = numval;
			else if (strcmp(name, "draw-cpanel") == 0)
			  default_draw_cpanel = numval;
			else if (strcmp(name, "drawothermaps") == 0)
			  default_drawothermaps = numval;
			else if (strcmp(name, "erase-names") == 0)
			  default_erase_names = numval;
			else if (strcmp(name, "x-buffer") == 0)
			  default_max_bufx = numval;
			else if (strcmp(name, "y-buffer") == 0)
			  default_max_bufy = numval;
			else if (strcmp(name, "use-colornames") == 0)
			  use_colornames = numval;
			else if (strcmp(name, "use-RGB-values") == 0)
			  use_RGB_values = numval;
			else if (strcmp(name, "unified-cicns") == 0)
			  unified_cicns = numval;
			else if (strcmp(name, "show-instructions") == 0)
			  show_instructions = numval;
			else if (strcmp(name, "fullsize-map") == 0)
			  fullsize_map = numval;
			else if (strcmp(name, "power") == 0)
			  default_power = numval;

			else if (strcmp(name, "main-icon-color") == 0)
			  default_main_icon_color = numval;
			else if (strcmp(name, "half-icon-color") == 0)
			  default_half_icon_color = numval;
			else if (strcmp(name, "icon-mask-color") == 0)
			  default_icon_mask_color = numval;
			else if (strcmp(name, "two-color-icons") == 0)
			  default_split_icon = numval;

			else if (use_colornames) {

			   	if (strcmp(name, "side0-colors") == 0)
				  default_colorscheme[0] = strval;
			   	else if (strcmp(name, "side1-colors") == 0)
				  default_colorscheme[1] = strval;
			 	else if (strcmp(name, "side2-colors") == 0)
				  default_colorscheme[2] = strval;
			   	else if (strcmp(name, "side3-colors") == 0)
				  default_colorscheme[3] = strval;
			   	else if (strcmp(name, "side4-colors") == 0)
				  default_colorscheme[4] = strval;
			   	else if (strcmp(name, "side5-colors") == 0)
				  default_colorscheme[5] = strval;
			   	else if (strcmp(name, "side6-colors") == 0)
				  default_colorscheme[6] = strval;
			   	else if (strcmp(name, "side7-colors") == 0)
				  default_colorscheme[7] = strval;
			   	else if (strcmp(name, "side8-colors") == 0)
				  default_colorscheme[8] = strval;
			   	else if (strcmp(name, "side9-colors") == 0)
				  default_colorscheme[9] = strval;
			   	else if (strcmp(name, "side10-colors") == 0)
				  default_colorscheme[10] = strval;
			   	else if (strcmp(name, "side11-colors") == 0)
				  default_colorscheme[11] = strval;
			   	else if (strcmp(name, "side12-colors") == 0)
				  default_colorscheme[12] = strval;
			   	else if (strcmp(name, "side13-colors") == 0)
				  default_colorscheme[13] = strval;
			   	else if (strcmp(name, "side14-colors") == 0)
				  default_colorscheme[14] = strval;
			   	else if (strcmp(name, "side15-colors") == 0)
				  default_colorscheme[15] = strval;
			   	else if (strcmp(name, "backcolor") == 0)
				  default_featColorName[0] = strval;
			   	else if (strcmp(name, "forecolor") == 0)
				  default_featColorName[1] = strval;
			   	else if (strcmp(name, "maskcolor") == 0)
				  default_featColorName[2] = strval;
			   	else if (strcmp(name, "textcolor") == 0)
				  default_featColorName[3] = strval;
			   	else if (strcmp(name, "gridcolor") == 0)
				  default_featColorName[4] = strval;
			   	else if (strcmp(name, "unseencolor") == 0)
				  default_featColorName[5] = strval;
			   	else if (strcmp(name, "shorecolor") == 0)
				  default_featColorName[6] = strval;
			   	else if (strcmp(name, "featurecolor") == 0)
				  default_featColorName[7] = strval;
			   	else if (strcmp(name, "frontcolor") == 0)
				  default_featColorName[8] = strval;
			   	else if (strcmp(name, "meridiancolor") == 0)
				  default_featColorName[9] = strval;
			   	else if (strcmp(name, "contourcolor") == 0)
				  default_featColorName[10] = strval;
			} else {
				/* Note unrecognized properties, but don't bother the user. */
				Dprintf("Preference binding `%s' unrecognized\n", name);
			}
		} else {
			/* As with above comment. */
			Dprintf("Syntax error in preference binding head?\n");
		}
	}
}

void
ui_update_state()
{
	Obj *state = lispnil;

	push_binding(&state, intern_symbol("want-statistics"), new_number(statistics_wanted));
	push_binding(&state, intern_symbol("checkpoint-interval"), new_number(checkpoint_interval));
	push_binding(&state, intern_symbol("default-draw-grid"), new_number(default_draw_grid));
	push_binding(&state, intern_symbol("default-draw-names"), new_number(default_draw_names));
	push_binding(&state, intern_symbol("default-draw-meridians"), new_number(default_draw_meridians));
	push_binding(&state, intern_symbol("default-meridian-interval"), new_number(default_meridian_interval));
	push_binding(&state, intern_symbol("color-unit-images"), new_number(pref_color_unit_images));
	push_binding(&state, intern_symbol("solid-color-terrain"), new_number(pref_solid_color_terrain));
	push_binding(&state, intern_symbol("small-font-size"), new_number(small_font_size));
	push_binding(&state, intern_symbol("large-font-size"), new_number(large_font_size));
	push_binding(&state, intern_symbol("small-font-id"), new_number(small_font_id));
	push_binding(&state, intern_symbol("large-font-id"), new_number(large_font_id));
	push_binding(&state, intern_symbol("sound"), new_number(playsounds));
	push_binding(&state, intern_symbol("sidecolors"), new_number(default_sidecolors));
	push_binding(&state, intern_symbol("iconmasks"), new_number(default_iconmasks));
	push_binding(&state, intern_symbol("boxmasks"), new_number(default_boxmasks));
	push_binding(&state, intern_symbol("textmasks"), new_number(default_textmasks));
	push_binding(&state, intern_symbol("featureborders"), new_number(default_featureborders));
	push_binding(&state, intern_symbol("featurenames"), new_number(default_featurenames));
	push_binding(&state, intern_symbol("shorelines"), new_number(default_shorelines));
	push_binding(&state, intern_symbol("simple-borders"), new_number(default_simple_borders));
	push_binding(&state, intern_symbol("draw-ai"), new_number(default_draw_ai));
	push_binding(&state, intern_symbol("depth-alert"), new_number(alert_256_colors));
	push_binding(&state, intern_symbol("optimize-fonts"), new_number(default_optimize_fonts));
	push_binding(&state, intern_symbol("draw-emblems"), new_number(default_draw_emblems));
	push_binding(&state, intern_symbol("draw-topline"), new_number(default_draw_topline));
	push_binding(&state, intern_symbol("draw-topunit"), new_number(default_draw_topunit));
	push_binding(&state, intern_symbol("draw-scrollbars"), new_number(default_draw_scrollbars));
	push_binding(&state, intern_symbol("draw-cpanel"), new_number(default_draw_cpanel));
	push_binding(&state, intern_symbol("drawothermaps"), new_number(default_drawothermaps));
	push_binding(&state, intern_symbol("erase-names"), new_number(default_erase_names));
	push_binding(&state, intern_symbol("x-buffer"), new_number(default_max_bufx));
	push_binding(&state, intern_symbol("y-buffer"), new_number(default_max_bufy));
	push_binding(&state, intern_symbol("use-colornames"), new_number(use_colornames));
	push_binding(&state, intern_symbol("use-RGB-values"), new_number(use_RGB_values));
	push_binding(&state, intern_symbol("unified-cicns"), new_number(unified_cicns));
	push_binding(&state, intern_symbol("show-instructions"), new_number(show_instructions));
	push_binding(&state, intern_symbol("fullsize-map"), new_number(fullsize_map));
	push_binding(&state, intern_symbol("power"), new_number(default_power));

	push_binding(&state, intern_symbol("main-icon-color"), new_number(default_main_icon_color));
	push_binding(&state, intern_symbol("half-icon-color"), new_number(default_half_icon_color));
	push_binding(&state, intern_symbol("icon-mask-color"), new_number(default_icon_mask_color));
	push_binding(&state, intern_symbol("two-color-icons"), new_number(default_split_icon));

	if (use_colornames) {

		push_binding(&state, intern_symbol("side0-colors"), new_string(default_colorscheme[0]));
		push_binding(&state, intern_symbol("side1-colors"), new_string(default_colorscheme[1]));
		push_binding(&state, intern_symbol("side2-colors"), new_string(default_colorscheme[2]));
		push_binding(&state, intern_symbol("side3-colors"), new_string(default_colorscheme[3]));
		push_binding(&state, intern_symbol("side4-colors"), new_string(default_colorscheme[4]));
		push_binding(&state, intern_symbol("side5-colors"), new_string(default_colorscheme[5]));
		push_binding(&state, intern_symbol("side6-colors"), new_string(default_colorscheme[6]));
		push_binding(&state, intern_symbol("side7-colors"), new_string(default_colorscheme[7]));
		push_binding(&state, intern_symbol("side8-colors"), new_string(default_colorscheme[8]));
		push_binding(&state, intern_symbol("side9-colors"), new_string(default_colorscheme[9]));
		push_binding(&state, intern_symbol("side10-colors"), new_string(default_colorscheme[10]));
		push_binding(&state, intern_symbol("side11-colors"), new_string(default_colorscheme[11]));
		push_binding(&state, intern_symbol("side12-colors"), new_string(default_colorscheme[12]));
		push_binding(&state, intern_symbol("side13-colors"), new_string(default_colorscheme[13]));
		push_binding(&state, intern_symbol("side14-colors"), new_string(default_colorscheme[14]));
		push_binding(&state, intern_symbol("side15-colors"), new_string(default_colorscheme[15]));
		push_binding(&state, intern_symbol("backcolor"), new_string(default_featColorName[0]));
		push_binding(&state, intern_symbol("forecolor"), new_string(default_featColorName[1]));
		push_binding(&state, intern_symbol("maskcolor"), new_string(default_featColorName[2]));
		push_binding(&state, intern_symbol("textcolor"), new_string(default_featColorName[3]));
		push_binding(&state, intern_symbol("gridcolor"), new_string(default_featColorName[4]));
		push_binding(&state, intern_symbol("unseencolor"), new_string(default_featColorName[5]));
		push_binding(&state, intern_symbol("shorecolor"), new_string(default_featColorName[6]));
		push_binding(&state, intern_symbol("featurecolor"), new_string(default_featColorName[7]));
		push_binding(&state, intern_symbol("frontcolor"), new_string(default_featColorName[8]));
		push_binding(&state, intern_symbol("meridiancolor"), new_string(default_featColorName[9]));
		push_binding(&state, intern_symbol("contourcolor"), new_string(default_featColorName[10]));
	}
	dside->uidata = replace_at_key(dside->uidata, "mac", state);
}

/* Dialogue to set the preferences. */

int cur_prefs_topic = 2;

static pascal void draw_font_sample(WindowPtr win, short ditem);
static void force_sample_redraw(WindowPtr win, short ditem);
static void add_preferences_topic_items(DialogPtr win);

UserItemUPP draw_font_sample_proc;

int tmp_small_font_id;
int tmp_small_font_size;
int tmp_large_font_id;
int tmp_large_font_size;

/* Dialog app-defined item callback to display the chosen font. */

static pascal void
draw_font_sample(WindowPtr win, short ditem)
{
	short curfont, cursize, curstyle;
	char *str;
	short itemtype;  Handle itemhandle;  Rect itemrect;

	/* Save the current font. */
	curfont = QD(thePort)->txFont;
	cursize = QD(thePort)->txSize;
	curstyle = QD(thePort)->txFace;

	GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
	if (ditem == diPrefsSmallFontSample) {
		str = "This is the font used in maps, lists and for general text.";
		TextFont(tmp_small_font_id);
		TextSize(tmp_small_font_size);
	} else {
		str = "This is the font used for headings and game status.";
		TextFont(tmp_large_font_id);
		TextSize(tmp_large_font_size);
	}
	TETextBox(str, strlen(str), &itemrect, teFlushDefault);
	/* Restore the current font. */
	TextFont(curfont);
	TextSize(cursize);
	TextFace(curstyle);
}

static void
force_sample_redraw(WindowPtr win, short ditem)
{
	short itemtype;  Handle itemhandle;  Rect itemrect;
	GrafPtr oldport;

	GetPort(&oldport);
	SetPort(win);
	GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
	InsetRect(&itemrect, 2, 2);
	EraseRect(&itemrect);
	InvalRect(&itemrect);
	SetPort(oldport);
}

void
set_preferences()
{
	short done = FALSE, ditem, oldcolor, oldsolid, new_prefs_topic;
	int new_interval, new_incr, new_font_size, fonts_changed;
	DialogPtr win;
	short itemtype;  Handle itemhandle;  Rect itemrect;
	Handle ditl;
	Str255 tmpstr;
	short old_unified_cicns = unified_cicns;
	short mitem, new_font_id;
	long *itemnum = 0;
	Map *map;
	int e = 0;

	/* Build popup menus for Color Schemes dialog */
	build_features_menu();
	build_sides_menu();

	if (draw_font_sample_proc == NULL)
	  draw_font_sample_proc = NewUserItemProc(draw_font_sample);
	tmp_small_font_id = small_font_id;
	tmp_small_font_size = small_font_size;
	tmp_large_font_id = large_font_id;
	tmp_large_font_size = large_font_size;
	fonts_changed = FALSE;
	oldcolor = pref_color_unit_images;
	oldsolid = pref_solid_color_terrain;
	win = GetNewDialog(dPreferences, NULL, (DialogPtr) -1L);
	GetDItem(win, diPrefsPopup, &itemtype, &itemhandle, &itemrect);
	SetCtlValue((ControlHandle) itemhandle, cur_prefs_topic);
	add_preferences_topic_items(win);
	ShowWindow(win);
	while (!done) {
		SetCursor(&QD(arrow));
		ModalDialog(NULL, &ditem);
		switch (ditem) {
			case diPrefsOK:
				/* Actually change the program's variables. */
				switch (cur_prefs_topic) {
					case miPreferencesTopicsFiles:
						GetDItem(win, diPrefsCheckpoint, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) {
							GetDItem(win, diPrefsInterval, &itemtype, &itemhandle, &itemrect);
				        			GetIText(itemhandle,tmpstr);
							StringToNum(tmpstr, itemnum);
							if (itemnum != 0) {
								checkpoint_interval = *itemnum;
							}
						} else {
							checkpoint_interval = 0;
						}

						GetDItem(win, diPrefsStatistics, &itemtype, &itemhandle, &itemrect);
						statistics_wanted = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsColXCpf, &itemtype, &itemhandle, &itemrect);
						use_colornames = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsColXCol, &itemtype, &itemhandle, &itemrect);
						use_RGB_values = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsShowInstr, &itemtype, &itemhandle, &itemrect);
						show_instructions = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsFullSize, &itemtype, &itemhandle, &itemrect);
						fullsize_map = GetCtlValue((ControlHandle) itemhandle);

						break;

					case miPreferencesTopicsMap:
						GetDItem(win, diPrefsGrid, &itemtype, &itemhandle, &itemrect);
						default_draw_grid = GetCtlValue((ControlHandle) itemhandle);
						GetDItem(win, diPrefsNames, &itemtype, &itemhandle, &itemrect);
						default_draw_names = GetCtlValue((ControlHandle) itemhandle);
						GetDItem(win, diPrefsSolidColorTerrain, &itemtype, &itemhandle, &itemrect);
						pref_solid_color_terrain = GetCtlValue((ControlHandle) itemhandle);
						GetDItem(win, diPrefsMeridians, &itemtype, &itemhandle, &itemrect);
						default_draw_meridians = GetCtlValue((ControlHandle) itemhandle);
						get_number_from_ditem(diPrefsIncr, new_incr);
						/* Can't use directly in get_number_from_ditem macro, can't
						   guarantee size of variable being written into. */
						default_meridian_interval = new_incr;

						GetDItem(win, diPrefsSideColors, &itemtype, &itemhandle, &itemrect);
						default_sidecolors = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsIconMasks, &itemtype, &itemhandle, &itemrect);
						default_iconmasks = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsBoxMasks, &itemtype, &itemhandle, &itemrect);
						default_boxmasks = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsTextMasks, &itemtype, &itemhandle, &itemrect);
						default_textmasks = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsFeatureBorders, &itemtype, &itemhandle, &itemrect);
						default_featureborders = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsFeatureNames, &itemtype, &itemhandle, &itemrect);
						default_featurenames = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsAIRegions, &itemtype, &itemhandle, &itemrect);
						default_draw_ai = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsSimpleBorders, &itemtype, &itemhandle, &itemrect);
						default_simple_borders = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsComplexBorders, &itemtype, &itemhandle, &itemrect);
						default_simple_borders = !GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsShoreLines, &itemtype, &itemhandle, &itemrect);
						default_shorelines = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsTopLine, &itemtype, &itemhandle, &itemrect);
						default_draw_topline = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsUnitInfo, &itemtype, &itemhandle, &itemrect);
						default_draw_topunit = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsScrollBars, &itemtype, &itemhandle, &itemrect);
						default_draw_scrollbars = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsControlPanel, &itemtype, &itemhandle, &itemrect);
						default_draw_cpanel = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsOtherMaps, &itemtype, &itemhandle, &itemrect);
						default_drawothermaps = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsEraseNames, &itemtype, &itemhandle, &itemrect);
						default_erase_names = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsMagPopup, &itemtype, &itemhandle, &itemrect);
						mitem = GetCtlValue((ControlHandle) itemhandle);
						switch (mitem) {
							case 1:	default_power = 0; 	break;
							case 2:	default_power = 1; 	break;
							case 3:	default_power = 2; 	break;
							case 4:	default_power = 3; 	break;
							case 5:	default_power = 4; 	break;
							case 6:	default_power = 5; 	break;
							case 7:	default_power = 6; 	break;
							case 8:	default_power = 7; 	break;
						}
						break;
					case miPreferencesTopicsFonts:
						GetDItem(win, diPrefsOptimFonts, &itemtype, &itemhandle, &itemrect);
						default_optimize_fonts = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsScaledFonts, &itemtype, &itemhandle, &itemrect);
						default_optimize_fonts = !GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsSmallFontPopup, 
							&itemtype, &itemhandle, &itemrect);
						mitem = GetCtlValue((ControlHandle) itemhandle);
						switch (mitem) {
							case miChicago: 	new_font_id = 0; 	break;
							case miCourier: 	new_font_id = 22; 	break;
							case miGeneva: 		new_font_id = 3; 	break;
							case miHelvetica: 	new_font_id = 21; 	break;
							case miMonaco: 		new_font_id = 4; 	break;
							case miNewYork: 	new_font_id = 2; 	break;
							case miSymbol: 		new_font_id = 23; 	break;
							case miTimes: 		new_font_id = 20; 	break;
						}
						if (new_font_id != small_font_id) {
							small_font_id = new_font_id;
							fonts_changed = TRUE;
						}

						GetDItem(win, diPrefsLargeFontPopup, 
							&itemtype, &itemhandle, &itemrect);
						mitem = GetCtlValue((ControlHandle) itemhandle);
						switch (mitem) {
							case miChicago: 	new_font_id = 0; 	break;
							case miCourier: 	new_font_id = 22; 	break;
							case miGeneva: 		new_font_id = 3; 	break;
							case miHelvetica: 	new_font_id = 21; 	break;
							case miMonaco: 		new_font_id = 4; 	break;
							case miNewYork: 	new_font_id = 2; 	break;
							case miSymbol: 		new_font_id = 23; 	break;
							case miTimes: 		new_font_id = 20; 	break;
						}
						if (new_font_id != large_font_id) {
							large_font_id = new_font_id;
							fonts_changed = TRUE;
						}

/* 	No need to call diPrefsSmallSizePopup and diPrefsLargeSizePopup here since their
	values already have been copied to diPrefsSmallSizeText and diPrefsLargeSizeText */
						
						get_number_from_ditem(diPrefsSmallSizeText, new_font_size);
						if (new_font_size != 0 && new_font_size != small_font_size) {
							small_font_size = new_font_size;
							fonts_changed = TRUE;
						}
						get_number_from_ditem(diPrefsLargeSizeText, new_font_size);
						if (new_font_size != 0 && new_font_size != large_font_size) {
							large_font_size = new_font_size;
							fonts_changed = TRUE;
						}
						break;

					case miPreferencesTopicsOffscreen:

						GetDItem(win, diPrefsHorizBuff, &itemtype, &itemhandle, &itemrect);
			        			GetIText(itemhandle,tmpstr);
						StringToNum(tmpstr, itemnum);
						/* Dont allow negative values */
						default_max_bufx = max(*itemnum, 0);

						GetDItem(win, diPrefsVertiBuff, &itemtype, &itemhandle, &itemrect);
			         			GetIText(itemhandle,tmpstr);
						StringToNum(tmpstr, itemnum);
						/* Dont allow negative values */
						default_max_bufy = max(*itemnum, 0);

						GetDItem(win, diPrefs256colors, &itemtype, &itemhandle, &itemrect);
						alert_256_colors = GetCtlValue((ControlHandle) itemhandle);

						break;

					case miPreferencesTopicsImageSets:

						GetDItem(win, diPrefsUseStandIcons, &itemtype, &itemhandle, &itemrect);
						pref_color_unit_images = !GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsUseColorIcons, &itemtype, &itemhandle, &itemrect);
						pref_color_unit_images = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsUnifiedCicns, &itemtype, &itemhandle, &itemrect);
						unified_cicns = GetCtlValue((ControlHandle) itemhandle);

						GetDItem(win, diPrefsDrawEmblems, &itemtype, &itemhandle, &itemrect);
						default_draw_emblems = GetCtlValue((ControlHandle) itemhandle);

						break;

					case miPreferencesTopicsColorSchemes:

						GetDItem(win, diPrefsSplitIcon, &itemtype, &itemhandle, &itemrect);
						default_split_icon = (GetCtlValue((ControlHandle) itemhandle));

						GetDItem(win, diPrefsMainColor0, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_main_icon_color = 0;

						GetDItem(win, diPrefsMainColor1, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_main_icon_color = 1;

						GetDItem(win, diPrefsMainColor2, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_main_icon_color = 2;

						GetDItem(win, diPrefsMainColor3, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_main_icon_color = 3;

						GetDItem(win, diPrefsHalfColor0, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_half_icon_color = 0;

						GetDItem(win, diPrefsHalfColor1, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_half_icon_color = 1;

						GetDItem(win, diPrefsHalfColor2, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_half_icon_color = 2;

						GetDItem(win, diPrefsHalfColor3, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_half_icon_color = 3;

						GetDItem(win, diPrefsMaskColor0, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_icon_mask_color = 0;

						GetDItem(win, diPrefsMaskColor1, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_icon_mask_color = 1;

						GetDItem(win, diPrefsMaskColor2, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_icon_mask_color = 2;

						GetDItem(win, diPrefsMaskColor3, &itemtype, &itemhandle, &itemrect);
						if (GetCtlValue((ControlHandle) itemhandle)) default_icon_mask_color = 3;

						break;

				}
				/* Remember all these settings. */
				ui_update_state();
				save_preferences();
				/* Fall into next case. */
			case diPrefsCancel:
				done = TRUE;
				break;
			case diPrefsPopup:
				GetDItem(win, diPrefsPopup, &itemtype, &itemhandle, &itemrect);
				new_prefs_topic = GetCtlValue((ControlHandle) itemhandle);
				if (new_prefs_topic != cur_prefs_topic) {
					/* (should collect pref settings from cur topic, but only set
					   when final ok given) */
					cur_prefs_topic = new_prefs_topic;
					ShortenDITL(win, CountDITL(win) - diPrefsCommon);
					add_preferences_topic_items(win);
				}
				break;
			default:
				switch (cur_prefs_topic) {
					case miPreferencesTopicsFiles:
						switch (ditem) {
							case diPrefsCheckpoint:
							case diPrefsStatistics:
							case diPrefsColXCpf:
							case diPrefsColXCol:
							case diPrefsShowInstr:
							case diPrefsFullSize:
								/* Toggle check boxes. */
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
								break;
							case diPrefsInterval:
								break;
							default:
								break;
						}
						break;

					case miPreferencesTopicsMap:
						switch (ditem) {
							case diPrefsGrid:
							case diPrefsNames:
							case diPrefsSolidColorTerrain:
							case diPrefsMeridians:
							case diPrefsSideColors:
							case diPrefsIconMasks:
							case diPrefsBoxMasks:
							case diPrefsTextMasks:
							case diPrefsFeatureBorders:
							case diPrefsFeatureNames:
							case diPrefsShoreLines:
							case diPrefsAIRegions:
							case diPrefsTopLine:
							case diPrefsUnitInfo:
							case diPrefsScrollBars:
							case diPrefsControlPanel:
							case diPrefsOtherMaps:
							case diPrefsEraseNames:
							/* Toggle checkboxes */
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
								break;
							/* Switch radio buttons */
							case diPrefsSimpleBorders:
								GetDItem(win, diPrefsSimpleBorders, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsComplexBorders, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsComplexBorders:
								GetDItem(win, diPrefsSimpleBorders, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsComplexBorders, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								break;
							default:
								break;
						}
						break;
					case miPreferencesTopicsFonts:
						switch (ditem) {
							case diPrefsSmallSizeText:
								get_number_from_ditem(diPrefsSmallSizeText, new_font_size);
								if (new_font_size != 0 && 
									new_font_size != tmp_small_font_size) {
									tmp_small_font_size = new_font_size;
								}
								force_sample_redraw(win, diPrefsSmallFontSample);
								break;
							case diPrefsSmallSizePopup:
								GetDItem(win, diPrefsSmallSizePopup, 
									&itemtype, &itemhandle, &itemrect);
								mitem = GetCtlValue((ControlHandle) itemhandle);
								switch (mitem) {
									case mi9Points: 	new_font_size = 9; 		break;
									case mi10Points: 	new_font_size = 10; 	break;
									case mi12Points: 	new_font_size = 12; 	break;
									case mi14Points: 	new_font_size = 14; 	break;
									case mi18Points: 	new_font_size = 18; 	break;
									case mi24Points: 	new_font_size = 24; 	break;
									case mi36Points: 	new_font_size = 36; 	break;
									case mi48Points: 	new_font_size = 48; 	break;
								}
								SetCtlValue((ControlHandle) itemhandle, mitem);

								if (new_font_size != 0 && 
									new_font_size != tmp_small_font_size) {
									tmp_small_font_size = new_font_size;
									put_number_into_ditem(diPrefsSmallSizeText, new_font_size);
								}
								force_sample_redraw(win, diPrefsSmallFontSample);
								break;
							case diPrefsSmallFontPopup:
								GetDItem(win, diPrefsSmallFontPopup, 
									&itemtype, &itemhandle, &itemrect);
								mitem = GetCtlValue((ControlHandle) itemhandle);
								switch (mitem) {
									case miChicago: 	new_font_id = 0; 	break;
									case miCourier: 	new_font_id = 22; 	break;
									case miGeneva: 		new_font_id = 3; 	break;
									case miHelvetica: 	new_font_id = 21; 	break;
									case miMonaco: 		new_font_id = 4; 	break;
									case miNewYork: 	new_font_id = 2; 	break;
									case miSymbol: 		new_font_id = 23; 	break;
									case miTimes: 		new_font_id = 20; 	break;
								}
								SetCtlValue((ControlHandle) itemhandle, mitem);

								if (new_font_id != tmp_small_font_id) {
									tmp_small_font_id = new_font_id;
								}
								force_sample_redraw(win, diPrefsSmallFontSample);
								break;
							case diPrefsLargeSizeText:
								get_number_from_ditem(diPrefsLargeSizeText, new_font_size);
								if (new_font_size != 0 && new_font_size != tmp_large_font_size) {
									tmp_large_font_size = new_font_size;
								}
								force_sample_redraw(win, diPrefsLargeFontSample);
								break;
							case diPrefsLargeSizePopup:
								GetDItem(win, diPrefsLargeSizePopup, 
									&itemtype, &itemhandle, &itemrect);
								mitem = GetCtlValue((ControlHandle) itemhandle);
								switch (mitem) {
									case mi9Points: 	new_font_size = 9; 		break;
									case mi10Points:	new_font_size = 10; 	break;
									case mi12Points: 	new_font_size = 12; 	break;
									case mi14Points: 	new_font_size = 14; 	break;
									case mi18Points: 	new_font_size = 18; 	break;
									case mi24Points: 	new_font_size = 24; 	break;
									case mi36Points: 	new_font_size = 36; 	break;
									case mi48Points: 	new_font_size = 48; 	break;
								}
								if (new_font_size != 0 && 
									new_font_size != tmp_large_font_size) {
									tmp_large_font_size = new_font_size;
									put_number_into_ditem(diPrefsLargeSizeText, new_font_size);
								}
								force_sample_redraw(win, diPrefsLargeFontSample);
								break;
							case diPrefsLargeFontPopup:
								GetDItem(win, diPrefsLargeFontPopup, 
									&itemtype, &itemhandle, &itemrect);
								mitem = GetCtlValue((ControlHandle) itemhandle);
								switch (mitem) {
									case miChicago: 		new_font_id = 0; 	break;
									case miCourier: 		new_font_id = 22; 	break;
									case miGeneva: 		new_font_id = 3; 	break;
									case miHelvetica: 	new_font_id = 21; 	break;
									case miMonaco: 		new_font_id = 4; 	break;
									case miNewYork: 	new_font_id = 2; 	break;
									case miSymbol: 		new_font_id = 23; 	break;
									case miTimes: 		new_font_id = 20; 	break;
								}
								SetCtlValue((ControlHandle) itemhandle, mitem);

								if (new_font_id != tmp_large_font_id) {
									tmp_large_font_id = new_font_id;
								}
								force_sample_redraw(win, diPrefsLargeFontSample);
								break;
							case diPrefsOptimFonts:
								GetDItem(win, diPrefsOptimFonts, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsScaledFonts, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;

							case diPrefsScaledFonts:
								GetDItem(win, diPrefsOptimFonts, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsScaledFonts, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								break;
							default:
								break;
						}
						break;
					case miPreferencesTopicsOffscreen:
						switch (ditem) {
							/* Toggle the checkbox. */
							case diPrefs256colors:
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
								break;
							default:
								break;
						}
						break;
					case miPreferencesTopicsImageSets:
						switch (ditem) {

							/* Switch radiobuttons */
							case diPrefsUseStandIcons:
								GetDItem(win, diPrefsUseStandIcons, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsUseColorIcons, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsUseColorIcons:
								GetDItem(win, diPrefsUseStandIcons, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsUseColorIcons, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								break;

							/* Toggle checkbox */
							case diPrefsUnifiedCicns:
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
								break;

							/* Toggle checkbox */
							case diPrefsDrawEmblems:
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
								break;

							default:
								break;
						}
						break;
					case miPreferencesTopicsColorSchemes:
						switch (ditem) {

							/* Toggle split icon checkbox */
							case diPrefsSplitIcon:
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
								break;

							/* Activate set side colors dialog */
							case diPrefsSidesPopup:
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								if (GetCtlValue((ControlHandle) itemhandle) > 2)
									set_default_side_colors(GetCtlValue((ControlHandle) itemhandle) - 3);
								break;

							/* Activate set feature colors dialog */
							case diPrefsFeaturesPopup:
								GetDItem(win, ditem, &itemtype, &itemhandle, &itemrect);
								if (GetCtlValue((ControlHandle) itemhandle) > 2)
									set_default_feature_color(GetCtlValue((ControlHandle) itemhandle) - 3);
								break;

							/* Main icon color radio buttons */
							case diPrefsMainColor0:
								GetDItem(win, diPrefsMainColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsMainColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsMainColor1:
								GetDItem(win, diPrefsMainColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsMainColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsMainColor2:
								GetDItem(win, diPrefsMainColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsMainColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsMainColor3:
								GetDItem(win, diPrefsMainColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMainColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								break;

							/* Half icon color radio buttons */
							case diPrefsHalfColor0:
								GetDItem(win, diPrefsHalfColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsHalfColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsHalfColor1:
								GetDItem(win, diPrefsHalfColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsHalfColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsHalfColor2:
								GetDItem(win, diPrefsHalfColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsHalfColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsHalfColor3:
								GetDItem(win, diPrefsHalfColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsHalfColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								break;
							
							/* Icon mask color radio buttons */
							case diPrefsMaskColor0:
								GetDItem(win, diPrefsMaskColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsMaskColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsMaskColor1:
								GetDItem(win, diPrefsMaskColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsMaskColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsMaskColor2:
								GetDItem(win, diPrefsMaskColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								GetDItem(win, diPrefsMaskColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								break;
							case diPrefsMaskColor3:
								GetDItem(win, diPrefsMaskColor0, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor1, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor2, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, FALSE);
								GetDItem(win, diPrefsMaskColor3, &itemtype, &itemhandle, &itemrect);
								SetCtlValue((ControlHandle) itemhandle, TRUE);
								break;
							default:
								break;
						}
						break;
				}
				break;
		}
	}
	DisposeDialog(win);
	update_all_map_windows();
	/* Reflect some kinds of changes immediately. */
	if (pref_color_unit_images != oldcolor)
	  init_unit_images();
	if (fonts_changed)
	  recalc_spacing();
	if (pref_color_unit_images != oldcolor
		|| pref_solid_color_terrain != oldsolid
		|| fonts_changed) {
		force_overall_update();
	}
	if (unified_cicns != old_unified_cicns) {
		for_all_maps(map) {
			force_update(map->window);
		}
	}
}

void
add_preferences_topic_items(DialogPtr win)
{
	Str255 tmpstr;
	Handle ditl;
	short itemtype;  Handle itemhandle;  Rect itemrect;

	switch (cur_prefs_topic) {
		case miPreferencesTopicsFiles:
			ditl = GetResource('DITL', dsPreferencesFiles);
			AppendDITL(win, ditl, overlayDITL);
			ReleaseResource(ditl);
			GetDItem(win, diPrefsCheckpoint, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, (checkpoint_interval > 0));
			put_number_into_ditem(diPrefsInterval, checkpoint_interval);
			GetDItem(win, diPrefsStatistics, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, statistics_wanted);
			GetDItem(win, diPrefsColXCpf, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, use_colornames);
			GetDItem(win, diPrefsColXCol, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, use_RGB_values);
			GetDItem(win, diPrefsShowInstr, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, show_instructions);
			GetDItem(win, diPrefsFullSize, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, fullsize_map);
			break;
			
		case miPreferencesTopicsMap:
			ditl = GetResource('DITL', dsPreferencesMap);
			AppendDITL(win, ditl, overlayDITL);
			ReleaseResource(ditl);
			/* (should use helper function) */
			GetDItem(win, diPrefsGrid, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_grid);
			GetDItem(win, diPrefsNames, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_names);
			GetDItem(win, diPrefsSolidColorTerrain, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, pref_solid_color_terrain);
			GetDItem(win, diPrefsMeridians, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_meridians);
			put_number_into_ditem(diPrefsIncr, default_meridian_interval);
			GetDItem(win, diPrefsSideColors, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_sidecolors);
			GetDItem(win, diPrefsIconMasks, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_iconmasks);
			GetDItem(win, diPrefsBoxMasks, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_boxmasks);
			GetDItem(win, diPrefsTextMasks, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_textmasks);
			GetDItem(win, diPrefsFeatureBorders, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_featureborders);
			GetDItem(win, diPrefsFeatureNames, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_featurenames);
			GetDItem(win, diPrefsSimpleBorders, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_simple_borders);
			GetDItem(win, diPrefsComplexBorders, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, !default_simple_borders);
			GetDItem(win, diPrefsShoreLines, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_shorelines);
			GetDItem(win, diPrefsAIRegions, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_ai);
			GetDItem(win, diPrefsTopLine, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_topline);
			GetDItem(win, diPrefsUnitInfo, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_topunit);
			GetDItem(win, diPrefsScrollBars, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_scrollbars);
			GetDItem(win, diPrefsControlPanel, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_cpanel);
			GetDItem(win, diPrefsOtherMaps, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_drawothermaps);
			GetDItem(win, diPrefsEraseNames, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_erase_names);
			GetDItem(win, diPrefsMagPopup, &itemtype, &itemhandle, &itemrect);
			switch (default_power) {
				case 0: 	SetCtlValue((ControlHandle) itemhandle, 1);	break;
				case 1: 	SetCtlValue((ControlHandle) itemhandle, 2);	break;
				case 2: 	SetCtlValue((ControlHandle) itemhandle, 3);	break;
				case 3: 	SetCtlValue((ControlHandle) itemhandle, 4);	break;
				case 4: 	SetCtlValue((ControlHandle) itemhandle, 5);	break;
				case 5: 	SetCtlValue((ControlHandle) itemhandle, 6);	break;
				case 6:	SetCtlValue((ControlHandle) itemhandle, 7);	break;
				case 7: 	SetCtlValue((ControlHandle) itemhandle, 8);	break;
			}
			break;
		case miPreferencesTopicsFonts:
			ditl = GetResource('DITL', dsPreferencesFonts);
			AppendDITL(win, ditl, overlayDITL);
			ReleaseResource(ditl);
			GetDItem(win, diPrefsOptimFonts, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_optimize_fonts);

			GetDItem(win, diPrefsScaledFonts, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, !default_optimize_fonts);

			GetDItem(win, diPrefsSmallFontPopup, &itemtype, &itemhandle, &itemrect);
			switch (tmp_small_font_id) {
				case 0: 	SetCtlValue((ControlHandle) itemhandle, miChicago);	break;
				case 2: 	SetCtlValue((ControlHandle) itemhandle, miNewYork);	break;
				case 3: 	SetCtlValue((ControlHandle) itemhandle, miGeneva);	break;
				case 4: 	SetCtlValue((ControlHandle) itemhandle, miMonaco);	break;
				case 20: 	SetCtlValue((ControlHandle) itemhandle, miTimes);	break;
				case 21: 	SetCtlValue((ControlHandle) itemhandle, miHelvetica);	break;
				case 22:	SetCtlValue((ControlHandle) itemhandle, miCourier);	break;
				case 23: 	SetCtlValue((ControlHandle) itemhandle, miSymbol);	break;
			}

			GetDItem(win, diPrefsLargeFontPopup, &itemtype, &itemhandle, &itemrect);
			switch (tmp_large_font_id) {
				case 0: 	SetCtlValue((ControlHandle) itemhandle, miChicago);	break;
				case 2: 	SetCtlValue((ControlHandle) itemhandle, miNewYork);	break;
				case 3: 	SetCtlValue((ControlHandle) itemhandle, miGeneva);	break;
				case 4: 	SetCtlValue((ControlHandle) itemhandle, miMonaco);	break;
				case 20: 	SetCtlValue((ControlHandle) itemhandle, miTimes);	break;
				case 21: 	SetCtlValue((ControlHandle) itemhandle, miHelvetica);	break;
				case 22:	SetCtlValue((ControlHandle) itemhandle, miCourier);	break;
				case 23: 	SetCtlValue((ControlHandle) itemhandle, miSymbol);	break;
			}
			/* (should set up font id items here) */
			put_number_into_ditem(diPrefsSmallSizeText, tmp_small_font_size);
			put_number_into_ditem(diPrefsLargeSizeText, tmp_large_font_size);
			/* (should set up starting value of size popups) */
			GetDItem(win, diPrefsSmallFontSample, &itemtype, &itemhandle, &itemrect);
			SetDItem(win, diPrefsSmallFontSample, itemtype, (Handle) draw_font_sample_proc, &itemrect);
			GetDItem(win, diPrefsLargeFontSample, &itemtype, &itemhandle, &itemrect);
			SetDItem(win, diPrefsLargeFontSample, itemtype, (Handle) draw_font_sample_proc, &itemrect);
			break;
		case miPreferencesTopicsOffscreen:
			ditl = GetResource('DITL', dsPreferencesOffscreen);
			AppendDITL(win, ditl, overlayDITL);
			ReleaseResource(ditl);

			GetDItem(win, diPrefsHorizBuff, &itemtype, &itemhandle, &itemrect);
			NumToString(default_max_bufx, tmpstr);
			SetIText(itemhandle, tmpstr);

			GetDItem(win, diPrefsVertiBuff, &itemtype, &itemhandle, &itemrect);
			NumToString(default_max_bufy, tmpstr);
			SetIText(itemhandle, tmpstr);
			GetDItem(win, diPrefs256colors, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, alert_256_colors);
			break;
		case miPreferencesTopicsImageSets:
			ditl = GetResource('DITL', dsPreferencesImageSets);
			AppendDITL(win, ditl, overlayDITL);
			ReleaseResource(ditl);

			GetDItem(win, diPrefsUseStandIcons, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, !pref_color_unit_images);

			GetDItem(win, diPrefsUseColorIcons, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, pref_color_unit_images);
			
			GetDItem(win, diPrefsUnifiedCicns, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, unified_cicns);
			
			GetDItem(win, diPrefsDrawEmblems, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_draw_emblems);

			break;
		case miPreferencesTopicsColorSchemes:
			ditl = GetResource('DITL', dsPreferencesColorSchemes);
			AppendDITL(win, ditl, overlayDITL);
			ReleaseResource(ditl);

			GetDItem(win, diPrefsSplitIcon, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, default_split_icon);

			GetDItem(win, diPrefsSidesPopup, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, 1);

			GetDItem(win, diPrefsFeaturesPopup, &itemtype, &itemhandle, &itemrect);
			SetCtlValue((ControlHandle) itemhandle, 1);

			GetDItem(win, diPrefsMainColor0, &itemtype, &itemhandle, &itemrect);
			if (default_main_icon_color == 0) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMainColor1, &itemtype, &itemhandle, &itemrect);
			if (default_main_icon_color == 1) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMainColor2, &itemtype, &itemhandle, &itemrect);
			if (default_main_icon_color == 2) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMainColor3, &itemtype, &itemhandle, &itemrect);
			if (default_main_icon_color == 3) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsHalfColor0, &itemtype, &itemhandle, &itemrect);
			if (default_half_icon_color == 0) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsHalfColor1, &itemtype, &itemhandle, &itemrect);
			if (default_half_icon_color == 1) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsHalfColor2, &itemtype, &itemhandle, &itemrect);
			if (default_half_icon_color == 2) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsHalfColor3, &itemtype, &itemhandle, &itemrect);
			if (default_half_icon_color == 3) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMaskColor0, &itemtype, &itemhandle, &itemrect);
			if (default_icon_mask_color == 0) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMaskColor1, &itemtype, &itemhandle, &itemrect);
			if (default_icon_mask_color == 1) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMaskColor2, &itemtype, &itemhandle, &itemrect);
			if (default_icon_mask_color == 2) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			GetDItem(win, diPrefsMaskColor3, &itemtype, &itemhandle, &itemrect);
			if (default_icon_mask_color == 3) SetCtlValue((ControlHandle) itemhandle, TRUE);
			else SetCtlValue((ControlHandle) itemhandle, FALSE);

			break;
	}
}
