/* Toplevel code for the tcl/tk interface to Xconq.
   Copyright (C) 1998, 1999 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

#include "conq.h"
extern void notify_instructions(void);
extern int sendnow;
#include "kpublic.h"
#include "tkconq.h"

void place_legends(Side *side);

extern int connection_method;

extern Tk_Window tmp_root_window;
extern int tmp_valid;

Tcl_Interp *interp;

Side *dside;

int lastrawx, lastrawy;

int tk_version_string(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[]);
int tk_copyright_string(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[]);
int tk_run_game(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[]);
int tk_run_game_idle(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int tk_animate_selection(ClientData cldata, Tcl_Interp *interp,
			 int argc, char *argv[]);
int tk_interp_key(ClientData cldata, Tcl_Interp *interp,
		  int argc, char *argv[]);
int tk_execute_long_command(ClientData cldata, Tcl_Interp *interp,
			    int argc, char *argv[]);
int tk_numgames(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[]);
int tk_interpret_variants(ClientData cldata, Tcl_Interp *interp,
			  int argc, char *argv[]);
int tk_set_variant_value(ClientData cldata, Tcl_Interp *interp,
			  int argc, char *argv[]);
int tk_implement_variants(ClientData cldata, Tcl_Interp *interp,
			  int argc, char *argv[]);
int tk_numttypes(ClientData cldata, Tcl_Interp *interp,
		 int argc, char *argv[]);
int tk_numutypes(ClientData cldata, Tcl_Interp *interp,
		 int argc, char *argv[]);
int tk_numsides(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[]);
int tk_maxsides(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[]);
int tk_dside(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[]);
int tk_numfeatures(ClientData cldata, Tcl_Interp *interp,
		   int argc, char *argv[]);
int tk_game_info(ClientData cldata, Tcl_Interp *interp,
		 int argc, char *argv[]);
int tk_ttype_name(ClientData cldata, Tcl_Interp *interp,
		  int argc, char *argv[]);
int tk_t_image_name(ClientData cldata, Tcl_Interp *interp,
		    int argc, char *argv[]);
int tk_utype_name(ClientData cldata, Tcl_Interp *interp,
		  int argc, char *argv[]);
int tk_u_image_name(ClientData cldata, Tcl_Interp *interp,
		    int argc, char *argv[]);
int tk_side_name(ClientData cldata, Tcl_Interp *interp,
		 int argc, char *argv[]);
int tk_side_adjective(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[]);
int tk_side_emblem(ClientData cldata, Tcl_Interp *interp,
		   int argc, char *argv[]);
int tk_short_side_title(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[]);
int tk_side_ingame(ClientData cldata, Tcl_Interp *interp,
		   int argc, char *argv[]);
int tk_long_player_title(ClientData cldata, Tcl_Interp *interp,
			 int argc, char *argv[]);
int tk_player_advantage(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[]);
int tk_min_advantage(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int tk_max_advantage(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int tk_assigned_side(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int tk_assigned_player(ClientData cldata, Tcl_Interp *interp,
		       int argc, char *argv[]);
int tk_can_rename(ClientData cldata, Tcl_Interp *interp,
		  int argc, char *argv[]);
int tk_adjust_advantage(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[]);
int tk_add_side_and_player(ClientData cldata, Tcl_Interp *interp,
			   int argc, char *argv[]);
int tk_rename_side_for_player(ClientData cldata, Tcl_Interp *interp,
			      int argc, char *argv[]);
int tk_set_ai_for_player(ClientData cldata, Tcl_Interp *interp,
			 int argc, char *argv[]);
int tk_exchange_players(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[]);
int tk_feature_name(ClientData cldata, Tcl_Interp *interp,
		    int argc, char *argv[]);
int tk_start_new_game(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[]);
int tk_launch_game(ClientData cldata, Tcl_Interp *interp,
		   int argc, char *argv[]);
int tk_launch_game_2(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int tk_set_unit_type(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int tk_mouse_down_cmd(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[]);
int tk_mouse_up_cmd(ClientData cldata, Tcl_Interp *interp,
		    int argc, char *argv[]);
int tk_mouse_over_cmd(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[]);
int tk_world_mouse_down_cmd(ClientData cldata, Tcl_Interp *interp,
			    int argc, char *argv[]);
int tk_world_mouse_up_cmd(ClientData cldata, Tcl_Interp *interp,
			  int argc, char *argv[]);
int tk_help_goto(ClientData cldata, Tcl_Interp *interp,
		 int argc, char *argv[]);
int tk_set_design_tool(ClientData cldata, Tcl_Interp *interp,
		       int argc, char *argv[]);
int tk_set_design_data(ClientData cldata, Tcl_Interp *interp,
		       int argc, char *argv[]);
int tk_create_new_feature(ClientData cldata, Tcl_Interp *interp,
			  int argc, char *argv[]);
int tk_designer_save(ClientData cldata, Tcl_Interp *interp,
		     int argc, char *argv[]);
int mapw_cmd(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[]);
int imfsample_cmd(ClientData cldata, Tcl_Interp *interp,
		  int argc, char *argv[]);

void update_mouseover(Map *map, int rawx, int rawy);

/* Create the one global interpreter, add Xconq-specific commands to it. */

initial_ui_init()
{
    char pathbuf[500];

    Tk_Window tkwin;

    Tcl_FindExecutable("xconq");

    interp = Tcl_CreateInterp();

    if (Tcl_Init(interp) == TCL_ERROR) {
	init_error("tcl init failed, exiting");
    }

    if (Tk_Init(interp) == TCL_ERROR) {
	init_error("tk init failed, exiting");
    }

    Tcl_CreateCommand(interp, "version_string", tk_version_string,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "copyright_string", tk_copyright_string,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "run_game", tk_run_game,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "run_game_idle", tk_run_game_idle,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "animate_selection", tk_animate_selection,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "interp_key", tk_interp_key,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "execute_long_command", tk_execute_long_command,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "numgames", tk_numgames,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "interpret_variants", tk_interpret_variants,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "set_variant_value", tk_set_variant_value,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "implement_variants", tk_implement_variants,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "numttypes", tk_numttypes,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "numutypes", tk_numutypes,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "numsides", tk_numsides,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "maxsides", tk_maxsides,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "numfeatures", tk_numfeatures,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "game_info", tk_game_info,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ttype_name", tk_ttype_name,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "t_image_name", tk_t_image_name,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "utype_name", tk_utype_name,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "u_image_name", tk_u_image_name,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "side_name", tk_side_name,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "side_adjective", tk_side_adjective,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "side_emblem", tk_side_emblem,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "short_side_title", tk_short_side_title,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "side_ingame", tk_side_ingame,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "long_player_title", tk_long_player_title,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "player_advantage", tk_player_advantage,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "min_advantage", tk_min_advantage,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "max_advantage", tk_max_advantage,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "assigned_side", tk_assigned_side,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "assigned_player", tk_assigned_player,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "can_rename", tk_can_rename,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "adjust_advantage", tk_adjust_advantage,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "add_side_and_player", tk_add_side_and_player,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "rename_side_for_player", tk_rename_side_for_player,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "set_ai_for_player", tk_set_ai_for_player,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "exchange_players", tk_exchange_players,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "dside", tk_dside,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "feature_name", tk_feature_name,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "start_new_game", tk_start_new_game,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "launch_game", tk_launch_game,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "launch_game_2", tk_launch_game_2,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "set_unit_type", tk_set_unit_type,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "mouse_down_cmd", tk_mouse_down_cmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "mouse_up_cmd", tk_mouse_up_cmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "mouse_over_cmd", tk_mouse_over_cmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "world_mouse_down_cmd", tk_world_mouse_down_cmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "world_mouse_up_cmd", tk_world_mouse_up_cmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "help_goto", tk_help_goto,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "set_design_tool", tk_set_design_tool,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "set_design_data", tk_set_design_data,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "create_new_feature", tk_create_new_feature,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "designer_save", tk_designer_save,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    tkwin = Tk_MainWindow(interp);

    Tcl_CreateCommand(interp, "map", mapw_cmd,
		      (ClientData) tkwin, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(interp, "imfsample", imfsample_cmd,
		      (ClientData) tkwin, (Tcl_CmdDeleteProc *) NULL);

    {
	int loaded = FALSE;
	LibraryPath *p;
	FILE *fp;

	for_all_library_paths(p) {
	    make_pathname(p->path, "../tcltk/tkconq", "tcl", pathbuf);
	    if ((fp = fopen(pathbuf, "r")) != NULL) {
		fclose(fp);
		Tcl_EvalFile(interp, pathbuf);
		loaded = TRUE;
		break;
	    }
	    make_pathname(p->path, "../tkconq", "tcl", pathbuf);
	    if ((fp = fopen(pathbuf, "r")) != NULL) {
		fclose(fp);
		Tcl_EvalFile(interp, pathbuf);
		loaded = TRUE;
		break;
	    }
	}
	if (!loaded)
	  init_error("tkconq.tcl file not found");
    }

    imf_interp_hook = tk_interp_imf;
    imf_load_hook = tk_load_imf;
}

ui_init()
{
    int u;
    Side *side;

    eval_tcl_cmd("do_initial_setup");

    for_all_sides(side) {
	if (side->ui)
	  dside = side;
    }

    /* cache the highest feature number; don't forget to update this 
       if new featured are added (presently not implemented)  */
    dside->ui->numfeatures = num_features();
    place_legends(dside);
}

int
tk_numgames(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    collect_possible_games();
    sprintf(interp->result, "%d", numgames);
    return TCL_OK;
}

int
tk_interpret_variants(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    interpret_variants();
    return TCL_OK;
}

int
tk_set_variant_value(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int which, val;

    which = strtol(argv[1], NULL, 10);
    if (which == -1) {
	set_variant_world_size(strtol(argv[2], NULL, 10),
			       strtol(argv[3], NULL, 10),
			       strtol(argv[4], NULL, 10),
			       strtol(argv[5], NULL, 10),
			       strtol(argv[6], NULL, 10));
    } else if (which == -2) {
	set_variant_real_time(strtol(argv[2], NULL, 10),
			      strtol(argv[3], NULL, 10),
			      strtol(argv[4], NULL, 10));
    } else {
	val = strtol(argv[2], NULL, 10);
	set_variant_value(which, val);
    }
    return TCL_OK;
}

int
tk_implement_variants(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    implement_variants();
    return TCL_OK;
}

int
tk_numttypes(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", numttypes);
    return TCL_OK;
}

int
tk_numutypes(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", numutypes);
    return TCL_OK;
}

int
tk_numsides(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", numsides);
    return TCL_OK;
}

int
tk_maxsides(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", g_sides_max());
    return TCL_OK;
}

int
tk_numfeatures(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", num_features());
    return TCL_OK;
}

int
tk_ttype_name(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int t;

    t = strtol(argv[1], NULL, 10);
    if (is_terrain_type(t))
      sprintf(interp->result, "%s", t_type_name(t));
    else
      sprintf(interp->result, "?t?");
    return TCL_OK;
}

int
tk_t_image_name(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int t;
    char *str;

    t = strtol(argv[1], NULL, 10);
    if (is_terrain_type(t)) {
	str = t_image_name(t);
	if (empty_string(str))
	  str = t_type_name(t);
	sprintf(interp->result, "%s", str);
    } else
      sprintf(interp->result, "?t?");
    return TCL_OK;
}

int
tk_utype_name(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int u;

    u = strtol(argv[1], NULL, 10);
    if (is_unit_type(u))
      sprintf(interp->result, "%s", u_type_name(u));
    else
      sprintf(interp->result, "?u?");
    return TCL_OK;
}

int
tk_u_image_name(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int u;
    char *str;

    u = strtol(argv[1], NULL, 10);
    if (is_unit_type(u)) {
	str = u_image_name(u);
	if (empty_string(str))
	  str = u_internal_name(u);
	sprintf(interp->result, "%s", str);
    } else
      sprintf(interp->result, "?u?");
    return TCL_OK;
}

int
tk_side_name(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;

    s = strtol(argv[1], NULL, 10);
    if (between(0, s, numsides))
      sprintf(interp->result, "%s", side_name(side_n(s)));
    else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_side_adjective(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;

    s = strtol(argv[1], NULL, 10);
    if (between(0, s, numsides))
      sprintf(interp->result, "%s", side_adjective(side_n(s)));
    else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_side_emblem(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;

    s = strtol(argv[1], NULL, 10);
    if (s == 0)
      sprintf(interp->result, indepside->emblemname);
    else if (between(1, s, numsides)) {
	/* Try to get the emblem that was actually set up - it may be
	   a default or something else different from the official
	   emblem name. */
	if (dside && dside->ui && dside->ui->eimages[s])
	  sprintf(interp->result, "%s", dside->ui->eimages[s]->name);
	else if (side_n(s)->emblemname)
	  sprintf(interp->result, "%s", (side_n(s))->emblemname);
	else
	  sprintf(interp->result, "null");
    } else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_short_side_title(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;

    s = strtol(argv[1], NULL, 10);
    if (between(0, s, numsides))
      /* (should handle empty string?) */
      sprintf(interp->result, "%s", short_side_title(side_n(s)));
    else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_side_ingame(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;

    s = strtol(argv[1], NULL, 10);
    if (between(1, s, numsides))
      sprintf(interp->result, "%d", side_n(s)->ingame);
    else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_long_player_title(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int p;
    char abuf[300];
    Player *player;

    p = strtol(argv[1], NULL, 10);
    player = find_player(p);
    long_player_title(abuf, player, NULL);
    sprintf(interp->result, "%s", abuf);
    return TCL_OK;
}

int
tk_player_advantage(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int p;
    Player *player;

    p = strtol(argv[1], NULL, 10);
    player = find_player(p);
    sprintf(interp->result, "%d", (player ? player->advantage : 0));
    return TCL_OK;
}

int
tk_min_advantage(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;
    Side *side;

    s = strtol(argv[1], NULL, 10);
    side = side_n(s);
    sprintf(interp->result, "%d", (side ? side->minadvantage : 0));
    return TCL_OK;
}

int
tk_max_advantage(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;
    Side *side;

    s = strtol(argv[1], NULL, 10);
    side = side_n(s);
    sprintf(interp->result, "%d", (side ? side->maxadvantage : 0));
    return TCL_OK;
}

int
tk_assigned_side(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int i;
    Side *side;

    i = strtol(argv[1], NULL, 10);
    if (between(0, i, numsides)) {
	side = assignments[i].side;
	sprintf(interp->result, "%d", side_number(side));
    } else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_assigned_player(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int i;
    Player *player;

    i = strtol(argv[1], NULL, 10);
    if (between(0, i, numsides)) {
	player = assignments[i].player;
	sprintf(interp->result, "%d", (player ? player->id : 0));
    } else
      sprintf(interp->result, "?s?");
    return TCL_OK;
}

int
tk_can_rename(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int s;
    Side *side;

    s = strtol(argv[1], NULL, 10);
    side = side_n(s);
    sprintf(interp->result, "%d",
	    g_side_lib() != lispnil && side != NULL && !side->nameslocked);
    return TCL_OK;
}

int
tk_adjust_advantage(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int p, amt;
    Player *player;

    p = strtol(argv[1], NULL, 10);
    amt = strtol(argv[2], NULL, 10);
    player = assignments[p].player;
    player->advantage += amt;
    return TCL_OK;
}

int
tk_add_side_and_player(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    add_side_and_player();
    /* Return the position of the new side/player in the assignment array. */
    sprintf(interp->result, "%d", numsides - 1);
    return TCL_OK;
}

int
tk_rename_side_for_player(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int n;
    Side *side;

    n = strtol(argv[1], NULL, 10);
    side = assignments[n].side;
    side->name = side->noun = side->pluralnoun = side->adjective = NULL;
    make_up_side_name(side);
    return TCL_OK;
}

int
tk_set_ai_for_player(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int n;
    char *aitype, *mode;
    Player *player;

    n = strtol(argv[1], NULL, 10);
    aitype = copy_string(argv[2]);
    mode = argv[3];
    player = assignments[n].player;
    if (strcmp(mode, "toggle") == 0) {
	if (player->aitypename)
	  player->aitypename = NULL;
	else
	  player->aitypename = aitype;
    }
    /* (should add other cases eventually) */
    return TCL_OK;
}

int
tk_exchange_players(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int n, n2;

    n = strtol(argv[1], NULL, 10);
    n2 = strtol(argv[2], NULL, 10);
    if (between(0, n, numsides)) {
	n2 = exchange_players(n, n2);
	sprintf(interp->result, "%d", n2);
    } else {
	sprintf(interp->result, "?s?");
    }
    return TCL_OK;
}

int
tk_dside(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", dside->id);
    return TCL_OK;
}

int
tk_feature_name(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int fid;
    Feature *feature;

    fid = strtol(argv[1], NULL, 10);
    feature = find_feature(fid);
    if (feature && feature->name)
      sprintf(interp->result, "%s", feature->name);
    else
      sprintf(interp->result, "?f?");
    return TCL_OK;
}

int
tk_version_string(ClientData cldata, Tcl_Interp *interp, int argc,
		  char *argv[])
{
    sprintf(interp->result, "%s", version_string());
    return TCL_OK;
}

int
tk_copyright_string(ClientData cldata, Tcl_Interp *interp, int argc,
		  char *argv[])
{
    sprintf(interp->result, "%s", copyright_string());
    return TCL_OK;
}

int
tk_game_info(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int gamei;
    char *title, *blurb, *bname;

    gamei = strtol(argv[1], NULL, 10);
    title = possible_games[gamei]->title;
    if (title == NULL)
      title = "";
    blurb = possible_games[gamei]->blurb;
    if (blurb == NULL)
      blurb = "";
    if (empty_string(blurb))
      blurb = "(no description)";
    bname = (possible_games[gamei]->basemodulename ? "-  " : "");
    sprintf(interp->result, "{\"%s\" \"%s\" \"%s\" \"%s\"}",
	    possible_games[gamei]->name, title, blurb, bname);
    return TCL_OK;
}

int
tk_start_new_game(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int gamei;

    gamei = strtol(argv[1], NULL, 10);
    mainmodule = possible_games[gamei];
    load_game_module(mainmodule, TRUE);
    check_game_validity();
    return TCL_OK;
}

int
tk_launch_game(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    make_trial_assignments();
    return TCL_OK;
}

int
tk_launch_game_2(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    launch_game();
    ui_init();
    /* Get the displays set up, but don't draw anything yet. */
    init_all_displays();
    /* Now bring up the init data on each display. */
    init_redraws();
    /* Set up the signal handlers. */
    init_signal_handlers();
    /*    notify_all("Command was \"%s %s\"", argv[0], args_used); */
    notify_instructions();
    return TCL_OK;
}

int
tk_run_game(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int maxactions, rslt = 0;
    unsigned long interval;

    maxactions = strtol(argv[1], NULL, 10);

    /* Check for any input from remotes. */
    if (connection_method > 0 && my_rid > 0) {
	flush_outgoing_queue();
	receive_data(0);
	++rslt;
    }
    run_local_ai(1, 20);
    /* Run the kernel itself. */
    rslt += net_run_game(maxactions);
    run_local_ai(2, 20);
    /* Check for any input from remotes. */
    if (connection_method > 0 && my_rid > 0) {
	flush_outgoing_queue();
	receive_data(0);
	++rslt;
    }
    /* Set up to call it again in a little while. */
    /* If things are happening, call 40 times/sec, for responsiveness. */
    interval = 25;
    /* If nothing is happening right now, do at 4 times/sec. */
    if (rslt == 0)
      interval = 250;
    sprintf(interp->result, "%d", interval);
    return TCL_OK;
}

/* (should be per-map var) */
int was_on_screen = FALSE;

int
tk_run_game_idle(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    Map *map;
    Unit *unit, *unit2;

    if (!active_display(dside))
      return TCL_OK;
    /* See if we should jump to another unit and make it current. */
    for_all_maps(map) {
	if (map->widget == NULL)
	  continue;
	unit = map->curunit;
	if (map->autoselect) {
	    if (in_play(unit)
		&& unit->id == map->curunit_id
		&& side_controls_unit(dside, unit)
		&& unit->act
		&& unit->act->acp > 0
		&& (unit->plan ? !unit->plan->asleep : TRUE)
		&& (unit->plan ? !unit->plan->reserve : TRUE)
		&& (unit->plan ? !unit->plan->delayed : TRUE)
		) {
		if (!in_middle(map, unit->x, unit->y) && !was_on_screen) {
		    put_on_screen(map, unit->x, unit->y);
		    was_on_screen = TRUE;
		}
	    } else {
		if (!in_play(unit)
		    || unit->id != map->curunit_id
		    || !side_controls_unit(dside, unit))
		  unit = NULL;
		unit2 = autonext_unit(dside, unit);
		if (unit2 != unit)
		  was_on_screen = FALSE;
		if (unit2 != NULL)
		  set_current_unit(map, unit2);
	    }
	} else {
	    /* Even when not auto-selecting, the selected unit may pass
	       out of our control. */
	    if (!in_play(unit)
		|| unit->id != map->curunit_id
		|| !side_controls_unit(dside, unit))
	      unit = NULL;
	    set_current_unit(map, unit);
	}
	update_mouseover(map, lastrawx, lastrawy);
    }
    return TCL_OK;
}

int
tk_animate_selection(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    Map *map;
    Unit *unit;

    if (active_display(dside)) {
	for_all_maps(map) {
	    if (map->autoselect
		&& map->widget != NULL
		&& in_play(map->curunit)
		&& map->curunit->id == map->curunit_id) {
		map->anim_state = (map->anim_state + 1) % 8;
		update_at_unit(map, map->curunit);
	    }
	}
    }
    return TCL_OK;
}

int
tk_set_unit_type(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int u;

    u = strtol(argv[1], NULL, 10);
    dside->ui->curmap->inptype = u;
    return TCL_OK;
}

int
tk_mouse_down_cmd(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int rawx, rawy, button;

    rawx = strtol(argv[1], NULL, 10);  rawy = strtol(argv[2], NULL, 10);
    button = strtol(argv[3], NULL, 10);
    DGprintf("down %d %d %d\n", rawx, rawy, button);
    handle_mouse_down(dside->ui->curmap, rawx, rawy, button);
    return TCL_OK;
}

int
tk_mouse_up_cmd(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int rawx, rawy, button;

    rawx = strtol(argv[1], NULL, 10);  rawy = strtol(argv[2], NULL, 10);
    button = strtol(argv[3], NULL, 10);
    DGprintf("down %d %d %d\n", rawx, rawy, button);
    handle_mouse_up(dside->ui->curmap, rawx, rawy, button);
    return TCL_OK;
}

int
tk_mouse_over_cmd(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int rawx, rawy;

    rawx = strtol(argv[1], NULL, 10);  rawy = strtol(argv[2], NULL, 10);
    if (dside->designer)
      paint_on_drag(dside->ui->curmap, rawx, rawy);
    update_mouseover(dside->ui->curmap, rawx, rawy);
    lastrawx = rawx;  lastrawy = rawy;
    return TCL_OK;
}

void
update_mouseover(Map *map, int rawx, int rawy)
{
    DGprintf("over %d %d\n", rawx, rawy);
    tmpbuf[0] = '\0';
    if (rawx >= 0 || rawy >= 0)
      oneliner(dside, widget_vp(map), rawx, rawy);
    eval_tcl_cmd("update_mouseover \"%s\"", tmpbuf);
}

int
tk_world_mouse_down_cmd(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[])
{
    int rawx, rawy, button;

    rawx = strtol(argv[1], NULL, 10);  rawy = strtol(argv[2], NULL, 10);
    button = strtol(argv[3], NULL, 10);
    DGprintf("world down %d %d %d\n", rawx, rawy, button);
    handle_world_mouse_down(dside->ui->curmap, rawx, rawy, button);
    return TCL_OK;
}

int
tk_world_mouse_up_cmd(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[])
{
    int rawx, rawy, button;

    rawx = strtol(argv[1], NULL, 10);  rawy = strtol(argv[2], NULL, 10);
    button = strtol(argv[3], NULL, 10);
    DGprintf("world up %d %d %d\n", rawx, rawy, button);
    handle_world_mouse_up(dside->ui->curmap, rawx, rawy, button);
    return TCL_OK;
}

set_current_unit(Map *map, Unit *unit)
{
    Unit *oldunit = map->curunit;

    if (unit == oldunit)
      return;
    if (unit == NULL || (in_play(unit) && side_controls_unit(dside, unit))) {
	map->curunit = unit;
	map->curunit_id = (unit ? unit->id : 0);
    }
    /* Make sure the unit is actually visible on-screen. */
    if (unit != NULL) {
	put_on_screen(map, unit->x, unit->y);
    }
    /* (should only do this if map not scrolled) */
    if (oldunit)
      update_at_unit(map, oldunit);
    if (map->curunit)
      update_at_unit(map, map->curunit);
    draw_unit_info(map);
}

draw_unit_info(Map *map)
{
    char infobuf[BUFSIZE];
    int u, s, t, i, mrow, x = -1, y = -1, uview, len;
    Unit *unit;
    Task *task;

    unit = map->curunit;
    if (!in_play(unit)) {
	eval_tcl_cmd("update_unit_info curunit 0");
	eval_tcl_cmd("update_unit_info handle \"(no unit)\"");
	eval_tcl_cmd("update_unit_info loc \"\"");
	eval_tcl_cmd("update_unit_info occ \"\"");
	eval_tcl_cmd("update_unit_info hp \"\"");
	eval_tcl_cmd("update_unit_info stack \"\"");
	eval_tcl_cmd("update_unit_info s0 \"\"");
	eval_tcl_cmd("update_unit_info s1 \"\"");
	eval_tcl_cmd("update_unit_info s2 \"\"");
	eval_tcl_cmd("update_unit_info s3 \"\"");
	eval_tcl_cmd("update_unit_info plan \"\"");
	eval_tcl_cmd("update_unit_info t0 \"\"");
	eval_tcl_cmd("update_unit_info t1 \"\"");
	return;
    }
    u = unit->type;
    eval_tcl_cmd("update_unit_info curunit %d", unit->id);
    /* Say which unit this is. */
    eval_tcl_cmd("update_unit_info handle \"%s\"", unit_handle(dside, unit));
    x = unit->x;  y = unit->y;
    location_desc(infobuf, dside, unit, u, x, y);
    eval_tcl_cmd("update_unit_info loc \"%s\"", infobuf);
    /* Very briefly list the numbers and types of the occupants. */
    occupants_desc(infobuf, unit);
    eval_tcl_cmd("update_unit_info occ \"%s\"", infobuf);
    /* Display the "important" parameters. */
    /* (should say something about parts?) */
    hp_desc(infobuf, unit, TRUE);
    strcat(infobuf, "   ");
    acp_desc(tmpbuf, unit, TRUE);
    strcat(infobuf, tmpbuf);
    cxp_desc(tmpbuf, unit, TRUE);
    strcat(infobuf, tmpbuf);
    morale_desc(tmpbuf, unit, TRUE);
    strcat(infobuf, tmpbuf);
    eval_tcl_cmd("update_unit_info hp \"%s\"", infobuf);
    /* List other stack members here. */
    others_here_desc(infobuf, unit);
    eval_tcl_cmd("update_unit_info stack \"%s\"", infobuf);
    /* Describe the state of all the supplies. */
    for (mrow = 0; mrow < 4; ++mrow) {
	supply_desc(infobuf, unit, mrow);
	eval_tcl_cmd("update_unit_info s%d \"%s\"", mrow, infobuf);
    }
    /* Describe the current plan and task agenda. */
    plan_desc(infobuf, unit);
    eval_tcl_cmd("update_unit_info plan \"%s\"", infobuf);
    task = (unit->plan ? unit->plan->tasks : NULL);
    for (i = 0; i < 2; ++i) {
	task_desc(infobuf, unit->side, task);
	eval_tcl_cmd("update_unit_info t%d \"%s\"", i, infobuf);
	if (task)
	  task = task->next;
    }
}
    
ui_mainloop()
{
    Tk_MainLoop();
}

/* All update_xxx_display callbacks are here. */

void
update_area_display(Side *side)
{
    Map *map;

    if (!active_display(side))
      return;
    for_all_maps(map)
      redraw_map(map);
}

/* Draw an individual detailed hex, as a row of one, on all maps. */

void
update_cell_display(Side *side, int x, int y, int flags)
{
    Map *map;

    if (!active_display(side))
      return;
    for_all_maps(map)
      update_at_cell(map, x, y, flags);
}

/* The kernel calls this to update info about the given side. */

void
update_side_display(Side *side, Side *side2, int rightnow)
{
    int sx, sy, totacp, percentleft, percentresv, activ;
    char sidebuf[BUFSIZE];
    extern int curpriority;

    if (!active_display(side))
      return;
    if (side2 == NULL)
      return;

    /* Build up and write the textual description of the side. */
    sidebuf[0] = '\0';
#ifdef DESIGNERS
    if (side2->designer)
      strcat(sidebuf, "(designer)");
#endif /* DESIGNERS */
    strcat(sidebuf, short_side_title(side2));
    if (side2->willingtodraw) {
	strcat(sidebuf, "[draw]");
    }
    if (side2->player) {
	strcat(sidebuf, "(");
	short_player_title(sidebuf+strlen(sidebuf), side2->player, NULL);
	strcat(sidebuf, ")");
    }
    eval_tcl_cmd("update_game_side_info %d \"%s\" %d %d %d",
		 side_number(side2), sidebuf, side2->everingame, side2->ingame,
		 side2->status);
    if (keeping_score() && side2->everingame) {
	int i;
	char *scoredesc;
	Scorekeeper *sk;

	i = 0;
	for_all_scorekeepers(sk) {
	    scoredesc = side_score_desc(spbuf, side2, sk);
	    eval_tcl_cmd("update_game_side_score score%d_%d \"%s\"",
			 i, side_number(side2), scoredesc);
	    ++i;
	}
    }
    totacp = percentleft = percentresv = 0;
    if (side2->ingame && !endofgame) {
	totacp = side_initacp(side2);
	if (totacp > 0) {
	    percentleft = (100 * side_acp(side2)) / totacp;
	    percentleft = limitn(0, percentleft, 100);
	    percentresv = (100 * side_acp_reserved(side2)) / totacp;
	    percentresv = limitn(0, percentresv, percentleft);
	}
    }
    activ = (!g_use_side_priority() || curpriority == side2->priority);
    eval_tcl_cmd("update_side_progress %d %d %d %d",
		 side_number(side2), activ, percentleft, percentresv);
}

/* The kernel calls this to update info about the given unit. */

void
update_unit_display(Side *side, Unit *unit, int rightnow)
{
    Map *map;

    if (!active_display(side))
      return;
    if (unit == NULL)
      return;
    if (inside_area(unit->x, unit->y)) {
	update_cell_display(side, unit->x, unit->y, rightnow);
    }
    for_all_maps(map)
      draw_unit_info(map);
    update_side_display(side, unit->side, rightnow);
    update_unit_type_list(side, unit->type);
}

void
update_unit_acp_display(Side *side, Unit *unit, int rightnow)
{
    if (!active_display(side))
      return;

    update_unit_type_list(side, unit->type);
}

void
update_action_result_display(Side *side, Unit *unit, int rslt, int rightnow)
{
    if (!active_display(side))
      return;

    update_unit_type_list(side, unit->type);
}

/* The kernel calls this to update the global game state. */

int told_outcome;

void
update_turn_display(Side *side, int rightnow)
{
    Map *map;

    if (!active_display(side))
      return;

    eval_tcl_cmd("update_game_state \"%s\"", curdatestr);
    if (endofgame && !told_outcome && side == dside) {
	dside->may_set_see_all = TRUE;
	update_view_controls_info();
	if (!dside->see_all) {
	    dside->see_all = TRUE;
	    for_all_maps(map) {
		map->see_all = TRUE;
		redraw_map(map);
	    }
#if 0
	    if (side_won(dside)) {
		won_game_dialog();
	    } else if (side_lost(dside)) {
		lost_game_dialog();
	    } else {
		game_over_dialog();
	    }
#endif
	}
	told_outcome = TRUE;
    }
}

void
update_action_display(Side *side, int rightnow)
{
}

void
update_event_display(side, hevt, rightnow)
Side *side;
HistEvent *hevt;
int rightnow;
{
}

void
update_fire_at_display(side, unit, unit2, m, rightnow)
Side *side;
Unit *unit, *unit2;
int m, rightnow;
{
    int i;
    Map *map;

    if (!active_display(side))
      return;
    for_all_maps(map) {
	draw_fire_line(map, unit, unit2, 0, 0);
    }
}

/* This is for animation of fire-into actions. */

void
update_fire_into_display(side, unit, x, y, z, m, rightnow)
Side *side;
Unit *unit;
int x, y, z, m, rightnow;
{
    int i;
    Map *map;

    if (!active_display(side))
      return;
    for_all_maps(map) {
	draw_fire_line(map, unit, NULL, x, y);
    }
}

/* Updates to clock need to be sure that display changes immediately. */

void
update_clock_display(Side *side, int rightnow)
{
}

void
update_message_display(Side *side, Side *sender, char *str, int rightnow)
{
    if (!active_display(side))
      return;

    notify(side, "From %s: \"%s\"",
	   (sender != NULL ? short_side_title(sender) : "<anon>"), str);
}

void
update_all_progress_displays(char *str, int s)
{
}

void
action_point(Side *side, int x, int y)
{
    Map *map;

    if (!active_display(side))
      return;
    if (!inside_area(x, y))
      return;

    for_all_maps(map) {
	if (map->follow_action && !in_middle(map, x, y)) {
	    put_on_screen(map, x, y);
	}
    }
}

void
flush_display_buffers(Side *side)
{
}

void
update_everything()
{
}

/* This is called by generic Unix crash-handling code, no need for
   us to do anything special. */

void
close_displays()
{
}

int
schedule_movie(Side *side, char *movie, ...)
{
    va_list ap;
    int i;
    enum movie_type itype;

    if (!active_display(side))
      return FALSE;
    if (side->ui->numscheduled >= 10)
      return FALSE;
    memset(&(side->ui->movies[side->ui->numscheduled]), 0, sizeof(struct a_movie));
    side->ui->movies[side->ui->numscheduled].type = movie;
    itype = movie_null;
    if (strcmp(movie, "miss") == 0)
      itype = movie_miss;
    else if (strcmp(movie, "hit") == 0)
      itype = movie_hit;
    else if (strcmp(movie, "death") == 0)
      itype = movie_death;
    else if (strcmp(movie, "nuke") == 0)
      itype = movie_nuke;
    else if (strcmp(movie, "sound") == 0)
      itype = movie_sound;
    else if (strcmp(movie, "flash") == 0)
      itype = movie_flash;
    side->ui->movies[side->ui->numscheduled].itype = itype;
    va_start(ap, movie);
    for (i = 0; i < 5; ++i)
      side->ui->movies[side->ui->numscheduled].args[i] = va_arg(ap, int);
    va_end(ap);
    ++side->ui->numscheduled;
    return TRUE;
}

void
play_movies(SideMask sidemask)
{
    int j, unitid, sx, sy, sw, sh, btype;
    Map *map;
    Unit *unit;
    Side *side;

    if (!active_display(dside))
      return;
    if (side_in_set(dside, sidemask)) {
	for (j = 0; j < dside->ui->numscheduled; ++j) {
	    btype = -1;
	    switch (dside->ui->movies[j].itype) {
	      case movie_null:
		break;
	      case movie_miss:
		if (btype < 0)
		  btype = 0;
	      case movie_hit:
		if (btype < 0)
		  btype = 1;
	      case movie_death:
		if (btype < 0)
		  btype = 2;
		unitid = dside->ui->movies[j].args[0];
		unit = find_unit(unitid);
		if (unit == NULL || !in_area(unit->x, unit->y))
		  continue;
		for_all_maps(map) {
		    draw_blast(map, unit, btype);
		}
		break;
	      case movie_nuke:
		break;
	      case movie_flash:
		unitid = dside->ui->movies[j].args[0];
		unit = find_unit(unitid);
		if (unit == NULL || !in_area(unit->x, unit->y))
		  continue;
		for_all_maps(map) {
		    draw_blast(map, unit, 3);
		}
		break;
	      default:
		break;
	    }
	}
	dside->ui->numscheduled = 0;
    }
}

/* Beep the beeper! */

void
beep(Side *side)
{
    eval_tcl_cmd("bell");
}

void
low_notify(Side *side, char *str)
{
    eval_tcl_cmd("low_notify \"%s\n\"", str);
}

popup_game_dialog()
{
    eval_tcl_cmd("popup_splash_screen");
}

check_player_displays() {}

init_redraws()
{
    int u;
    Side *side2;
    Map *map;

    if (dside->ui != NULL) {
	/* The moment of truth - up to now output has been suppressed. */
	dside->ui->active = TRUE;
    }

    for_all_sides(side2) {
	update_side_display(dside, side2, TRUE);
    }
    for_all_unit_types(u) {
	update_unit_type_list(dside, u);
    }
    update_turn_display(dside, TRUE);

    for_all_maps(map)
      set_tool_cursor(dside, map);
}

update_unit_type_list(Side *side, int u)
{
    int i, len, num;
    extern int longest_shortest;

    if (!between(0, u, numutypes))
      return;
    /* Only show a char for the unit type if all have single chars,
       otherwise blank all. */
    eval_tcl_cmd("update_unitlist_name %d \"%c\"",
		 u, (longest_shortest == 1 ? *(shortest_unique_name(u)) : ' '));
    /* Our unit total (right-justified) */
    spbuf[0] = '\0';
    num = num_units_in_play(side, u);
    if (num > 0) {
	sprintf(spbuf, "%d", num);
    }
    eval_tcl_cmd("update_unitlist_count %d \"%s\"", u, spbuf);
    /* Our units under construction (left-justified) */
    num = num_units_incomplete(side, u);
    spbuf[0] = '\0';
    if (num > 0) {
	sprintf(spbuf, "(%d)", num);
    }
    eval_tcl_cmd("update_unitlist_incomplete %d \"%s\"", u, spbuf);
    /* (should do other columns too) */
}

popup_help(Side *side, HelpNode *node)
{
    eval_tcl_cmd("create_help_window");
}

HelpNode *cur_help_node;

static void describe_map PARAMS ((int arg, char *key, TextBuffer *buf));

static void
describe_map(arg, key, buf)
int arg;
char *key;
TextBuffer *buf;
{
    tbcat(buf, "In move mode:\n");
    tbcat(buf, "  The next unit that can do anything will be selected automatically.\n");
    tbcat(buf, "  Left-click on a destination to move the selected unit there.\n");
#if 0
    tbcat(buf, "  To select a unit, center-click it.\n");
#endif
    tbcat(buf, "\n");
    tbcat(buf, "In survey mode:\n");
    tbcat(buf, "  Left-click on a unit to make the current one.\n");
    tbcat(buf, "  To move it, right-click, use 'm'ove command or 'z' to switch back ");
    tbcat(buf, "to move mode.\n");
    tbcat(buf, "\n");
#if 0
#ifdef DESIGNERS
    tbcat(buf, "In design mode:\n");
    tbcat(buf, "  Left-click to perform the tool-specific action: add unit or paint ");
    tbcat(buf, "terrain, population, or feature.\n");
    tbcat(buf, "  Shift-left-click to perform the tool-specific current object ");
    tbcat(buf, "selection: pick current unit type, terrain type, side, or feature.\n");
    tbcat(buf, "\n");
#endif /* DESIGNERS */
    tbcat(buf, "In any mode:\n");
    tbcat(buf, "  Center-click on a unit or cell to make the current one.\n");
    tbcat(buf, "  Drag-shift-center-click to measure distance (gasp! hold down Shift, ");
    tbcat(buf, "press center button at origin, drag to destination, release button ");
    tbcat(buf, "and Shift).\n");
    tbcat(buf, "  Right-click on a destination to move the selected unit there.\n");
    tbcat(buf, "  Move the mouse while holding Meta to change the current cell.\n");
    tbcat(buf, "  Press a keypad key to scroll the look.\n");
#endif
}

int
tk_help_goto(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    char *arg = argv[1];
    char *nclassname;
    char tclbuf[15000];
    HelpNode *node, *tmp;

    /* If this is the first time we came here, build up the table of
       contents. */
    if (cur_help_node == NULL) {
	key_commands_help_node =
	  add_help_node("commands", describe_key_commands, 0, first_help_node);
	long_commands_help_node =
	  add_help_node("long commands", describe_long_commands, 0,
			key_commands_help_node);
	add_help_node("map", describe_map, 0, first_help_node);
	eval_tcl_cmd("add_help_topic_key \"%s\"", first_help_node->key);
	for (tmp = first_help_node->next; tmp != first_help_node; tmp = tmp->next) {
	    eval_tcl_cmd("add_help_topic_key \"%s\"", tmp->key);
	}
	cur_help_node = first_help_node;
    }
    if (strcmp(arg, "help") == 0) {
	/* (should go to help node) */
    } else if (strcmp(arg, "prev") == 0) {
	cur_help_node = cur_help_node->prev;
    } else if (strcmp(arg, "next") == 0) {
	cur_help_node = cur_help_node->next;
    } else if (strcmp(arg, "back") == 0) {
	/* (should go back) */
    } else {
	node = find_help_node(cur_help_node, arg);
	if (node)
	  cur_help_node = node;
	else
	  beep(dside);
    }
    get_help_text(cur_help_node);
    /* Make a string representing the node class. */
    switch (cur_help_node->nclass) {
      case utypenode:
	nclassname = "u";
	break;
      case mtypenode:
	nclassname = "m";
	break;
      case ttypenode:
	nclassname = "t";
	break;
      case atypenode:
	nclassname = "a";
	break;
      default:
	nclassname = "?";
    }
    sprintf (tclbuf, "update_help {%s} {%s} {%s} %d",
	     cur_help_node->key, cur_help_node->text, nclassname,
	     cur_help_node->arg);
    Tcl_Eval(interp, tclbuf);
    return TCL_OK;
}

/* Given a typed character, decide what to do with it. */

int
tk_interp_key(ClientData cldata, Tcl_Interp *interp, int argc, char *argv[])
{
    int rawx, rawy, winx, winy, x, y, cancelled;
    Map *map = dside->ui->curmap;
    Tk_Window tkwin;
    void (*fn)(Side *sidex, Map *mapx, int cancelledx);

    map->inpch = *(argv[1]);
    rawx = strtol(argv[2], NULL, 10);  rawy = strtol(argv[3], NULL, 10);
    tkwin = Tk_CoordsToWindow(rawx, rawy, Tk_MainWindow(interp));
    map->inpsx = map->inpsy = -1;
    map->inpx = map->inpy = -1;
    if (tkwin) {
	Tk_GetRootCoords(tkwin, &winx, &winy);
	map->inpsx = rawx - winx;  map->inpsy = rawy - winy;
	if (nearest_cell(widget_vp(map), map->inpsx, map->inpsy,
			 &x, &y, NULL, NULL)) {
	    map->inpx = x;  map->inpy = y;
	}
    }
    DGprintf("key %d %d -> %d %d\n", map->inpsx, map->inpsy, map->inpx, map->inpy);

    if (isdigit(map->inpch)) {
	if (map->prefixarg < 0) {
	    map->prefixarg = 0;
	} else {
	    map->prefixarg *= 10;
	}
	map->prefixarg += (map->inpch - '0');
	sprintf(interp->result, "%d", map->prefixarg);
	return TCL_OK;
    }
    /* Call the modal handler if defined, giving it side and cancel
       flag. */
    if (map->modalhandler) {
	fn = map->modalhandler;
	cancelled = (map->inpch == ESCAPE_CHAR);
	/* Remove the handler - will restore itself if needed. */
	map->modalhandler = NULL;
	(*fn)(dside, map, cancelled);
	if (cancelled) {
	    eval_tcl_cmd("clear_command_line");
	    notify(dside, "Cancelled.");
	}
    } else {
	dside->prefixarg = map->prefixarg;
	dside->ui->beepcount = 0;
	execute_command(dside, map->inpch);
    }
    /* Reset the prefix arg unless there is still more modal input
       to be read. */
    if (map->modalhandler == NULL)
      map->prefixarg = -1;
    sprintf(interp->result, "%d", map->prefixarg);
    return TCL_OK;
}

int
tk_execute_long_command(ClientData cldata, Tcl_Interp *interp,
			int argc, char *argv[])
{
    char *cmd = argv[1], *ncmd;

    /* If the long command has a numeric prefix, take it to be the
       prefix argument and peel off.  */
    if (isdigit(*cmd)) {
        dside->ui->curmap->prefixarg = strtol(cmd, &ncmd, 10);
	cmd = cmd;
    }
    execute_long_command(dside, cmd);
    return TCL_OK;
}

int
tk_set_design_tool(ClientData cldata, Tcl_Interp *interp,
		   int argc, char *argv[])
{
    char *arg = argv[1];
    Map *map;

    if (strcmp(arg, "normal") == 0) {
	dside->ui->curdesigntool = survey_mode;
    } else if (strcmp(arg, "terrain") == 0) {
	/* (should choose subtype) */
	dside->ui->curdesigntool = cell_paint_mode;
    } else if (strcmp(arg, "unit") == 0) {
	dside->ui->curdesigntool = unit_paint_mode;
    } else if (strcmp(arg, "people") == 0) {
	dside->ui->curdesigntool = people_paint_mode;
    } else if (strcmp(arg, "control") == 0) {
	dside->ui->curdesigntool = control_paint_mode;
    } else if (strcmp(arg, "feature") == 0) {
	dside->ui->curdesigntool = feature_paint_mode;
    } else if (strcmp(arg, "material") == 0) {
	dside->ui->curdesigntool = material_paint_mode;
    } else if (strcmp(arg, "elevation") == 0) {
	dside->ui->curdesigntool = elevation_paint_mode;
    } else if (strcmp(arg, "temperature") == 0) {
	dside->ui->curdesigntool = temperature_paint_mode;
    } else if (strcmp(arg, "clouds") == 0) {
	dside->ui->curdesigntool = clouds_paint_mode;
    } else if (strcmp(arg, "winds") == 0) {
	dside->ui->curdesigntool = winds_paint_mode;
    } else if (strcmp(arg, "view") == 0) {
	dside->ui->curdesigntool = view_paint_mode;
    } else {
	run_warning("bogus design tool %s, ignoring\n", arg);
	return TCL_OK;
    }
    for_all_maps(map) {
	map->mode = dside->ui->curdesigntool;
	set_tool_cursor(dside, map);
    }
    return TCL_OK;
}

/* This macro implements cycling of a variable through a set of consecutive
   values, with direction controlled by the shift key.  If the limit is 0,
   then the cycling part is not done. */

#define OPTION_CYCLE(var, lo, hi, n, dir)  \
  if ((hi) - (lo) > 0) {  \
    (var) = (((var) + ((dir) < 0 ? -(n) : (n)) - (lo) + ((hi) - (lo))) % ((hi) - (lo))) + (lo);  \
  } else {  \
    (var) = ((var) + ((dir) < 0 ? -(n) : (n)));  \
  }

int
tk_set_design_data(ClientData cldata, Tcl_Interp *interp,
		   int argc, char *argv[])
{
    int set, dir, val;
    char *type = argv[1];

    if (strcmp(argv[2], "incr") == 0) {
	set = FALSE;
	dir = 1;
    } else if (strcmp(argv[2], "decr") == 0) {
	set = FALSE;
	dir = -1;
    } else {
	set = TRUE;
	val = strtol(argv[2], NULL, 10);
    }

    if (strcmp(type, "curttype") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curttype, 0, numttypes, 1, dir);
	    val = dside->ui->curttype;
	}
	dside->ui->curttype = val;
    } else if (strcmp(type, "curbgttype") == 0) {
	dside->ui->curbgttype = val;
    } else if (strcmp(type, "curutype") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curutype, 0, numutypes, 1, dir);
	    val = dside->ui->curutype;
	}
	dside->ui->curutype = val;
    } else if (strcmp(type, "curusidenumber") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curusidenumber, 0, numsides + 1, 1, dir);
	    val = dside->ui->curusidenumber;
	}
	dside->ui->curusidenumber = val;
    } else if (strcmp(type, "curpeoplenumber") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curpeoplenumber, 0, numsides + 1, 1, dir);
	    val = dside->ui->curpeoplenumber;
	}
	dside->ui->curpeoplenumber = val;
    } else if (strcmp(type, "curcontrolnumber") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curcontrolnumber, 0, numsides + 1, 1, dir);
	    val = dside->ui->curcontrolnumber;
	}
	dside->ui->curcontrolnumber = val;
    } else if (strcmp(type, "curfid") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curfid, 0, num_features(), 1, dir);
	    val = dside->ui->curfid;
	}
	dside->ui->curfid = val;
    } else if (strcmp(type, "curbrushradius") == 0) {
	if (!set) {
	    OPTION_CYCLE(dside->ui->curbrushradius, 0, 11, 1, dir);
	    val = dside->ui->curbrushradius;
	}
	dside->ui->curbrushradius = val;
    }
    set_tool_cursor(dside, dside->ui->curmap);
    sprintf(interp->result, "%d", val);
    return TCL_OK;
}

int
tk_create_new_feature(ClientData cldata, Tcl_Interp *interp,
		      int argc, char *argv[])
{
    extern int nextfid;
    Feature *feature;

    sprintf(spbuf, "%d", nextfid);
    feature = net_create_feature("feature", copy_string(spbuf));
    sprintf(interp->result, "%d", (feature ? feature->id : 0));
    return TCL_OK;
}

int
tk_designer_save(ClientData cldata, Tcl_Interp *interp,
		 int argc, char *argv[])
{
    Module *module;

    module = create_game_module(NULL);
    module->title = "Designer-saved data";
    module->compress_layers = TRUE;
    init_module_reshape(module);

    module->filename = copy_string(argv[1]);
    if (strstr(argv[2], " !compress "))
      module->compress_layers = FALSE;
    if (strstr(argv[2], " all "))
      module->def_all = TRUE;
    if (strstr(argv[2], " types "))
      module->def_types = TRUE;
    if (strstr(argv[2], " tables "))
      module->def_tables = TRUE;
    if (strstr(argv[2], " globals "))
      module->def_globals = TRUE;
    if (strstr(argv[2], " world "))
      module->def_world = TRUE;
    if (strstr(argv[2], " area "))
      module->def_areas = TRUE;
    if (strstr(argv[2], " terrain "))
      module->def_area_terrain = TRUE;
    if (strstr(argv[2], " areamisc "))
      module->def_area_misc = TRUE;
    if (strstr(argv[2], " weather "))
      module->def_area_weather = TRUE;
    if (strstr(argv[2], " material "))
      module->def_area_material = TRUE;
    if (strstr(argv[2], " sides "))
      module->def_sides = TRUE;
    if (strstr(argv[2], " views "))
      module->def_side_views = TRUE;
    if (strstr(argv[2], " units "))
      module->def_units = TRUE;
    if (!write_game_module(module))
      run_warning("Could not write the module \"%s\"!", module->filename);
    return TCL_OK;
}

void
place_legends(side)
Side *side;
{
    int nf = side->ui->numfeatures;

    if (!features_defined() || nf <= 0)
      return;
    if (side->ui->legends == NULL)
      side->ui->legends = (Legend *) xmalloc((nf + 1) * sizeof(Legend));
    place_feature_legends(side->ui->legends, nf, side, 0, 1);
}

void
eval_tcl_cmd(char *fmt, ...)
{
    char tclbuf[500];
    va_list ap;

    if (empty_string(fmt))
      return;

    va_start(ap, fmt);
    vsprintf(tclbuf, fmt, ap);
    va_end(ap);
    Tcl_Eval(interp, tclbuf);
}

ImageFamily *
hack_find_imf(char *name)
{
    int u, t, s;
    ImageFamily *imf;

    if (dside == NULL || dside->ui == NULL) {
	Tk_Window tkwin = Tk_MainWindow(interp);

	tmp_root_window = tkwin;
	tmp_valid = TRUE;
	imf = get_generic_images(dside, name);
	tmp_valid = FALSE;
	return imf;
    }
    for (u = 0; u < numutypes; ++u) {
	if (dside->ui->uimages[u]
	    && strcmp(dside->ui->uimages[u]->name, name) == 0)
	  return dside->ui->uimages[u];
    }
    for (t = 0; t < numttypes; ++t) {
	if (dside->ui->timages[t]
	    && strcmp(dside->ui->timages[t]->name, name) == 0)
	  return dside->ui->timages[t];
    }
    for (s = 0; s <= numsides; ++s) {
	if (dside->ui->eimages[s]
	    && strcmp(dside->ui->eimages[s]->name, name) == 0)
	  return dside->ui->eimages[s];
    }
    return NULL;
}

/* True if any variants are available. */

int any_variants;

/* These are true when a standard variant is available to be chosen. */

int vary_world;
int vary_real_time;

/* These are the values that will be used to set world geometry. */

int new_circumference;
int new_width;
int new_height;
int new_latitude;
int new_longitude;
int new_scale;

int new_time_for_game;
int new_time_per_side;
int new_time_per_turn;

#define MAXCHECKBOXES 10

int numcheckboxes = MAXCHECKBOXES;

Variant *checkboxes[MAXCHECKBOXES];

int checkboxvalues[MAXCHECKBOXES];

Obj *variants;

/* Go through all the game's variants and set up appropriate flags. */

/*static void*/
interpret_variants()
{
    int i;
    char *vartypename;
    Obj *vartmp;
    Variant *var;

    any_variants = FALSE;
    for (i = 0; i < MAXCHECKBOXES; ++i) {
	eval_tcl_cmd("set varianttext(%d) Unused", i);
	eval_tcl_cmd("set variantstate(%d) disabled", i);
    }
    vary_world = vary_real_time = FALSE;
    if (mainmodule == NULL || mainmodule->variants == NULL)
      return;
    numcheckboxes = 3;
    for (i = 0; mainmodule->variants[i].id != lispnil; ++i) {
	var = &(mainmodule->variants[i]);
	any_variants = TRUE;
	vartypename = c_string(var->id);
	switch (keyword_code(vartypename)) {
	  case K_WORLD_SEEN:
	    interpret_checkbox(var, 0);
	    break;
	  case K_SEE_ALL:
	    interpret_checkbox(var, 1);
	    break;
	  case K_SEQUENTIAL:
	    interpret_checkbox(var, 2);
	    break;
	  case K_WORLD_SIZE:
	    /* If the area is already set up, it's too late. */
	    if (area.width > 0 || area.height > 0)
	      break;
	    vary_world = TRUE;
	    /* Start with some defaults. */
	    new_circumference = DEFAULTCIRCUMFERENCE;
	    new_width = DEFAULTWIDTH;  new_height = DEFAULTHEIGHT;
	    new_latitude = 0;  new_longitude = 0;
	    /* If we have explicit defaults, use them. */
	    if (var->dflt != lispnil) {
		vartmp = var->dflt;
		new_width = c_number(eval(car(vartmp)));
		vartmp = cdr(vartmp);
		if (vartmp != lispnil) {
		    new_height = c_number(eval(car(vartmp)));
		    vartmp = cdr(vartmp);
		} else {
		    new_height = new_width;
		}
		if (vartmp != lispnil) {
		    new_circumference = c_number(eval(car(vartmp)));
		    vartmp = cdr(vartmp);
		}
		if (vartmp != lispnil) {
		    new_latitude = c_number(eval(car(vartmp)));
		    vartmp = cdr(vartmp);
		}
		if (vartmp != lispnil) {
		    new_longitude = c_number(eval(car(vartmp)));
		}
	    }
	    eval_tcl_cmd("set new_width %d", new_width);
	    eval_tcl_cmd("set new_height %d", new_height);
	    eval_tcl_cmd("set new_circumference %d", new_circumference);
	    eval_tcl_cmd("set new_latitude %d", new_latitude);
	    eval_tcl_cmd("set new_longitude %d", new_longitude);
	    break;
	  case K_REAL_TIME:
	    vary_real_time = TRUE;
	    /* Start with some defaults. */
	    new_time_for_game = new_time_per_side = new_time_per_turn = 0;
	    /* If we have explicit defaults, use them. */
	    if (var->dflt != lispnil) {
		vartmp = var->dflt;
		new_time_for_game = c_number(eval(car(vartmp)));
		vartmp = cdr(vartmp);
		if (vartmp != lispnil) {
		    new_time_per_side = c_number(eval(car(vartmp)));
		} else {
		    new_time_per_side = 0;
		}
		vartmp = cdr(vartmp);
		if (vartmp != lispnil) {
		    new_time_per_turn = c_number(eval(car(vartmp)));
		} else {
		    new_time_per_turn = 0;
		}
	    }
	    eval_tcl_cmd("set new_time_for_game %d", new_time_for_game);
	    eval_tcl_cmd("set new_time_per_side %d", new_time_per_side);
	    eval_tcl_cmd("set new_time_per_turn %d", new_time_per_turn);
	    break;
	  default:
	    if (numcheckboxes >= MAXCHECKBOXES) {
		init_warning("too many variants, can't set all of them");
		break;
	    }
	    interpret_checkbox(var, numcheckboxes);
	    ++numcheckboxes;
	    break;
	}
    }
    eval_tcl_cmd("set vary_world %d", vary_world);
    eval_tcl_cmd("set vary_real_time %d", vary_real_time);
}

interpret_checkbox(Variant *var, int i)
{
    int newval;
    Obj *vartmp;

    eval_tcl_cmd("set variantstate(%d) active", i);
    eval_tcl_cmd("set varianttext(%d) \"%s\"", i, var->name);
    newval = FALSE;
    if (var->dflt != lispnil) {
	vartmp = eval(var->dflt);
	if (numberp(vartmp)) {
	    newval = c_number(vartmp);
	}
    }
    eval_tcl_cmd("set variantvalue(%d) %d", i, newval);
    checkboxes[i] = var;
    checkboxvalues[i] = newval;
}

set_variant_value(int which, int val)
{
    checkboxvalues[which] = val;
}

set_variant_world_size(int wid, int hgt, int circumf, int lat, int lon)
{
    new_width = wid;
    new_height = hgt;
    new_circumference = circumf;
    new_latitude = lat;
    new_longitude = lon;
}

set_variant_real_time(int total, int perside, int perturn)
{
    new_time_for_game = total;
    new_time_per_side = perside;
    new_time_per_turn = perturn;
}

/* This is where we actually change the state of the game according to
   the variants.  Actually, the kernel does the changing, this just
   collects the variant values and passes them on. */

/*static void*/
implement_variants()
{
    int i;

    variants = lispnil;
    /* Implement the checkbox variants. */
    for (i = 0; i < numcheckboxes; ++i) {
	if (checkboxes[i]) {
	    push_int_binding(&variants, checkboxes[i]->id, checkboxvalues[i]);
	}
    }
    if (vary_world) {
	/* It is critically important that users not be able to
	   reshape already-alloced areas, but do let them know that
	   their request had to be overridden. */
	if (((area.width > 0 && area.width != new_width)
	     || (area.height > 0 && area.height != new_height)
	     || (world.circumference > 0
		 && world.circumference != new_circumference))
	    && (1 /* some layers (probably) allocated already */)) {
	    /* (this is misleading, is an "expected" alert) */
	    init_warning("Area dimensions must remain %d x %d, %d around world",
			 area.width, area.height, world.circumference);
	    new_width = area.width;  new_height = area.height;
	    new_circumference = world.circumference;
	}
	/* Make a world-size-setting and glue it into the list of variants. */
	push_key_cdr_binding(&variants, K_WORLD_SIZE, 
			     cons(new_number(new_width),
				  cons(new_number(new_height),
				       cons(new_number(new_circumference),
					    cons(new_number(new_longitude),
						 cons(new_number(new_latitude),
						      lispnil))))));
    }
    if (vary_real_time) {
	push_key_cdr_binding(&variants, K_REAL_TIME, 
			     cons(new_number(new_time_for_game),
				  cons(new_number(new_time_per_side),
				       cons(new_number(new_time_per_turn),
					    lispnil))));
    }
    do_module_variants(mainmodule, variants);
}

static void help_unit_type(Side *side, Map *map);
static void help_terrain_type(Side *side, Map *map);

create_map()
{
    int u;
    Map *map;
    Side *side2;

    DGprintf("Creating map\n");
    map = (Map *) xmalloc(sizeof(Map));

    map->mode = move_mode;
    map->autoselect = TRUE;
    map->move_on_click = TRUE;

    map->see_all = dside->see_all;

    map->terrain_style = 1;

    map->prefixarg = -1;
    map->inptype = NONUTYPE;

    /* Newest map goes on the front of the list. */
    map->next = dside->ui->maps;
    dside->ui->maps = map;

    dside->ui->curmap = map;

    eval_tcl_cmd("create_map_window");

    for_all_sides(side2) {
	update_side_display(dside, side2, TRUE);
    }
    for_all_unit_types(u) {
	update_unit_type_list(dside, u);
    }
    update_turn_display(dside, TRUE);

    set_tool_cursor(dside, map);
    eval_tcl_cmd("update_mode move");
}

/* planned reduction in number of mag powers */
#undef NUMPOWERS
#define NUMPOWERS 7

zoom_in_out(Side *side, Map *map, int incr)
{
    int newpower;
    VP *vp = widget_vp(map);

    newpower = vp->power + incr;
    if (newpower < 0)
      newpower = 0;
    if (newpower > NUMPOWERS - 1)
      newpower = NUMPOWERS - 1;
    if (newpower != vp->power) {
	set_view_power(vp, newpower);
	center_on_focus(vp);
	redraw_map(map);
	update_mouseover(map, lastrawx, lastrawy);
	eval_tcl_cmd("update_zoom %d %d",
		     (newpower < (NUMPOWERS - 1)), (newpower > 0)); 
    }
}

/* Prompt for a type of a unit from player, maybe only allowing some types
   to be accepted.  Also allow specification of no unit type.  We do this
   by scanning the vector, building a string of chars and a vector of
   unit types, so as to be able to map back when done. */

int
ask_unit_type(side, map, prompt, possibles, handler)
Side *side;
Map *map;
char *prompt;
int *possibles;
void (*handler)();
{
    int u, numtypes = 0;

    for_all_unit_types(u) {
	if (possibles == NULL || possibles[u]) {
	    map->uvec[numtypes] = u;
	    map->ustr[numtypes] = utype_name_n(u, 1)[0];
	    ++numtypes;
	    enable_in_unit_type_list(side, map, u, 1);
	} else {
	    enable_in_unit_type_list(side, map, u, -1);
	}
    }
    map->ustr[numtypes] = '\0';
    if (numtypes > 1) {
	eval_tcl_cmd("ask_unit_type_mode {%s [%s]}", prompt, map->ustr);
	map->modalhandler = handler;
    }
    return numtypes;
}

/* Do something with the char or unit type that the player entered. */

int
grok_unit_type(side, map, typep)
Side *side;
Map *map;
int *typep;
{
    int i, u;

    *typep = NONUTYPE;
    if (map->inptype != NONUTYPE) {
	*typep = map->inptype;
	/* Reset so doesn't affect subsequent unit type queries. */
	map->inptype = NONUTYPE;
    } else if (map->inpch != '\0') {
	if (map->inpch == '?') {
	    help_unit_type(side, map);
	    return FALSE;
	}
	i = iindex(map->inpch, map->ustr);
	if (i >= 0) {
	    *typep = map->uvec[i];
	} else {
	    notify(side, "Must type a unit type char from the list, or <esc>");
	    return FALSE;
	}
    } else {
	notify(side, "weird");
	return FALSE;
    }
    eval_tcl_cmd("ask_unit_type_done");
    /* Reset the appearance of the unit type list. */
    for_all_unit_types(u) {
	enable_in_unit_type_list(side, map, u, 0);
    }
    /* Make the unit type string be empty. */
    map->ustr[0] = '\0';
    return TRUE;
}

int
cancel_unit_type(side, map)
Side *side;
Map *map;
{
    int u;

    /* Reset the appearance of the unit type list. */
    for_all_unit_types(u) {
	enable_in_unit_type_list(side, map, u, 0);
    }
}

static void
help_unit_type(side, map)
Side *side;
Map *map;
{
    int i;
    char helpbuf[BUFSIZE];

    helpbuf[0] = '\0';
    for (i = 0; map->ustr[i] != '\0'; ++i) {
	/* Put out several types on each line. */
	if (i % 4 == 0) {
	    if (i > 0) {
		notify(side, "%s", helpbuf);
	    }
	    /* Indent each line a bit (also avoids notify's
	       auto-capitalization). */
	    strcpy(helpbuf, "  ");
	}
	tprintf(helpbuf, "%c %s, ", map->ustr[i], u_type_name(map->uvec[i]));
    }
    /* Add an extra helpful comment, then dump any leftovers. */
    tprintf(helpbuf, "? for this help info"); 
    notify(side, "%s", helpbuf);
}

enable_in_unit_type_list(side, map, u, flag)
Side *side;
Map *map;
int u, flag;
{
    eval_tcl_cmd("enable_unitlist %d %d", u, flag);
}

int
ask_terrain_type(side, map, prompt, possibles, handler)
Side *side;
Map *map;
char *prompt;
int *possibles;
void (*handler)(Side *side, Map *map, int cancelled);
{
    int numtypes = 0, t;

    for_all_terrain_types(t) {
	if (possibles == NULL || possibles[t]) {
	    map->tvec[numtypes] = t;
	    map->tstr[numtypes] =
	      (!empty_string(t_char(t)) ? t_char(t)[0] : (t - 'a'));
	    ++numtypes;
	}
    }
    map->tstr[numtypes] = '\0';
    if (numtypes > 1) {
	eval_tcl_cmd("ask_terrain_type_mode {%s [%s]}", prompt, map->tstr);
	map->modalhandler = handler;
    }
    return numtypes;
}

/* Do something with the char or terrain type that the player entered. */

int
grok_terrain_type(side, map, typep)
Side *side;
Map *map;
int *typep;
{
    int i;

    *typep = NONTTYPE;
    if (map->inpch == '?') {
	help_terrain_type(dside, map);
	return FALSE;
    }
    i = iindex(map->inpch, map->tstr);
    if (i >= 0) {
	*typep = map->tvec[i];
	eval_tcl_cmd("ask_terrain_type_done");
	return TRUE;
    } else {
	notify(dside, "Must type a terrain type char or <esc>");
	return FALSE;
    }
}

static void
help_terrain_type(side, map)
Side *side;
Map *map;
{
    int i;
    char helpbuf[BUFSIZE];

    for (i = 0; map->tstr[i] != '\0'; ++i) {
	/* Put out several types on each line. */
	if (i % 4 == 0) {
	    if (i > 0) {
		notify(side, "%s", helpbuf);
	    }
	    /* Indent each line a bit (also avoids confusion due to
	       notify's capitalization). */
	    strcpy(helpbuf, "  ");
	}
	tprintf(helpbuf, "%c %s, ", map->tstr[i], t_type_name(map->tvec[i]));
    }
    /* Add an extra helpful comment, then dump any leftovers. */
    tprintf(helpbuf, "? for this help info"); 
    notify(side, "%s", helpbuf);
}

/* User is asked to pick a position on map.  This will iterate until the
   space bar designates the final position. */

/* (should change the cursor temporarily) */

void
ask_position(side, map, prompt, handler)
Side *side;
Map *map;
char *prompt;
void (*handler)(Side *side, Map *map, int cancelled);
{
    eval_tcl_cmd("ask_position_mode {%s [click to set]}", prompt);
    map->answer[0] = '\0';
    map->modalhandler = handler;
}

int
grok_position(side, map, xp, yp, unitp)
Side *side;
Map *map;
int *xp, *yp;
Unit **unitp;
{
    if (in_area(map->inpx, map->inpy)) {
	*xp = map->inpx;  *yp = map->inpy;
	if (unitp != NULL)
	  *unitp = map->inpunit;
	eval_tcl_cmd("ask_position_done");
	return TRUE;
    } else {
	/* Make any possible usage attempts fail. */
	*xp = *yp = -1;
	if (unitp != NULL)
	  *unitp = NULL;
	return FALSE;
    }
}

/* Prompt for a yes/no answer with a settable default. */

void
ask_bool(side, map, question, dflt, handler)
Side *side;
Map *map;
char *question;
int dflt;
void (*handler)(Side *side, Map *map, int cancelled);
{
    eval_tcl_cmd("ask_bool_mode {%s} %d", question, dflt);
    map->answer[0] = '\0';
    map->tmpint = dflt;
    map->modalhandler = handler;
}

/* Figure out what the answer actually is, keeping the default in mind. */

int
grok_bool(side, map)
Side *side;
Map *map;
{
    int dflt = map->tmpint;
    char ch = map->inpch;

    if (dflt ? (lowercase(ch) == 'n') : (lowercase(ch) == 'y'))
      dflt = !dflt;
    eval_tcl_cmd("ask_bool_done");
    return dflt;
}

/* Read a string from the prompt window.  Deletion is allowed, and a
   text cursor (an underscore) is displayed. */

void
ask_string(side, map, prompt, dflt, handler)
Side *side;
Map *map;
char *prompt, *dflt;
void (*handler)(Side *side, Map *map, int cancelled);
{
    /* Default must be non-NULL. */
    if (dflt == NULL)
      dflt = "";
    sprintf(map->answer, "%s", dflt);
    eval_tcl_cmd("ask_string_mode {%s} {%s}", prompt, map->answer);
    map->modalhandler = handler;
}

/* Dig a character from the input and add it into the string.
   Keep returning FALSE until we get something, then make a copy
   of the result string and return TRUE. */

int
grok_string(side, map, strp)
Side *side;
Map *map;
char **strp;
{
    char ch = map->inpch;
    int len;

    if (ch == '\r' || ch == '\n') {
	*strp = copy_string(map->answer);
	eval_tcl_cmd("ask_string_done");
	return TRUE;
    } else {
	len = strlen(map->answer);
	if (ch == BACKSPACE_CHAR || ch == DELETE_CHAR) {
	    if (len > 0)
	      --len;
	} else {
	    map->answer[len++] = ch;
	}
	map->answer[len] = '\0';
	eval_tcl_cmd("update_string_mode \"%s\"", map->answer);
	return FALSE;
    }
}

void
ask_side(side, map, prompt, dfltside, handler)
Side *side;
Map *map;
char *prompt;
Side *dfltside;
void (*handler)(Side *side, Map *map, int cancelled);
{
    char *dfltstr;

    dfltstr = (dfltside == NULL ? "nobody" : side_name(dfltside));
    sprintf(map->answer, "%s", dfltstr);
    eval_tcl_cmd("ask_side_mode {%s} {%s}", prompt, map->answer);
    map->modalhandler = handler;
}

int
grok_side(side, map, side2p)
Side *side;
Map *map;
Side **side2p;
{
    char ch = map->inpch;
    int len;
    Side *side3;

    *side2p = NULL;
    if (ch == '\r' || ch == '\n') {
	if (empty_string(map->answer)
	    || strcmp(map->answer, "nobody") == 0) {
	    eval_tcl_cmd("ask_side_done");
	    return TRUE;
	}
	for_all_sides(side3) {
	    if (!empty_string(side3->name)
		&& strcmp(map->answer, side3->name) == 0) {
		*side2p = side3;
		eval_tcl_cmd("ask_side_done");
		return TRUE;
	    }
	    if (!empty_string(side3->noun)
		&& strcmp(map->answer, side3->noun) == 0) {
		*side2p = side3;
		eval_tcl_cmd("ask_side_done");
		return TRUE;
	    }
	    if (!empty_string(side3->adjective)
		&& strcmp(map->answer, side3->adjective) == 0) {
		*side2p = side3;
		eval_tcl_cmd("ask_side_done");
		return TRUE;
	    }
	}
	beep(side);
	return FALSE;
    } else {
	len = strlen(map->answer);
	if (ch == BACKSPACE_CHAR || ch == DELETE_CHAR) {
	    if (len > 0)
	      --len;
	} else {
	    map->answer[len++] = ch;
	}
	map->answer[len] = '\0';
	eval_tcl_cmd("update_side_mode \"%s\"", map->answer);
	return FALSE;
    }
}
