/* Copyright (c) 1987, 1988  Stanley T. Shebs. */
/* Copyright (c) 1995 Michael J. Peters */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* This file is devoted to commands and other functions related specifically */
/* to help.  Almost all of the xconq help code is here. */

/* At least in X, the help window covers most of the xconq display, so */
/* total redraws are necessary.  This can be slow - fix would be to do */
/* per-window redrawing only as need, and place help window to cover as few */
/* of the other windows as possible (especially world map). */

#include "config.h"
#include "misc.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "Motif.h"
#include "global.h"
#include "version.h"

/* Display the news file on stdout if it exists, be silent if it doesn't. */
/* (Used only during program startup.) */

void maybe_dump_news() {

    FILE *fp;

    make_pathname(xconqlib, NEWSFILE, "", spbuf);
    if ((fp = fopen(spbuf, "r")) != NULL) {
        while (fgets(spbuf, BUFSIZE-1, fp) != NULL) {
            fputs(spbuf, stdout);
        }
        fclose(fp);
    }
}

/* A simple table-printing utility. Hyphens out default values so they don't */
/* clutter the table. */

void append_number(char *buf, int value, int dflt) {

    if (value != dflt) {
        sprintf(tmpbuf, "%5d ", value);
        strcat(buf, tmpbuf);
    } else {
        strcat(buf, "  -   ");
    }
}

/* Full details on the given type of unit.  This may be used either for */
/* online help or for building a descriptive file.  The icon will only */
/* show up for online help, otherwise the display calls have no effect. */

void describe_utype(Side *side, int u) {

    int r, t, u2;

    wprintf(side, "territory value: %d", utypes[u].territory);
    if (utypes[u].bitmapname != NULL)
        wprintf(side, "bitmap: \"%s\"", utypes[u].bitmapname);
    else
        wprintf(side, "char: '%c'", utypes[u].uchar);
    wprintf(side, "Initially: %d in country, %d/10000 hexes density, %s.",
            utypes[u].incountry, utypes[u].density,
            (utypes[u].named ? "named" : "unnamed"));
    wprintf(side, "Maximum speed %d hexes/turn.",
            utypes[u].speed);
    wprintf(side, "%d HP, crippled at %d HP, chance to retreat %d%%.",
            utypes[u].hp, utypes[u].crippled, utypes[u].retreat);
    wprintf(side, "%d%% extra for start up, %d%% extra for R&D.",
            utypes[u].startup, utypes[u].research);
    if (utypes[u].research > 0) {
      for_all_unit_types(u2)
        if (utypes[u].research_contrib[u2] > 0)
          wprintf(side, "Research decreased by %d%% of research on %s.",
              utypes[u].research_contrib[u2], utypes[u2].name);
    }
    wprintf(side, "%d extra moves used up by an attack.",
            utypes[u].hittime);
    wprintf(side,
        "%d%% to succumb to siege, %d%% to revolt, attrition damage %d HP.",
            utypes[u].siege, utypes[u].revolt, utypes[u].attdamage);
    if (utypes[u].seerange == 1) {
        wprintf(side, "Chance to see others %d%%.", utypes[u].seebest);
    } else {
        wprintf(side, "Chance to see %d%% at 1 hex, to %d%% at %d hexes.",
            utypes[u].seebest, utypes[u].seeworst, utypes[u].seerange);
    }
    wprintf(side, "Own visibility is %d.", utypes[u].visibility);
    if (utypes[u].volume > 0 || utypes[u].holdvolume > 0)
        wprintf(side, "Volume is %d, volume of hold is %d.",
                utypes[u].volume, utypes[u].holdvolume);
    wprintf(side, "%s %s",
            (utypes[u].changeside ? "Can be made to change sides." : ""),
            (utypes[u].disband ? "Can be disbanded and sent home." : ""));
    wprintf(side, "%s %s",
            (utypes[u].maker ? "Builds units all the time." : ""),
            (utypes[u].selfdestruct ? "Hits by self-destruction." : ""));
    wprintf(side, "");
    wprintf(side, "%s",
            "  Resource   ToBui  Prod Store  Eats ToMov  Hits HitBy");
    wprintf(side, "%s",
            "               (0)   (0)   (0)   (0)   (0)   (0)   (0)");
    for_all_resource_types(r) {
        sprintf(spbuf, "%10s: ", rtypes[r].name);
        append_number(spbuf, utypes[u].tomake[r], 0);
        append_number(spbuf, utypes[u].produce[r], 0);
        append_number(spbuf, utypes[u].storage[r], 0);
        append_number(spbuf, utypes[u].consume[r], 0);
        append_number(spbuf, utypes[u].tomove[r], 0);
        append_number(spbuf, utypes[u].hitswith[r], 0);
        append_number(spbuf, utypes[u].hitby[r], 0);
        wprintf(side, "%s", spbuf);
    }
    wprintf(side, "");
    wprintf(side, "%s",
            "   Terrain  Slowed Rand% Hide% Defn% Prod% Attr% Acdn%");
    wprintf(side, "%s",
            "               (-)   (0)   (0)   (0)   (0)   (0)   (0)");
    for_all_terrain_types(t) {
        sprintf(spbuf, "%10s: ", ttypes[t].name);
        append_number(spbuf, utypes[u].moves[t], -1);
        append_number(spbuf, utypes[u].randommove[t], 0);
        append_number(spbuf, utypes[u].conceal[t], 0);
        append_number(spbuf, utypes[u].defense[t], 0);
        append_number(spbuf, utypes[u].productivity[t], 0);
        append_number(spbuf, utypes[u].attrition[t], 0);
        append_number(spbuf, utypes[u].accident[t], 0);
        wprintf(side, "%s", spbuf);
    }
    wprintf(side, "");
    wprintf(side, "%s%s",
            "     Hit%  Damg  Cap% Guard  Pro%",
            " Holds Enter Leave  Mob% Bridg Build   Fix");
    wprintf(side, "%s%s",
            "      (0)   (0)   (0)   (0)   (0)",
            "   (0)   (1)   (0) (100)   (0)   (0)   (0)");
    for_all_unit_types(u2) {
        sprintf(spbuf, "%c: ", utypes[u2].uchar);
        append_number(spbuf, utypes[u].hit[u2], 0);
        append_number(spbuf, utypes[u].damage[u2], 0);
        append_number(spbuf, utypes[u].capture[u2], 0);
        append_number(spbuf, utypes[u].guard[u2], 0);
        append_number(spbuf, utypes[u].protect[u2], 0);
        append_number(spbuf, utypes[u].capacity[u2], 0);
        append_number(spbuf, utypes[u].entertime[u2], 1);
        append_number(spbuf, utypes[u].leavetime[u2], 0);
        append_number(spbuf, utypes[u].mobility[u2], 100);
        append_number(spbuf, utypes[u].bridge[u2], 0);
        append_number(spbuf, utypes[u].make[u2], 0);
        append_number(spbuf, utypes[u].repair[u2], 0);
        wprintf(side, "%s", spbuf);
    }
    wprintf(side, "");
}

void x_unit_info(Widget unit_icon, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    int u;

    for_all_unit_types(u) {
        if (unit_icon == side->unitslist[u].icon)
            break;
    }     

    if (u != NOTHING) {
        create_utype_dialog(side, u);
        init_wprintf(side, NULL);
        describe_utype(side, u);
        map_help_dialog(side);
    }
}

/* Help for the main command mode just dumps part of the table, and a little */
/* extra info about what's not in the table.  This may go onto a screen or */
/* into a file, depending on where this was called from. */

void command_help(Side *side, MenuItem *items) {

    int i;

    wprintf(side, "");
    for (i = 0; items[i].label != NULL; i++) {
        if (items[i].accel_text != NULL) 
            wprintf(side, "%7s %s", items[i].accel_text, items[i].help);
        else
            wprintf(side, "        %s", items[i].help);
    }
}

void do_side_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    command_help(side, GAME_menu);
    command_help(side, MESSAGE_menu);
    command_help(side, OPTIONS_menu);
    command_help(side, HELP_menu);
    map_help_dialog(side);
}

void do_map_commands_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    command_help(side, CONSCIOUSNESS_menu);
    command_help(side, MOVE_menu);
    command_help(side, PRODUCTION_menu);
    command_help(side, TRANSPORT_menu);
    command_help(side, MAP_menu);
    command_help(side, MISC_menu);
    map_help_dialog(side);
}

/* Display the mapfile header info. */

void describe_mapfiles(Side *side) {

    int i;

    wprintf(side, "The world is %d hexes around by %d hexes high.",
            world.width, world.height);
    wprintf(side, "");
    if (mainhead != NULL && mainhead->numnotes > 0) {
        for (i = 0; i < mainhead->numnotes; ++i) {
            wprintf(side, "%s", mainhead->notes[i]);
        }
    } else {
        wprintf(side, "(No other documentation is available.)");
    }
}

void do_map_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    describe_mapfiles(side);
    map_help_dialog(side);
}

/* Spit out all the general period parameters in a readable fashion. */

describe_period(Side *side) {

    int u, r, t, i;

    wprintf(side, "This period is named \"%s\".", period.name);
    wprintf(side, "");
    wprintf(side,
        "It includes %d unit types, %d resource types, and %d terrain types.",
            period.numutypes, period.numrtypes, period.numttypes);
    wprintf(side, "First unit type is %s, first product type is %s.",
            (period.firstutype == NOTHING ? "none" :
             utypes[period.firstutype].name),
            (period.firstptype == NOTHING ? "none" :
             utypes[period.firstptype].name));
    wprintf(side, "");
    wprintf(side,
            "Countries are %d hexes across, between %d and %d hexes apart.",
            period.countrysize, period.mindistance, period.maxdistance);
    wprintf(side, "Known area is %d hexes across.", period.knownradius);
    if (period.allseen)
        wprintf(side, "All units are always seen by all sides.");
    wprintf(side, "");
    if (period.counterattack)
        wprintf(side, "Defender always gets a counter-attack.");
    else
        wprintf(side, "Defender does not get a counter-attack.");
    wprintf(side, "Neutral units add %d%% to defense, hit over %d is a nuke.",
            period.neutrality, period.nukehit);
    if (period.efficiency > 0)
        wprintf(side, "Unit recycling is %d%% efficient.", period.efficiency);
    wprintf(side, "");
    for_all_terrain_types(t) {
        wprintf(side, "Terrain:  %c  %s (%s)",
                ttypes[t].tchar, ttypes[t].name, ttypes[t].color);
    }
    wprintf(side, "");
    for_all_resource_types(r) {
        wprintf(side, "Resource: %c  %s (%s)",
                ' ', rtypes[r].name, rtypes[r].help);
    }
    wprintf(side, "");
    for_all_unit_types(u) {
        wprintf(side, "Unit:     %c  %s (%s)",
                utypes[u].uchar, utypes[u].name, utypes[u].help);
    }
    wprintf(side, "");
    if (period.notes != NULL) {
        for (i = 0; period.notes[i] != NULL; ++i) {
            wprintf(side, "%s", period.notes[i]);
        }
    }
}

void do_period_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    describe_period(side);
    map_help_dialog(side);
}

void do_terrain_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_legend_dialog(side);
}

/* Dump assorted information into files, so they can be studied at leisure, */
/* or by people with screens too small for online help. */

void do_printables(Side *side) {

    int u;

    init_wprintf(side, CMDFILE);
    command_help(side, GAME_menu);
    command_help(side, MESSAGE_menu);
    command_help(side, OPTIONS_menu);
    command_help(side, CONSCIOUSNESS_menu);
    command_help(side, MOVE_menu);
    command_help(side, PRODUCTION_menu);
    command_help(side, TRANSPORT_menu);
    command_help(side, MAP_menu);
    command_help(side, MISC_menu);
    finish_wprintf();
    notify(side, "Dumped command list to \"%s\".", CMDFILE);
    init_wprintf(side, PARMSFILE);
    describe_period(side);
    for_all_unit_types(u) {
        wprintf(side, "--------------------------------------------------");
        wprintf(side, "");
        wprintf(side, "%s (%s)", utypes[u].name, utypes[u].help);
        describe_utype(side, u);
        wprintf(side, "");
    }
    finish_wprintf();
    notify(side, "Dumped period description to \"%s\".", PARMSFILE);
}

void do_dump_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    do_printables(side);
}

/* Command to display the program version.  Looks wired in, but of course */
/* this is not something that we want to be easily changeable! */
/* This will also show data about other sides. */

void do_version(Side *side) {

   wprintf(side, "XCONQ version %s", version);
   wprintf(side, "(c) Copyright 1987, 1988, 1991  Stanley T. Shebs");
   wprintf(side, "(c) Copyright 1995  Michael J. Peters");
   wprintf(side, "");

   if (Debug || Build)
       reveal_side((Side *) NULL, side, 100);

   /* Now check that a lot of numbers are ok for the units. */

   {
     Unit *unit;
     Side *loop_side;

     for_all_units(loop_side, unit) {
       if (unit->x < 0 || unit->x >= world.width ||
           unit->y <= 0 || unit->y >= (world.height - 1) ||
           unit->hp <= 0)
         wprintf(side, "Unit off map %s id %d (%d, %d) hp %d",
                unit_handle((Side *) NULL, unit),
                unit->id, unit->x, unit->y, unit->hp);
     }
   }
}

void do_version_help(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    do_version(side);
    map_help_dialog(side);
}

void do_time(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    wprintf(side, "Time will end in %d turns", (global.endtime - global.time));
   wprintf(side, "");
    map_help_dialog(side);
}

void do_turnclock(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    if (global.timeout > 0 )
        wprintf(side, "Turn timeout is enabled and ends after %d seconds.", global.timeout/100);
    else
        wprintf(side, "Turn timeout is not enabled");
    wprintf(side, "");
    map_help_dialog(side);
}

void do_sideclock(Widget menu, XtPointer client_data, XtPointer call_data) {

    Side *side = (Side *)client_data;
    
    create_help_dialog(side);
    init_wprintf(side, NULL);
    if (global.giventime > 0 )
        wprintf(side, "Side timeout is enabled and ends after %d seconds.", global.giventime/100);
    else
        wprintf(side, "Side timeout is not enabled");
    wprintf(side, "");
    map_help_dialog(side);
}
