// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
#ifndef _INTI_BASE_MAIN_H_
#define _INTI_BASE_MAIN_H_

/*
  Inti is a C++ development platform based on glib, GTK+, and the
  Standard Template Library. This header file is machine-generated
  by a script; do not edit it directly.

  This file is distributed under the GNU Library General Public License.
*/
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <inti/io.h>
#include <inti/slot.h>
#include <inti/timedate.h>
#include <inti/signal-system.h>

/*$ doc header
  $short The #Inti::Main class, which represents a main event loop
  $*/

namespace Inti
{
  
  class Main
  {    
    /*$ doc class Main
      $short A main event loop

      $memory c++
      $copy none
      
      $long

      <para>
      #Main allows your application to manage events. Examples
      of events are a user interaction with the GUI, network data
      arriving over a socket, or a certain length of time elapsing.
      Only one instance of #Main may exist per application thread.
      However, multiple instances of #Loop may exist for each
      #Main.
      </para>

      <para>
      To loop infinitely waiting for events, one calls the
      Main::Loop::run() method, as follows:
      <programlisting>
      Inti::Main::Loop loop;
      loop.run ();
      </programlisting>
      To exit the event loop, call Main::Loop::quit(). This exits the event
      loop and resumes execution of your program just after the call
      to Main::Loop::run().
      </para>

      <para>
      One instance of #Loop is called the <firstterm>primary
      loop</firstterm>.  The first argument to the #Loop constructor
      (which defaults to true) specifies whether or not a #Loop
      instance should become the primary loop.  The primary loop
      will be returned from the Main::primary() function, for
      convenience.
      
      <programlisting>
      Main::Loop loop; // primary loop, by default
      assert (Main::primary() == &amp;loop); // assertion should hold
      Main::Loop loop2 (false); // must specify that we are not the primary loop
      </programlisting>
      
      </para>
      
      <para>
      In a graphical application, at least one event source exists by
      default (user interaction with the GUI). In response to GUI
      events, signals are emitted on widgets. For example, if the user
      clicks a #Gtk::Button, a \\Gtk::Button::clicked signal will be emitted.
      </para>
      
      $*/

  public:
    
    class Loop
    {
    public:
      /*$ doc class Loop
        $short Context that allows you to locally run() and quit() the main event loop

        $memory c++
        $copy none
        
        $long

        <para>

        A #Loop object is a local entry point for the current
        thread's main loop. You call run() to enter the main loop, and
        quit() to leave it again.
        
        For example:
        <programlisting>

        Main::Loop loop;

        // block until loop.quit() is called (you must connect
        // to signals or otherwise arrange for this)
        loop.run();
        
        </programlisting>
        </para>

        <para>
        The documentation for #Main has more details.
        </para>
        
        $*/
      
      /*$ doc constructor (important)
        $short constructor for #Loop
        @primary whether this is the primary main loop for the thread to be returned by Main::primary ()
        @main_loop a #Inti::Main instance (defaults to the current #Inti::Main)
        $*/
      Loop (bool primary=true, Main *main_loop = 0);
      
      void run ();
      void quit ();

      /*$ doc destructor Loop
        $short destructor for #Loop
        $long

        <para>
        Calls quit() automatically if the #Loop is currently inside
        a call to run().
        </para>

        $*/
      ~Loop ();
      
    private:
      Loop (const Loop&);
      Loop& operator= (const Loop&);
      GMainLoop *loop () { return loop_; }
      void* expansion_;
      GMainLoop *loop_;
      Main *main_;
    };

    bool iterate (bool block = false);
    bool events_pending () const;
    void flush ();

    /*$ doc topic priorities "Event Source Priorities"
      $short The main loop's priority system

      $long

      <para>

      Inti's main loop abstraction (#Inti::Main) keeps track of
      all event sources, and dispatches handlers to process the
      events. However, not all events are equally important; thus,
      when adding event handlers to the main loop, you can typically
      specify a <firstterm>priority</firstterm>. Priorities are used
      to call certain event handlers with greater frequency, just as
      the UNIX operating system assigns priorities to processes in
      order to schedule CPU time.
      
      </para>

      <para>

      As with UNIX processes, a priority is represented by an integer.
      Lower integers have a higher priority than higher integers.  The
      #Main class contains some static member constants you can use,
      instead of picking an arbitrary integer; from highest to lowest
      priority, these are: ##HIGH, ##DEFAULT, ##HIGH_IDLE,
      ##DEFAULT_IDLE, ##LOW_IDLE.
      
      </para>
      
      $*/
    
    /*$ doc variable DEFAULT
      $short default priority for main loop event handlers (see [topic:priorities])
      $*/
    static const int DEFAULT;
    /*$ doc variable HIGH
      $short high priority for main loop event handlers (see [topic:priorities])
      $*/
    static const int HIGH;
    /*$ doc variable HIGH_IDLE
      $short lower priority than ##DEFAULT, but higher than ##DEFAULT_IDLE (see [topic:priorities])
      $*/
    static const int HIGH_IDLE;
    /*$ doc variable DEFAULT_IDLE
      $short default priority for idle handlers; lower than ##HIGH_IDLE (see [topic:priorities])
      $*/
    static const int DEFAULT_IDLE;
    /*$ doc variable LOW
      $short even lower priority than standard idle handlers (##DEFAULT_IDLE) (see [topic:priorities])
      $*/
    static const int LOW;

    Connection connect_timeout (int interval,
                                Slot0<bool> *handler,
                                int priority = DEFAULT);
    
    Connection connect_idle (Slot0<bool> *handler,
                             int priority = DEFAULT_IDLE);
    
    Connection connect_io (int file_descriptor,
                           IO::ConditionFlagsField condition_flags,
                           Slot2<bool,int,IO::ConditionFlagsField> *handler,
                           int priority = DEFAULT);

    // Sources
    
    class Source
    {
      /*$ doc class Source
        $short Abstract base class for custom event sources
        $memory floating
        $copy none
        $long

        <para>
        Creating a custom event source can be very
        powerful. 95% of cases are covered by the pre-existing
        sources: timeout, idle, and IO. When those aren't quite
        what you need, you can do your own by subclassing #Main::Source
        and overriding its three virtual functions.
        </para>

        $*/
    public:
      void ref ();
      void unref ();
      void sink ();

      /*$ doc method
        $short Method called before the event loop blocks
        @current_time the current time
        @timeout_milliseconds location to store maximum time the event
        loop should block
        $returns whether events are currently pending

        $long

        <para>
        The prepare() method is called for each source _before_ the main
        loop invokes the poll() system call to block waiting for
        events on file descriptors.
        </para>

        <para>
        If you want to be sure the main loop checks for events
        again in a finite amount of time, fill timeout_milliseconds
        with a timeout. Otherwise, timeout_milliseconds can be left
        untouched.
        </para>

        <para>
        prepare() should return true if you already know that
        you have events ready to process, and false if you
        won't know until after the poll() system call.
        </para>
        $*/
      virtual bool prepare (const TimeStamp& current_time,
                            int *timeout_milliseconds) = 0;

      /*$ doc method
        $short Method called after the event loop blocks
        @current_time the current time
        $returns whether events are currently pending
        $long

        <para>
        The check() method is called for each source _after_ the
        main loop invokes the poll() system call to block waiting
        for events on file descriptors.
        </para>

        <para>
        check() should return true if you have an event ready to be
        processed.
        </para>
        
        $*/
        
      virtual bool check (const TimeStamp& current_time) = 0;

      /*$ doc method
        $short Method called to handle events, if prepare() or
        check() returned true
        @current_time the current time
        $returns false to remove this event source, true to
        continue monitoring it

        $long

        <para>
        dispatch() is called to process an event, if either check()
        or prepare() returned true to indicate that an event was pending.
        </para>

        <para>
        dispatch() should return true to continue watching this
        event source, and false to remove this event source.
        If false is returned, the Source object may be deleted.
        </para>
        $*/
        
      virtual bool dispatch (const TimeStamp& current_time) = 0;

    protected:
      Source ();
      virtual ~Source();

    private:
      Source (const Source&);
      Source& operator=(const Source&);
      unsigned int ref_count_;
      bool floating_;
    };

    
    Connection connect_source (Source* source,
                               bool can_recurse = false,
                               int priority = DEFAULT);

    
    // get the current main loop for this thread, if any.
    static Main *get ();

    static Loop *primary ();
    
  private:
    Main ();
    ~Main ();
    Main(const Main&);
    Main& operator=(const Main&);
    void *expansion_;
    static Loop *primary_loop_;
    friend class Loop;
  }; // class Main
  
}; // namespace Inti

#endif // _INTI_BASE_MAIN_H_
