// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
// Example/test hello world program
/*
  The following license applies to this example code, but not to
  the Inti library itself:
  
    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
    DISTRIBUTOR OF THE SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
    OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
    OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <inti/pango/font.h>

#include <inti/gdk/color.h>

#include <inti/gtk/adjustment.h>
#include <inti/gtk/box.h>
#include <inti/gtk/button.h>
#include <inti/gtk/buttonbox.h>
#include <inti/gtk/checkbutton.h>
#include <inti/gtk/checkmenuitem.h>
#include <inti/gtk/dnd.h>
#include <inti/gtk/entry.h>
#include <inti/gtk/gtkbase.h>
#include <inti/gtk/label.h>
#include <inti/gtk/menu.h>
#include <inti/gtk/menubar.h>
#include <inti/gtk/menuitem.h>
#include <inti/gtk/optionmenu.h>
#include <inti/gtk/paned.h>
#include <inti/gtk/radiobutton.h>
#include <inti/gtk/radiomenuitem.h>
#include <inti/gtk/rc.h>
#include <inti/gtk/scale.h>
#include <inti/gtk/scrolledwindow.h>
#include <inti/gtk/selection.h>
#include <inti/gtk/separator.h>
#include <inti/gtk/statusbar.h>
#include <inti/gtk/table.h>
#include <inti/gtk/tearoffmenuitem.h>
#include <inti/gtk/togglebutton.h>
#include <inti/gtk/tooltips.h>
#include <inti/gtk/window.h>

#include <inti/bind.h>
#include <inti/i18n.h>
#include <inti/main.h>
#include <inti/ptr.h>
#include <inti/text.h>

#include <vector>
#include <string>
#include <iostream>
#include <stdlib.h>

using namespace Inti;

/// Base class for all the little test windows we pop up
class TestWindow : public Gtk::Window
{
public:
  TestWindow ();
  
protected:

  Gtk::VBox * vbox () const { return vbox_; }

  ~TestWindow () {}
  
private:
  ptr<Gtk::VBox> vbox_;

  void on_close_clicked ();
};

TestWindow::TestWindow ()
  : vbox_ (new Gtk::VBox)
{
  using namespace Gtk;
  
  add (vbox_);

  Button * b = new Button (_("Close"));
  b->sig_clicked ().connect (this, &TestWindow::on_close_clicked);
  
  vbox_->pack_end (b, false, true, 5);
  vbox_->pack_end (new HSeparator, false, false, 5);
  vbox_->show_all ();
}

void
TestWindow::on_close_clicked ()
{
  destroy ();
}

/////////////////////////////////////////////////////

class LabelTest : public TestWindow
{
public:
  LabelTest ();

protected:
  ~LabelTest () {}
  
};

LabelTest::LabelTest ()
{
  using namespace Gtk;
  
  Label * label = new Label (_("Testing the label widget"));
  vbox ()->pack_start (label, true, true, 5);

  label = new Label;
  label->parse_uline (_("This _label has _underlines you\ncould _use to show _accelerators"));
  vbox ()->pack_start (label, true, true, 5);

  label = new Label (_("This is a multiline label\nand it's\nright justified"));
  label->set_justify (JUSTIFY_RIGHT);
  vbox ()->pack_start (label, true, true, 5);

  label = new Label (_("This is a multiline label\nand it's\nleft justified"));
  label->set_justify (JUSTIFY_LEFT);
  vbox ()->pack_start (label, true, true, 5);
  
  label = new Label (_("By default labels are center-justified\nand center-aligned,\nlike this"));
  vbox ()->pack_start (label, true, true, 5);  

  label = new Label (_("This is a multiline label\nand it's\nleft justified\nand left-aligned"));
  label->set_justify (JUSTIFY_LEFT);
  label->set_alignment (0.0, 0.5);
  vbox ()->pack_start (label, true, true, 5);
  
  label = new Label (_("This is a top-left aligned label"));
  label->set_alignment (0.0, 0.0);
  vbox ()->pack_start (label, true, true, 5);

  label = new Label (_("This is a bottom-right aligned label"));
  label->set_alignment (1.0, 1.0);
  vbox ()->pack_start (label, true, true, 5);
  
  label = new Label (_("This is a long label that goes on for quite a while, with no explicit line breaks. It has line wrap turned on though, so the label ends up on several lines instead of one."));
  label->set_line_wrap (true);
  vbox ()->pack_start (label, true, true, 5);
  
  vbox ()->show_all ();
}

///////////////////////////////////////////////////////

class EntryTest : public TestWindow
{
public:
  EntryTest ();

protected:
  ~EntryTest () {}

  ptr<Gtk::Label> label_;
  
  void on_entry_changed (Gtk::Entry * entry);
};

class AlphaEntry : public Gtk::Entry
{
public:

protected:
  ~AlphaEntry () {} 

  // override this implementation in order to filter out digits.
  virtual void insert_text_impl (const string & text, int * position);

};

void AlphaEntry::insert_text_impl (const string & text, int * position)
{
  using namespace Gtk;
    
  bool found_digits = false;
  string real_text;
  string::const_iterator i = text.begin ();
  while (i != text.end ())
    {
      if (isdigit (*i))
        found_digits = true;
      else
        real_text += *i;
        
      ++i;
    }

  if (found_digits)
    {
      // don't chain up to the standard function that actually inserts
      // the text; instead, call insert_text with our new digit-free
      // text. We call insert_text again instead of simply chaining
      // up, in order to re-emit the insert_text signal with the
      // modified digit-free text. Chaining up works fine as long as
      // there are no signal handlers connected.
      Gdk::beep ();
      if (!real_text.empty ())
        insert_text (real_text, position);
    }
  else
    {
      // chain up
      Entry::insert_text_impl (text, position);
    }
}

EntryTest::EntryTest ()
  : label_(new Gtk::Label)
{
  using namespace Gtk;
  
  Entry * entry = new Entry;
  entry->set_text (_("This entry has some text in it"));
  vbox ()->pack_start (entry, true, true, 5);

  entry = new Entry;
  entry->set_text (_("MyPassword"));
  entry->set_password_mode (false);
  vbox ()->pack_start (entry, true, true, 5);

  entry = new Entry;
  entry->set_text (_("Initial Text"));
  entry->append_text (_("[Appended]"));
  entry->prepend_text (_("[Prepended]"));
  vbox ()->pack_start (entry, true, true, 5);  

  
  vbox ()->pack_start (label_, true, true, 5);
  
  entry = new Entry;
  entry->set_text (_("This entry is synchronized with the label above it"));
  entry->sig_changed ().connect (bind (slot (this, &EntryTest::on_entry_changed),
                                       entry));
  label_->set_text (entry->text ());
  vbox ()->pack_start (entry, true, true, 5);  

  entry = new AlphaEntry;
  entry->set_text (_("This entry won't let you enter digits"));
  vbox ()->pack_start (entry, true, true, 5);
  
  vbox ()->show_all ();
}

void
EntryTest::on_entry_changed (Gtk::Entry * entry)
{
  label_->set_text (entry->text ());
}

/////////////////////////////////////////////////////

class StatusbarTest : public TestWindow
{
public:
  StatusbarTest ();

protected:
  ~StatusbarTest () {}

  void on_push_clicked ();
  void on_pop_clicked ();

  ptr<Gtk::Statusbar> bar_;
  ptr<Gtk::Entry> entry_;
  ptr<Gtk::Button> push_button_;
  ptr<Gtk::Button> pop_button_;
};

StatusbarTest::StatusbarTest ()
  : bar_ (new Gtk::Statusbar),
    entry_ (new Gtk::Entry),
    push_button_ (new Gtk::Button ("Push")),
    pop_button_ (new Gtk::Button ("Pop"))
{
  using namespace Gtk;
  
  bar_->push (_("This is kind of a stupid location for a statusbar"));
  vbox ()->pack_end (bar_, false, true, 5);

  vbox ()->pack_start (entry_, true, true, 5);
  entry_->set_text (_("Click the 'Push' button to push this message into the statusbar"));

  Table * t = new Table (1, 5, true);
  t->attach (push_button_, 1, 2, 0, 1, EXPAND | FILL, EXPAND | FILL);
  t->attach (pop_button_, 3, 4, 0, 1, EXPAND | FILL, EXPAND | FILL);

  push_button_->sig_clicked ().connect (this,
                                        &StatusbarTest::on_push_clicked);
  
  pop_button_->sig_clicked ().connect (this,
                                       &StatusbarTest::on_pop_clicked);

  vbox ()->pack_start (t, true, true, 5);
  
  vbox ()->show_all ();
}

void
StatusbarTest::on_push_clicked ()
{
  bar_->push (entry_->text ());
}

void
StatusbarTest::on_pop_clicked ()
{
  bar_->pop ();
}

///////////////////////////////////////////////////////

class TooltipsTest : public TestWindow
{
public:
  TooltipsTest ();

protected:
  ~TooltipsTest () {}

  ptr<Gtk::Tooltips> group1_;
  ptr<Gtk::Tooltips> group2_;
  ptr<Gtk::Tooltips> toggle_tips_;
  ptr<Gtk::ToggleButton> group1_toggle_;
  ptr<Gtk::ToggleButton> group2_toggle_;

  void group_toggled (Gtk::ToggleButton * toggle,
                      Gtk::Tooltips * tips,
                      const string & groupname);
  
  void on_group1_toggled ();

  void on_group2_toggled ();
};

TooltipsTest::TooltipsTest ()
  : group1_ (new Gtk::Tooltips),
    group2_ (new Gtk::Tooltips),
    toggle_tips_ (new Gtk::Tooltips),
    group1_toggle_ (new Gtk::ToggleButton (_("Disable Group 1"))),
    group2_toggle_ (new Gtk::ToggleButton (_("Disable Group 2")))
{
  using namespace Gtk;

  Box * buttonbox = new HButtonBox;

  vbox ()->pack_start (buttonbox, true, true, 5);

  group1_toggle_->set_active (true);
  group2_toggle_->set_active (true);
  
  buttonbox->add (group1_toggle_);
  buttonbox->add (group2_toggle_);

  group1_toggle_->sig_toggled ().connect (this,
                                          &TooltipsTest::on_group1_toggled);
  group2_toggle_->sig_toggled ().connect (this,
                                          &TooltipsTest::on_group2_toggled);
  
  toggle_tips_->set_tip (group1_toggle_,
                         _("Toggles tooltips for the first group of widgets"));
  toggle_tips_->set_tip (group2_toggle_,
                         _("Toggles tooltips for the second group of widgets"));

  int i = 0;
  while (i < 4)
    {
      Button * b = new Button (_("Group 1 Widget"));
      group1_->set_tip (b,
                        _("Hi I'm a tooltip in group 1. Woo-hoo! Look tooltips even have automatic line wrap if you make a tooltip that's too long."));
      vbox ()->pack_start (b, true, true, 3);
      ++i;
    }
  
  i = 0;
  while (i < 4)
    {
      Button * b = new Button (_("Group 2 Widget"));
      group2_->set_tip (b, _("Hi I'm a tooltip for a widget in Group 2"));
      vbox ()->pack_start (b, true, true, 3);
      ++i;
    }

  vbox ()->show_all ();
}

void
TooltipsTest::group_toggled (Gtk::ToggleButton * toggle,
                             Gtk::Tooltips * tips,
                             const string & groupname)
{
  using namespace Gtk;
  
  Label * label = dynamic_cast<Label*> (toggle->child ());
  if (toggle->active ())
    {
      label->set_text (Text::printf_format(_("Disable %s"), groupname.c_str ()));
      tips->set_enabled (true);
    }
  else
    {
      label->set_text (Text::printf_format(_("Enable %s"), groupname.c_str ()));
      tips->set_enabled (false);
    }
}

void
TooltipsTest::on_group1_toggled ()
{
  group_toggled (group1_toggle_, group1_, _("Group 1"));
}

void
TooltipsTest::on_group2_toggled ()
{
  group_toggled (group2_toggle_, group2_, _("Group 2"));
} 

/////////////////////////////////////////////////////

class CheckButtonTest : public TestWindow
{
public:
  CheckButtonTest ();

protected:
  ~CheckButtonTest () {}

  ptr<Gtk::CheckButton> check1_;
  ptr<Gtk::CheckButton> check2_;
  ptr<Gtk::CheckButton> check3_;

  void update_checkbutton (Gtk::CheckButton * button,
                           Gtk::Label * label);
  
  void on_check1_toggled (Gtk::Label * label);

  void on_check2_toggled (Gtk::Label * label);

  void on_check3_toggled (Gtk::Label * label);
};

CheckButtonTest::CheckButtonTest ()
  : check1_ (new Gtk::CheckButton),
    check2_ (new Gtk::CheckButton),
    check3_ (new Gtk::CheckButton)
{
  using namespace Gtk;
  
  Label * label = new Label (_("Inactive"));
  check1_->add (label);
  vbox ()->pack_start (check1_, true, true, 5);
  check1_->sig_toggled ().connect (bind (slot (this,
                                               &CheckButtonTest::on_check1_toggled),
                                         label));

  label = new Label (_("Inactive"));
  check2_->add (label);
  vbox ()->pack_start (check2_, true, true, 5);

  check2_->sig_toggled ().connect (bind (slot (this,
                                               &CheckButtonTest::on_check2_toggled),
                                         label));


  label = new Label (_("Inactive"));
  check3_->add (label);
  vbox ()->pack_start (check3_, true, true, 5);

  check3_->sig_toggled ().connect (bind (slot (this,
                                               &CheckButtonTest::on_check3_toggled),
                                         label));

  vbox ()->show_all ();
}


void
CheckButtonTest::update_checkbutton (Gtk::CheckButton * button,
                                     Gtk::Label * label)
{
  if (button->active ())
    label->set_text (_("Active"));
  else
    label->set_text (_("Inactive"));
}
  
void
CheckButtonTest::on_check1_toggled (Gtk::Label * label)
{
  update_checkbutton (check1_, label);
}

void
CheckButtonTest::on_check2_toggled (Gtk::Label * label)
{
  update_checkbutton (check2_, label);
}

void
CheckButtonTest::on_check3_toggled (Gtk::Label * label)
{
  update_checkbutton (check3_, label);
}

/////////////////////////////////////////////////////

class RadioButtonTest : public TestWindow
{
public:
  RadioButtonTest ();

protected:
  ~RadioButtonTest () {}

  ptr<Gtk::RadioButton> radio1_;
  ptr<Gtk::RadioButton> radio2_;
  ptr<Gtk::RadioButton> radio3_;

  void update_radiobutton (Gtk::RadioButton * button,
                           Gtk::Label * label);
  
  void on_radio1_toggled (Gtk::Label * label);

  void on_radio2_toggled (Gtk::Label * label);

  void on_radio3_toggled (Gtk::Label * label);
};

RadioButtonTest::RadioButtonTest ()
  : radio1_ (new Gtk::RadioButton),
    radio2_ (new Gtk::RadioButton (radio1_->group ())),
    radio3_ (new Gtk::RadioButton (radio1_->group ()))
{
  using namespace Gtk;
  
  Label * label = new Label (_("Active"));
  radio1_->add (label);
  vbox ()->pack_start (radio1_, true, true, 5);
  radio1_->sig_toggled ().connect (bind (slot (this,
                                               &RadioButtonTest::on_radio1_toggled),
                                         label));

  label = new Label (_("Inactive"));
  radio2_->add (label);
  vbox ()->pack_start (radio2_, true, true, 5);

  radio2_->sig_toggled ().connect (bind (slot (this,
                                               &RadioButtonTest::on_radio2_toggled),
                                         label));


  label = new Label (_("Inactive"));
  radio3_->add (label);
  vbox ()->pack_start (radio3_, true, true, 5);

  radio3_->sig_toggled ().connect (bind (slot (this,
                                               &RadioButtonTest::on_radio3_toggled),
                                         label));

  vbox ()->show_all ();
}

void
RadioButtonTest::update_radiobutton (Gtk::RadioButton * button,
                                          Gtk::Label * label)
{
  if (button->active ())
    label->set_text (_("Active"));
  else
    label->set_text (_("Inactive"));
}
  
void
RadioButtonTest::on_radio1_toggled (Gtk::Label * label)
{
  update_radiobutton (radio1_, label);
}

void
RadioButtonTest::on_radio2_toggled (Gtk::Label * label)
{
  update_radiobutton (radio2_, label);
}

void
RadioButtonTest::on_radio3_toggled (Gtk::Label * label)
{
  update_radiobutton (radio3_, label);
}

/////////////////////////////////////////////////////

class ScaleTest : public TestWindow
{
public:
  ScaleTest ();
protected:
  ~ScaleTest () {}
private:
  ptr<Gtk::Entry> entry_;

  void on_entry_changed (Gtk::Adjustment * adj);
};

ScaleTest::ScaleTest ()
  : entry_ (new Gtk::Entry)
{
  using namespace Gtk;

  Adjustment * adj = new Adjustment (0.0, 100.0,
                                     10.0, // step (used for arrow keys)
                                     20.0, // page (used for Ctrl+arrow keys)
                                     0.0); // page size (ignored for Scales)

  vbox ()->pack_start (new Label (_("Type a number in the entry")),
                       true, true, 5);
  
  vbox ()->pack_start (entry_, false, true, 5);
  entry_->sig_changed ().connect (bind (slot (this,
                                              &ScaleTest::on_entry_changed),
                                        adj));
  
  Scale * s = new HScale (adj);
  vbox ()->pack_start (s, true, true, 5);

  s = new VScale (adj);
  vbox ()->pack_start (s, true, true, 5);

  s = new HScale (adj);
  s->set_decimal_places (10);
  s->set_value_position (POS_RIGHT);
  vbox ()->pack_start (s, true, true, 5);
  
  s = new HScale (adj);
  s->set_decimal_places (0);
  s->set_value_position (POS_BOTTOM);
  vbox ()->pack_start (s, true, true, 5);
  
  s = new HScale (adj);
  s->set_draws_value (false);
  vbox ()->pack_start (s, true, true, 5);

  s = new HScale (-100.0, 300.0);
  vbox ()->pack_start (s, true, true, 5);
  
  vbox ()->show_all ();
}

void
ScaleTest::on_entry_changed (Gtk::Adjustment * adj)
{
  using namespace Gtk;
  
  string text (entry_->text ());
  
  float f = atof (text.c_str ());
  
  adj->set_value (f);
};

/////////////////////////////////////////////////////

class PanedTest : public TestWindow
{
public:
  PanedTest ();

protected:
  ~PanedTest () {}
  
private:


};

PanedTest::PanedTest ()
{
  using namespace Gtk;
  
  Paned * p = new HPaned;
  
  vbox ()->pack_start (p, true, true, 5);
  
  Paned * p2 = new VPaned;
  p->add1 (p2);
  p2->pack1 (new Button (_("Not shrinkable, expands")), true, false);
  p2->pack2 (new Button (_("Not shrinkable, expands")), true, false);

  p2 = new VPaned;
  p->add2 (p2);
  p2->pack1 (new Button (_("Shrinkable, doesn't expand")), false, true);
  p2->pack2 (new Button (_("Expands on window resize, not shrinkable")), true, false);

  vbox ()->show_all ();
}

/////////////////////////////////////////////////////

class MenuTest : public TestWindow
{
public:
  MenuTest ();

protected:
  ~MenuTest () {}

  ptr<Gtk::Label> label_;
  ptr<Gtk::OptionMenu> optionmenu_;
  
  void on_menu_item_activate (string message);

  Gtk::MenuItem * make_menuitem (const char * text,
                                 const char * message);
  
  Gtk::MenuItem * make_checkitem (const char * text,
                                  const char * message);

  Gtk::RadioMenuItem * make_radioitem (Gtk::RadioMenuItem * group_item,
                                       const char * text,
                                       const char * message);
};

MenuTest::MenuTest ()
  : label_ (new Gtk::Label (_("Select a menu item"))),
    optionmenu_ (new Gtk::OptionMenu)
{
  using namespace Gtk;
  
  MenuBar * bar = new MenuBar;

  vbox ()->pack_start (bar, false, false, 0);

  vbox ()->pack_start (label_, true, true, 10);

  vbox ()->pack_start (optionmenu_, true, true, 10);

  MenuItem * mi = make_menuitem (_("Foo"), _("Foo activated"));
  bar->append (mi);

  Menu * submenu = new Menu;
  mi->set_submenu (submenu);

  mi = make_menuitem (_("Bar"), _("Bar activated"));
  submenu->append (mi);
  mi = make_menuitem (_("Baz"), _("Baz activated"));
  submenu->append (mi);

  
  mi = make_menuitem (_("Boom"), _("Boom activated"));
  bar->append (mi);

  submenu = new Menu;
  mi->set_submenu (submenu);

  mi = make_menuitem (_("Shazam"), _("Shazam activated"));
  submenu->append (mi);
  mi = make_menuitem (_("Pow"), _("Pow activated"));
  submenu->append (mi);


  mi = make_menuitem (_("Tearoff"), _("Tearoff activated"));
  bar->append (mi);

  submenu = new Menu;
  mi->set_submenu (submenu);

  submenu->append (new TearoffMenuItem);
  
  mi = make_menuitem (_("Shazam"), _("Shazam activated"));
  submenu->append (mi);
  submenu->append (new SeparatorMenuItem);
  mi = make_menuitem (_("Pow"), _("Pow activated"));
  submenu->append (mi);

  
  mi = make_menuitem (_("Checkitems"), _("Checkitems activated"));
  bar->append (mi);

  submenu = new Menu;
  mi->set_submenu (submenu);

  mi = make_checkitem (_("Check 1"), _("Check 1 activated"));
  submenu->append (mi);
  mi = make_checkitem (_("Check 2"), _("Check 2 activated"));
  submenu->append (mi);


  mi = make_menuitem (_("Radioitems"), _("Radioitems activated"));
  bar->append (mi);

  submenu = new Menu;
  mi->set_submenu (submenu);

  RadioMenuItem * rmi = make_radioitem (0, _("Radio 1"), _("Radio 1 activated"));
  submenu->append (rmi);
  rmi = make_radioitem (rmi, _("Radio 2"), _("Radio 2 activated"));
  submenu->append (rmi);
  rmi = make_radioitem (rmi, _("Radio 3"), _("Radio 3 activated"));
  submenu->append (rmi);
  submenu->append (new SeparatorMenuItem);
  rmi = make_radioitem (rmi, _("Radio 4"), _("Radio 4 activated"));
  submenu->append (rmi);

  
  mi = make_menuitem (_("Right"), _("Right activated"));
  mi->set_right_justified (true);
  bar->append (mi);

  submenu = new Menu;
  mi->set_submenu (submenu);

  mi = make_menuitem (_("Right Justified 1"), _("Right Justified 1 activated"));
  submenu->append (mi);
  mi = make_menuitem (_("Right Justified 2"), _("Right Justified 2 activated"));
  submenu->append (mi);

  // Set up the option menu
  submenu = new Menu;

  mi = make_menuitem (_("Bar"), _("Bar activated"));
  submenu->append (mi);
  mi = make_menuitem (_("Baz"), _("Baz activated"));
  submenu->append (mi);
  mi = make_menuitem (_("Boom"), _("Boom activated"));
  submenu->append (mi);
  
  optionmenu_->set_menu (submenu);
  optionmenu_->set_active (mi);
  
  vbox ()->show_all ();
}
  
void
MenuTest::on_menu_item_activate (string message)
{
  label_->set_text (message);
}

Gtk::MenuItem *
MenuTest::make_menuitem (const char * text,
                         const char * message)
{
  using namespace Gtk;
    
  MenuItem * mi = new MenuItem (text);

  // We would like on_menu_item_activate to take a const string &,
  // but C++ can't figure out how to do the type conversions.
  mi->sig_activate ().connect (bind (slot (this,
                                           &MenuTest::on_menu_item_activate),
                                     string (message)));
  
  return mi;
}
  
Gtk::MenuItem *
MenuTest::make_checkitem (const char * text,
                          const char * message)
{
  using namespace Gtk;
    
  MenuItem * mi = new CheckMenuItem (text);

  // We would like on_menu_item_activate to take a const string &,
  // but C++ can't figure out how to do the type conversions.
  mi->sig_activate ().connect (bind (slot (this,
                                           &MenuTest::on_menu_item_activate),
                                     string (message)));

  return mi;
}

Gtk::RadioMenuItem *
MenuTest::make_radioitem (Gtk::RadioMenuItem * group_item,
                          const char * text,
                          const char * message)
{
  using namespace Gtk;
    
  RadioMenuItem * mi;
  if (group_item)
    mi = new RadioMenuItem (group_item->group (), text);
  else
    mi = new RadioMenuItem (text);

  // We would like on_menu_item_activate to take a const string &,
  // but C++ can't figure out how to do the type conversions.
  mi->sig_activate ().connect (bind (slot (this,
                                           &MenuTest::on_menu_item_activate),
                                     string (message)));

  return mi;
}

/////////////////////////////////////////////////////

class ConnectionDisplay : public Gtk::VBox
{
public:
  ConnectionDisplay ();

  void set_connection (const Connection & cnxn);
  
protected:
  ~ConnectionDisplay ();

  Connection cnxn_;
  int block_count_;
  ptr<Gtk::Label> status_label_;
  ptr<Gtk::Label> block_label_;
  ptr<Gtk::Button> disconnect_button_;
  ptr<Gtk::Button> block_button_;
  ptr<Gtk::Button> unblock_button_;

  void update_block_label ();
  
  void on_disconnect_clicked ();

  void on_block_clicked ();

  void on_unblock_clicked ();
};

ConnectionDisplay::ConnectionDisplay ()
  : block_count_ (0),
    status_label_ (new Gtk::Label (_("Disconnected"))),
    block_label_ (new Gtk::Label (_("Blocked 0 times"))),
    disconnect_button_ (new Gtk::Button (_("Disconnect"))),
    block_button_ (new Gtk::Button (_("Block"))),
    unblock_button_ (new Gtk::Button (_("Unblock")))
{
  disconnect_button_->set_sensitive (false);
  block_button_->set_sensitive (false);
  unblock_button_->set_sensitive (false);
  
  pack_start (status_label_, true, true, 5);
  pack_start (disconnect_button_, true, true, 5);
  pack_start (block_label_, true, true, 5);
  pack_start (block_button_, true, true, 5);
  pack_start (unblock_button_, true, true, 5);

  disconnect_button_->sig_clicked ().connect (this,
                                              &ConnectionDisplay::on_disconnect_clicked);

  block_button_->sig_clicked ().connect (this,
                                         &ConnectionDisplay::on_block_clicked);

  unblock_button_->sig_clicked ().connect (this,
                                           &ConnectionDisplay::on_unblock_clicked);
  
  show_all ();
}

ConnectionDisplay::~ConnectionDisplay ()
{
  cnxn_.disconnect ();
}

void
ConnectionDisplay::set_connection (const Connection & cnxn)
{
  cnxn_.disconnect (); // disconnect the old one.
  cnxn_ = cnxn;
  block_count_ = 0;
  update_block_label ();
  unblock_button_->set_sensitive (false);
  disconnect_button_->set_sensitive (true);
  block_button_->set_sensitive (true);
  status_label_->set_text (_("Connected"));  
}


void
ConnectionDisplay::update_block_label ()
{
  ustring str (Text::printf_format (_("Blocked %d times"), block_count_));
  block_label_->set_text (str);
}
  
void
ConnectionDisplay::on_disconnect_clicked ()
{
  disconnect_button_->set_sensitive (false);
  block_button_->set_sensitive (false);
  unblock_button_->set_sensitive (false);
  status_label_->set_text (_("Disconnected"));
  cnxn_.disconnect ();
}

void
ConnectionDisplay::on_block_clicked ()
{
  block_count_ += 1;
  cnxn_.block ();
  update_block_label ();
  unblock_button_->set_sensitive (true);
}

void
ConnectionDisplay::on_unblock_clicked ()
{
  block_count_ -= 1;
  cnxn_.unblock ();
  update_block_label ();
  if (block_count_ == 0)
    unblock_button_->set_sensitive (false);
}

/////////////////////////////////////////////////////

class TimeoutTest : public TestWindow
{
public:
  TimeoutTest ();

protected:
  ~TimeoutTest () {}

private:
  ptr<Gtk::Scale> scale_;
  ptr<ConnectionDisplay> connection_;
  ptr<Gtk::Entry> entry_;
  
  bool on_timeout ();

  void on_start_clicked ();
};

TimeoutTest::TimeoutTest ()
  : scale_ (new Gtk::VScale (0.0, 100.0)),
    connection_ (new ConnectionDisplay),
    entry_ (new Gtk::Entry)
{
  using namespace Gtk;

  entry_->set_text ("200");  
  scale_->set_value_position (POS_RIGHT);
  
  Box * box = new HBox;
  
  box->pack_start (connection_, true, true, 5);
  box->pack_end (scale_, true, true, 5);
  
  vbox ()->pack_start (box, true, true, 5);

  box = new HBox;
  box->pack_start (new Label (_("Milliseconds:")), true, true, 5);
  box->pack_end (entry_, false, false, 5);
  vbox ()->pack_start (box, true, true, 5);
  
  Button * start_button = new Button (_("Connect a timeout"));
  vbox ()->pack_start (start_button, true, true, 5);

  start_button->sig_clicked ().connect (this,
                                        &TimeoutTest::on_start_clicked);
  
  vbox ()->show_all ();
}

bool
TimeoutTest::on_timeout ()
{
  using namespace Gtk;
    
  Adjustment * adj = scale_->adjustment ();
  if (adj->value () < adj->upper ())
    adj->set_value (adj->value () + adj->step_increment ());
  else
    adj->set_value (adj->lower ());

  return true;
}

void
TimeoutTest::on_start_clicked ()
{
  int length = Text::to_int (entry_->text ());
    
  Connection c = Main::get ()->connect_timeout (length,
                                                slot (this,
                                                      &TimeoutTest::on_timeout));

  connection_->set_connection (c);
}


/////////////////////////////////////////////////////

class IdleTest : public TestWindow
{
public:
  IdleTest ();

protected:
  ~IdleTest () {}

private:
  ptr<Gtk::Scale> scale_;
  ptr<ConnectionDisplay> connection_;
  ptr<Gtk::Entry> entry_;
  
  bool on_idle ();

  void on_start_clicked ();
};

IdleTest::IdleTest ()
  : scale_ (new Gtk::VScale (0.0, 100.0)),
    connection_ (new ConnectionDisplay),
    entry_ (new Gtk::Entry)
{
  using namespace Gtk;

  entry_->set_text (_("200"));  
  scale_->set_value_position (POS_RIGHT);
  
  Box * box = new HBox;
  
  box->pack_start (connection_, true, true, 5);
  box->pack_end (scale_, true, true, 5);
  
  vbox ()->pack_start (box, true, true, 5);

  box = new HBox;
  box->pack_start (new Label (_("Priority:")), true, true, 5);
  box->pack_end (entry_, false, false, 5);
  vbox ()->pack_start (box, true, true, 5);
  
  Button * start_button = new Button (_("Connect an idle function"));
  vbox ()->pack_start (start_button, true, true, 5);

  start_button->sig_clicked ().connect (this,
                                        &IdleTest::on_start_clicked);
  
  vbox ()->show_all ();
}

bool
IdleTest::on_idle ()
{
  using namespace Gtk;
    
  Adjustment * adj = scale_->adjustment ();
  if (adj->value () < adj->upper ())
    adj->set_value (adj->value () + adj->step_increment ());
  else
    adj->set_value (adj->lower ());

  return true;
}

void
IdleTest::on_start_clicked ()
{
  int priority = Text::to_int (entry_->text ());

  // avoid GUI lockup; an idle function with priority DEFAULT or
  // higher locks the GUI (note high priority means low integer value)
  int safe_priority = CLAMP (priority, Main::DEFAULT_IDLE, Main::LOW);

  if (priority != safe_priority)
    {
      // validate the entry. 
      entry_->set_text (Text::to_string (safe_priority));
      Gdk::beep ();
    }
  
  Connection c = Main::get ()->connect_idle (slot (this,
                                                   &IdleTest::on_idle),
                                             priority);

  connection_->set_connection (c);
}

///////

class StatusReporter : public SignalEmitter
{
public:
  void report (const char * message);

  typedef Signal1<void,const char*> ReportSignalType;
  typedef SignalProxy<SignalEmitter, ReportSignalType> ReportSignalProxyType;
  
  ReportSignalProxyType sig_report ()
  {
    return ReportSignalProxyType (this, &report_signal);
  }

protected:
  static ReportSignalType report_signal;

};

StatusReporter::ReportSignalType
StatusReporter::report_signal;

void
StatusReporter::report (const char * message)
{
  report_signal.emit (this, message);
}

class DragSource : public Gtk::Button, public StatusReporter
{
public:
  DragSource ();

protected:
  ~DragSource ();

  virtual void on_drag_data_get (Gtk::DragContext * context,
                                 Gtk::SelectionData * data,
                                 unsigned int info,
                                 unsigned int time);
  
  virtual void on_drag_data_delete (Gtk::DragContext * context);
};

DragSource::DragSource ()
  : Button (_("Drag me"))
{
  using namespace Gtk;
  
  vector<DragTargetEntry> targets;
  targets.push_back (DragTargetEntry ("STRING", 0, 0));
  
  drag_source_set (Gdk::BUTTON1_MASK,
                   targets,
                   Gdk::ACTION_COPY | Gdk::ACTION_MOVE);


  
}

DragSource::~DragSource ()
{

}

void
DragSource::on_drag_data_get (Gtk::DragContext * context,
                              Gtk::SelectionData * data,
                              unsigned int info,
                              unsigned int time)
{
  // This method is called when we need to provide the data
  // to be transferred to the drag destination.
  
  string drag_data ("Drag data string");
  
  data->set (Gdk::atom_intern ("STRING"),
             8,
             static_cast<const void*>(drag_data.c_str ()),
             drag_data.length ());

  report ("drag_data_get");
}

void
DragSource::on_drag_data_delete (Gtk::DragContext * context)
{
  // This method isn't needed, but is included for illustration
  // purposes. If the drag destination indicates that
  // the source data should be deleted (maybe because the
  // drag was a move rather than a copy), then this method
  // will be called.
  cout << _("Drag data delete received") << endl;

  report ("drag_data_delete");
}

class DragDest : public Gtk::Button, public StatusReporter
{
public:
  DragDest ();

protected:
  ~DragDest ();

  
  virtual void on_drag_leave (Gtk::DragContext * context,
                              unsigned int time);
  virtual bool on_drag_motion (Gtk::DragContext * context,
                               int x,
                               int y,
                               unsigned int time);
  virtual bool on_drag_drop (Gtk::DragContext * context,
                             int x, int y,
                             unsigned int time);
  virtual void on_drag_data_received (Gtk::DragContext * context,
                                      int x, int y,
                                      const Gtk::SelectionData & sel_data,
                                      unsigned int info,
                                      unsigned int time);

};

DragDest::DragDest ()
  : Button (_("Drop here"))
{
  using namespace Gtk;
  
  vector<DragTargetEntry> targets;
  targets.push_back (DragTargetEntry ("STRING", 0, 0));
  
  drag_dest_set (DEST_DEFAULT_HIGHLIGHT,
                 targets,
                 Gdk::ACTION_COPY | Gdk::ACTION_MOVE);

}

DragDest::~DragDest ()
{


}

void
DragDest::on_drag_leave (Gtk::DragContext * context,
                         unsigned int time)
{ 
  report ("drag_leave");
  
}

bool
DragDest::on_drag_motion (Gtk::DragContext * context,
                          int x,
                          int y,
                          unsigned int time)
{
  report ("drag_motion");

  // status() updates the current drag action, telling Inti::Gtk
  // that a drop is possible and what action the drop will cause.
  // return value of "true" means that we can take a drop and that
  // we called status().


  // In this case, passing the DEST_DEFAULT_MOTION flag in
  // drag_dest_set() (see constructor) would have implemented this
  // method for us, and we wouldn't need to override it. However
  // I did it the hard way to illustrate.
  
  context->status (context->suggested_action (),
                   time);
  return true;
}

bool
DragDest::on_drag_drop (Gtk::DragContext * context,
                        int x, int y,
                        unsigned int time)
{
  report ("drag_drop");

  // In this case, passing the DEST_DEFAULT_DROP flag in
  // drag_dest_set() (see constructor) would have implemented this
  // method for us, and we wouldn't need to override it. However
  // I did it the hard way to illustrate.
  
  // Ask that the source send us the drag data; when we get the
  // data, on_drag_data_received will be called.
  drag_get_data (context, Gdk::atom_intern ("STRING"), time);

  // returning true is a promise to call context->finish ()
  // when we receive the data.
  return true;
}

void
DragDest::on_drag_data_received (Gtk::DragContext * context,
                                 int x, int y,
                                 const Gtk::SelectionData & sel_data,
                                 unsigned int info,
                                 unsigned int time)
{
  report ("drag_data_received");

  // This method is called when the dropped data is received.
  // In this case, we're expecting data of type STRING.
  // If the source failed to provide the data, sel_data
  // would be invalid.

  // If you use the DEST_DEFAULT_DROP flag to drag_dest_set,
  // sel_data is guaranteed to be valid, and finish() will be
  // called on your behalf. However I'm doing it the hard
  // way to illustrate the concepts.
  
  if (sel_data.valid ())
    {
      string drop_data (static_cast<const char*>(sel_data.data()),
                        sel_data.length ());
      
      cout << "Drop data: " << drop_data << endl;
    }
    
  // finish() concludes a drop
  
  // whether the source should delete its copy of the data
  bool delete_source_data = context->action () == Gdk::ACTION_MOVE;
  
  context->finish (sel_data.valid (),  // whether the drop was successful (accepted)
                   delete_source_data,
                   time);
}

class DragTest : public TestWindow
{
public:
  DragTest ();

protected:
  ~DragTest ();

private:
  ptr<DragSource> source_;
  ptr<DragDest> target_;
  ptr<Gtk::Label> source_report_;
  ptr<Gtk::Label> target_report_;

  void on_source_report (const char * message);
  void on_dest_report (const char * message);
};

DragTest::DragTest ()
  : source_ (new DragSource),
    target_ (new DragDest),
    source_report_ (new Gtk::Label (_("Source widget status"))),
    target_report_ (new Gtk::Label (_("Target widget status")))
{
  using namespace Gtk;
  
  source_->sig_report ().connect (this, &DragTest::on_source_report);
  target_->sig_report ().connect (this, &DragTest::on_dest_report);

  vbox ()->pack_start (source_report_, false, false, 0);
  vbox ()->pack_start (source_, true, true, 0);

  vbox ()->pack_end (target_, true, true, 0);
  vbox ()->pack_end (target_report_, false, false, 0);

  show_all ();
}

DragTest::~DragTest ()
{


}

void
DragTest::on_source_report (const char * message)
{
  source_report_->set_text (message);
}

void
DragTest::on_dest_report (const char * message)
{
  target_report_->set_text (message);
}

/////

class RcTest : public TestWindow
{
public:
  RcTest ();

protected:
  ~RcTest ();

};

RcTest::RcTest ()
{
  using namespace Gtk;
  
  Button * b;
  RcStyle * rc;

  b = new Button ("Green background (state normal)");
  rc = new RcStyle;
  rc->set_bg (STATE_NORMAL, Gdk::Color ("green"));
  b->modify_style (rc);  
  vbox ()->pack_start (b, false, false, 0);

  b = new Button ("Green background (state prelight)");
  rc = new RcStyle;
  rc->set_bg (STATE_PRELIGHT, Gdk::Color ("green"));
  b->modify_style (rc);  
  vbox ()->pack_start (b, false, false, 0);

  b = new Button ("Green foreground (state normal)");
  rc = new RcStyle;
  rc->set_fg (STATE_NORMAL, Gdk::Color ("green"));
  b->modify_style (rc);
  // modify the label inside the button as well.
  b->child ()->modify_style (rc);
  vbox ()->pack_start (b, false, false, 0);

  b = new Button ("Big Bold Font");
  rc = new RcStyle;
  rc->set_font_description (Pango::FontDescription ("Sans Bold 18"));
  // modify label inside the button
  b->child ()->modify_style (rc);  
  vbox ()->pack_start (b, false, false, 0);

  b = new Button ("Small Italic Font");
  rc = new RcStyle;
  rc->set_font_description (Pango::FontDescription ("Sans Italic 8"));
  // modify label inside the button
  b->child ()->modify_style (rc);
  vbox ()->pack_start (b, false, false, 0);
  
  show_all ();
}

RcTest::~RcTest ()
{


}

/////////////////////////////////////////////////////

/// Class for the main window
class MainWindow : public TestWindow
{
public:
  MainWindow ();

protected:
  void on_destroy ();

  ~MainWindow () { }

private:
  
  // Utility function to add a button to the main window
  void add_button (const string & text,
                   void (* callback) (MainWindow * win));

  ptr<Gtk::Box> buttons_;
};

// create_test would be a global function except that it makes gcc
// 2.95 barf all over itself. Apparently gcc doesn't understand the
// syntax create_test<LabelTest> but it does understand
// TestCreator<LabelTest>
template <class Test>
class TestCreator
{
public:
  static void
  create_test (MainWindow * win)
  {
    TestWindow * t = new Test ();
    t->set_transient_for (win);
    t->show ();
  }
};

MainWindow::MainWindow ()
  : buttons_ (new Gtk::VBox)
{
  using namespace Gtk;

  ScrolledWindow * sw = new ScrolledWindow;
  sw->set_policy (POLICY_NEVER, POLICY_AUTOMATIC);

  sw->add_with_viewport (buttons_);
  vbox ()->pack_start (sw, true, true, 5);
  
  add_button ("Gtk::CheckButton", TestCreator<CheckButtonTest>::create_test);
  add_button ("Gtk::Entry", TestCreator<EntryTest>::create_test);
  add_button ("Gtk::Label", TestCreator<LabelTest>::create_test);
  add_button ("Gtk::Menu", TestCreator<MenuTest>::create_test);
  add_button ("Gtk::OptionMenu", TestCreator<MenuTest>::create_test);
  add_button ("Gtk::Paned", TestCreator<PanedTest>::create_test);
  add_button ("Gtk::RadioButton", TestCreator<RadioButtonTest>::create_test);
  add_button ("Gtk::RcStyle", TestCreator<RcTest>::create_test);
  add_button ("Gtk::Scale", TestCreator<ScaleTest>::create_test);
  add_button ("Gtk::Statusbar", TestCreator<StatusbarTest>::create_test);
  add_button ("Gtk::Tooltips", TestCreator<TooltipsTest>::create_test);
  add_button (_("Drag and Drop"), TestCreator<DragTest>::create_test);
  add_button (_("Idles"), TestCreator<IdleTest>::create_test);
  add_button (_("Timeouts"), TestCreator<TimeoutTest>::create_test);
  
  vbox ()->show_all ();

  set_title (_("Inti::Gtk test program"));
}

void
MainWindow::on_destroy ()
{
  // quit the application if we haven't
  if (Main::primary ())
    Main::primary ()->quit ();
}

void
MainWindow::add_button (const string & text,
                        void (* callback) (MainWindow * win))
{
  using namespace Gtk;
  
  Button * b = new Button (text);
  b->sig_clicked ().connect (bind (slot (callback), this));
  buttons_->pack_start (b, true, true, 0);
}

//////////////////////////////////////////////////////

int
main (int argc, char **argv)
{
  using namespace Gtk;

  cout << "Inti::Object uses this many bytes: " << sizeof (Object) << endl;
  cout << "Inti::Gtk::Widget uses this many: " << sizeof (Gtk::Widget) << endl;
  
  init (&argc, &argv);

  ptr<MainWindow> mw (new MainWindow);

  mw->set_default_height (300);
  
  mw->show ();

  Main::Loop loop;
  
  loop.run ();
  
  return 0;
}
