#include "system.h"
#include "makeinfo.h"

#include <stdio.h>

extern int current_chapter_number;
extern int current_equation_number;
extern int equation_number_pending;
extern char *equation_number_text;

extern int chapter_number;
extern int appendix_number;
extern int appendix;

/* @equation

   Increments the equation number and prepares a text string of the
   equation number for output. If there is an equation label, it is
   stored in a texinfo variable, so the user can refer to the
   equation. */

void 
do_equation ()
{
  char *label, *value;

  /* An extra blank line before equations makes them look much
     nicer. We can't put the blank line in the texinfo file itself,
     because that would make an extra line in the TeX version */

  add_char ('\n');

  current_equation_number++;
  equation_number_pending = 1;

  /* Make some space for the equation number string. */

  value = xmalloc(32 * sizeof(char)) ;

  /* FIXME: I'd like to use asprintf here, but that would need changes
     to configure.in and lib. 32 bytes should be safe though, since the
     user can't get at this directly.  */
  
  /* Now figure out how we want to write the equation number ... */

  if (appendix)
    {
      /* If we're in an appendix, make an equation number like B.27 */
      sprintf (value, "%c.%d", appendix_number + 'A' - 1, 
	       current_equation_number);
    }
  else if (chapter_number)
    {
      /* If we're in a numbered chapter, make an equation number like 3.27 */
      sprintf (value, "%d.%d", chapter_number, current_equation_number);
    }
  else
    {
      /* If we're not in a chapter, use a plain equation number like 27 */
      sprintf (value, "%d", current_equation_number);
    }

  if (equation_number_text == 0)
    {
      equation_number_text = (char *) xmalloc (strlen (value) + 1);
    }
  else
    {
      equation_number_text = (char *) xrealloc (equation_number_text,
						strlen (value) + 1);
    }

  strcpy (equation_number_text, value);

  get_rest_of_line (&label);
  backup_input_pointer ();
  canon_white (label);

  /* @equation [LABEL]

     Did we find a LABEL in the rest_of_line?  If so, we store the
     equation number in the corresponding texinfo variable. The user
     can then refer to it with @value{LABEL}. */

  if (*label)
    {
      /* The equation label should be unique, to prevent confusion. */
      if (set_p (label))
	{
	  line_error (_("equation label `%s' redefined"), label);
	} 

      /* Set the corresponding texinfo variable. */
      set (label, value);
    }

  free (label);
  free (value);

}

/* @end equation  

   add_equation_number() appends an equation number (flushright) to
   the current paragraph.

   To format flushright, search backwards to find the length of the
   final line, then add sufficient spaces to insert the equation
   number flush against the right margin (fill_column).  */

void
add_equation_number ()
{
  int tindex, c;
  
  /* Back up to last non-newline/space character */
  
  for (tindex = output_paragraph_offset - 1; tindex > 0; --tindex)
    {
      c = output_paragraph[tindex];
      
      if (c == ' '|| c == '\n')
	/* ignore trailing whitespace */ ;
      else
	break ;
    }
  
  if (tindex > 0) {  /* we have a non-blank line, find its length */

    int length, fill ;

    output_paragraph_offset = tindex + 1 ;
    
    for (; tindex >= 0; --tindex)
      {
	c = output_paragraph[tindex];
	
	if (c == '\n')
	  break ;
      }
    
    length = (output_paragraph_offset - tindex) ;

    /* add sufficient spaces to get the equation number aligned with
       the right margin of the text (fill_column). Actually
       fill_column is the maximum linewidth, so we subtract a few (4)
       spaces.  Actual text is a few characters short of fill_column
       on average. */

    fill = (fill_column - strlen(equation_number_text) - 4) - length ;
    
    for (tindex = 0 ; tindex < fill; tindex++) {
      insert(' ') ;
    }
    
    /* add equation number, in parens,  (3.7) */

    insert (' ') ;
    insert ('(') ;
    add_word (equation_number_text) ;
    insert (')') ;

    equation_number_pending = 0 ;
  }
}
