/* Common functions for each GNATS client.
   Copyright (C) 1994, 1995 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more detailmails.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"
#include "query.h"

#ifdef HAVE_MACHINE_ENDIAN_H
#include <machine/endian.h>			/* for htons */
#endif

#ifdef HAVE_KERBEROS
#include <des.h>
#include <krb.h>
#ifndef HAVE_KRB_GET_ERR_TEXT
#define krb_get_err_text(status) krb_err_txt[status]
#endif
int remember_krb_auth_ret = 0; /* yes, I like lisp. go away. */
#endif

/* For the GNATS server we'll be talking with.  */
FILE *serv_write, *serv_read;

/* Where our results should go.  */
FILE *outfp = stdout;

/* What the server said back to us.  */
char *reply = NULL;

/* The descriptor for talking to the server in non-stdio ways.  */
int sockfd, dupfd;

/* If 1, send information as we make the query.  */
extern int debug;

/* The name this program was run with.  */
extern char *program_name;

void
free_reply (r)
     Reply *r;
{
  xfree (r->text);
  xfree ((char*)r);
}

Reply *
server_reply ()
{
  Reply *r;
  static char recvline[MAXLINE + 1];
  char *p, c;

  if (fgets (recvline, MAXLINE, serv_read) == NULL)
    {
      fprintf (stderr, "%s: error reading from server\n", program_name);
      safe_exit ();
    }

  c = recvline[0];
  if (isascii (c) && isdigit(c))
    {
      for (p = &recvline[0];
	   *p && isascii (*p) && isdigit (*p);
	   p++)
	continue;
      if (*p)
	{
	  r = (Reply *) xmalloc (sizeof (Reply));
	  if (*p == '-')
	    r->type = REPLY_CONT;
	  else
	    {
	      if (*p != ' ')
		fprintf (stderr, "%s: bad type of reply from server\n",
			 program_name);
	      r->type = REPLY_END;
	    }
	  *p = '\0';
	  r->state = atoi (recvline);
	  r->text = (char *) strdup (p + 1);
#if 0
	  i = strlen (r->text) - 2;
	  r->text[i] = '\n'; r->text[i + 1] = '\0';
#endif
	  return r;
	}
    }

  return (Reply *)NULL;
}

void
read_server ()
{
  int i;
  char recvline[MAXLINE + 1], *recvptr = &recvline[0];

  while (fgets (recvline, MAXLINE, serv_read))
    {
      if (debug)
	fprintf (stderr, "%s: received `%s'\n", program_name, recvline);
      if (memcmp (recvptr, ".\r\n\0", 4) == 0)
	return;
      if (recvline[1])
	{
	  i = strlen (recvline) - 2;
	  if (recvline[i] == '\r')
	    {
	      /* Only correct this if we're actually at the end of a line.  */
	      recvline[i] = '\n';
	      recvline[i + 1] = '\0';
	    }
	}
      /* Remove escape character added by write_multiline() */
      i = 0;
      if (recvline[0] == '.')
        i = 1;
      fprintf (outfp, "%s", recvptr + i);
    }
} 


void
get_reply ()
{
  Reply *r;
  char *s, *p;

  /* Make sure anything we've written has gone to them.  */
  if (fflush (serv_write) == EOF || ferror (serv_write))
    fprintf (stderr, "%s: error writing to server\n", program_name);

  r = server_reply ();
  if (r == (Reply *)NULL)
    {
      if (debug)
	fprintf (stderr, "%s: null reply from the server\n",
		 program_name);
      return;
    }
  if (debug)
    fprintf (stderr, "%s: received %d `%s'\n", program_name,
	     r->state, r->text);
  switch (r->state)
    {
    case CODE_GREETING:
      while (r->type == REPLY_CONT)
	{
	  free_reply (r);
	  r = server_reply ();
	  if (r == (Reply *)NULL)
	    {
	      if (debug)
		fprintf (stderr, "%s: null reply from the server\n",
			 program_name);
	      return;
	    }
	  if (debug)
	    fprintf (stderr, "%s: received %d `%s'\n", program_name,
		     r->state, r->text);
	}
      break;

    case CODE_OK:
    case CODE_HELLO:
      break;

    case CODE_CLOSING:
      if (debug)
	fprintf (stderr, "%s: server closing down\n", program_name);
      break;

    case CODE_PR_READY:
      /* read it til we get a . */
      read_server ();
      break;

    case CODE_INFORMATION:
      /* XXX nquery-pr just uses read_server */
      free_reply (r);
      /* Do something with it.  */
      r = server_reply ();
      while (r->type == REPLY_CONT)
	{
	  int i = strlen (r->text) - 2;
	  if (r->text[i] == '\r')
	    {
	      r->text[i] = '\n';
	      r->text[i + 1] = '\0';
	    }
	  fprintf (outfp, "%s", r->text);
	  free_reply (r);
	  r = server_reply ();
	}
      break;

    case CODE_INVALID_PR:
      s = strchr (r->text, '.'); s[0] = '\n'; s[1] = '\0';
      s = strchr (r->text, ' ') + 1;
      fprintf (stderr, "%s: couldn't find %s", program_name, s);
      safe_exit ();
      break;

    case CODE_INVALID_CATEGORY:
      s = strchr (r->text, '.'); s[0] = '\0';
      s = strrchr (r->text, ' ') + 1;
      fprintf (stderr, "%s: no such category %s\n", program_name, s);
      safe_exit ();
      break;

    case CODE_INVALID_SUBMITTER:
      s = strchr (r->text, '.'); s[0] = '\0';
      s = strrchr (r->text, ' ') + 1;
      fprintf (stderr, "%s: no such submitter named %s\n",
	       program_name, s);
      safe_exit ();
      break;

    case CODE_INVALID_ENUM:
      s = strchr (r->text, '\r'); s[0] = '\0';
      while ((p = strchr (r->text, ',')))
        p[0] = '\n';
      fprintf (stderr, "%s: %s\n", program_name, r->text);
      safe_exit ();
      break;

    case CODE_INVALID_RESPONSIBLE:
      s = strchr (r->text, '.'); s[0] = '\0';
      s = strrchr (r->text, ' ') + 1;
      fprintf (stderr, "%s: no such responsible named %s\n",
	       program_name, s);
      safe_exit ();
      break;

    case CODE_INVALID_DATE:
      s = strrchr (r->text, '.'); s[0] = '\0';
      s = strchr (r->text, ':') + 2;
      fprintf (stderr, "%s: cannot parse the date: %s.\n",
	       program_name, s);
      safe_exit ();
      break;

    case CODE_UNREADABLE_PR:
      s = strchr (r->text, '.'); s[0] = '\n'; s[1] = '\0';
      s = strchr (r->text, ' ') + 1;
      fprintf (stderr, "%s: couldn't read %s", program_name, s);
      safe_exit ();
      break;

    case CODE_PR_NOT_LOCKED:
      fprintf (stderr, "%s: PR is not locked\n", program_name);
      safe_exit ();
      break;

    case CODE_LOCKED_PR:
    case CODE_FILE_ERROR:
    case CODE_ERROR:
      s = strchr (r->text, '\r'); s[0] = '\0';
      fprintf (stderr, "%s: %s\n", program_name, r->text);
      safe_exit ();
      break;

    case CODE_GNATS_LOCKED:
      fprintf (stderr, "%s: lock file exists\n", program_name);
      safe_exit ();
      break;

    case CODE_NO_PRS:
      fprintf (stderr, "%s: no PRs matched\n", program_name);
      safe_exit ();
      break;

    case CODE_NO_KERBEROS:
      fprintf (stderr, "%s: no Kerberos support, authentication failed\n",
	       program_name);
    case CODE_NO_ACCESS:
#ifdef HAVE_KERBEROS
      if (remember_krb_auth_ret)
	fprintf (stderr, "%s: error (%s) while attempting krb4-auth\n",
		 program_name, krb_get_err_text (remember_krb_auth_ret));
      else
	fprintf (stderr, "%s: access denied\n", program_name);
#else
      fprintf (stderr, "%s: access denied\n", program_name);
#endif
      safe_exit ();
      break;

    default:
      fprintf (stderr, "%s: cannot understand %d `%s'\n",
	       program_name, r->state, r->text);
      safe_exit ();
      break;
    }

  free_reply (r);
}

void
safe_exit ()
{
  static int doing_exit = 0;
  if (! doing_exit)
    {
      doing_exit = 1;
      client_exit ();
      exit (1);
    }
  else
    return;
}

void
tohex (ptr, buf, sz)
     char *ptr, *buf;
     int sz;
{
  static const char hex[] = "0123456789abcdef";
  int c;
  while (sz--)
    {
      c = *ptr++ & 0xff;
      *buf++ = hex[c >> 4];
      *buf++ = hex[c & 15];
    }
  *buf++ = 0;
}

int
hexdigit (c)
     char c;
{
  if (c >= '0' && c <= '9')
    return c - '0';
  if (c >= 'a' && c <= 'f')
    return c - 'a' + 10;
  if (c >= 'A' && c <= 'F')
    return c - 'A' + 10;
  return -1;
}

void
fromhex (ptr, buf, sz)
     char *ptr, *buf;
     int sz;
{
  int val;
  while (sz--)
    {
      val = hexdigit (*buf++);
      val <<= 4;
      val |= hexdigit (*buf++);
      *ptr++ = val;
    }
}

void
client_init ()
{
  register struct servent *servport;
  struct sockaddr_in server;
  struct hostent *host;
  FILE *file;

  bzero ((char *) &server, sizeof (server));
  server.sin_family = AF_INET;
  if (isdigit (host_name[0]))
    server.sin_addr.s_addr = inet_addr (host_name);
  else
    {
      host = gethostbyname (host_name);
      if (host == NULL)
	{
	  fprintf (stderr, "%s: %s: No such host\n", program_name, host_name);
	  exit (1);
	}
      bcopy (host->h_addr, &server.sin_addr, sizeof (server.sin_addr));
    }
  if (port > 0)
    server.sin_port = htons (port);
  else
    {
      servport = getservbyname (gnats_service, "tcp");
      if (servport == NULL)
	/* XXX should we emit a warning here? */
	server.sin_port = htons (gnats_server_port);
      else
	server.sin_port = servport->s_port;
    }

  if (debug)
    fprintf (stderr, "%s: opening connection to %s\n",
	     program_name, host_name);

  sockfd = socket (AF_INET, SOCK_STREAM, 0);
  if (sockfd < 0)
    {
      fprintf (stderr, "%s: could not open a socket\n", program_name);
      exit (1);
    }
  if (connect (sockfd, (struct sockaddr *) &server, sizeof (server)) < 0)
    {
      fprintf (stderr, "%s: could not connect to server on %s\n",
	       program_name, host_name);
      close (sockfd);
      exit (1);
    }
  if ((file = fdopen (sockfd, "r")) == NULL)
    {
      fprintf (stderr, "%s: cannot read from server on %s\n",
	       program_name, host_name);
      close (sockfd);
      exit (1);
    }
  serv_read = file;
  dupfd = dup (sockfd);
  if (dupfd < 0)
    {
      fprintf (stderr, "%s: could not duplicate a socket\n", program_name);
      exit (1);
    }
  if ((file = fdopen (dupfd, "w")) == NULL)
    {
      fprintf (stderr, "%s: cannot write to server on %s\n",
	       program_name, host_name);
      close (sockfd);
      exit (1);
    }
  serv_write = file;

  /* Get the hello.  */
  get_reply ();

#ifdef HAVE_KERBEROS
  {
    /* This is ugly, because krb_sendauth/krb_recvauth don't maintain
       a sensible connection state if authentication fails.  This is
       unacceptable here.  So I've broken out most of the interesting
       bits of those routines here.  I've also added a hex encoding of
       the data exchanged instead of the standard authenticator
       format, to maintain the text-only protocol used in GNATS.  */

    Reply *r;
    char *realm, *p, *me;
    int ret, i;
    struct sockaddr_in laddr;
    int laddrlen = sizeof (laddr);
    char hname[MAXHOSTNAMELEN]; /* for canonicalization by krb_mk_auth */
    /* Scratch space for Kerberos library, required to be provided by
       the application.  Keep it all together so we can erase it all
       at once later.  */
    struct {
      CREDENTIALS cred;
      MSG_DAT msg_data;
      Key_schedule sched;
      KTEXT_ST tkt, packet;
      char buf[MAX_KTXT_LEN * 2 + 10];
    } k;
    k.packet.mbz = 0;

#define PUNT(reason,subreason)	do{if (debug) fprintf(stderr,"%s: %s (%s)\n", program_name, reason, subreason);goto krb_exit;}while(0)
#define SKIP(reason)		PUNT("skipping Kerberos authentication",reason)
#define FAIL(reason)		PUNT("Kerberos authentication failed",  reason)

    /* This probably ought to be a crypto checksum of the
       authenticator, but a constant should be nearly as secure.  */
#define CKSUM			0x10291966

    strcpy (hname, host->h_name);

    realm = krb_realmofhost (hname);

    ret = krb_mk_auth (KOPT_DO_MUTUAL, &k.tkt,
		       GNATS_KRB4_PRINCIPAL_NAME, hname, realm,
		       CKSUM, GNATS_KRB4_VERSIONID, &k.packet);
    remember_krb_auth_ret = ret;
    if (ret)
      SKIP (krb_get_err_text (ret));

    ret = krb_get_cred (GNATS_KRB4_PRINCIPAL_NAME, hname, realm, &k.cred);
    /* This should always work unless the ticket file has been changed
       out from under the client at just the wrong time.  */
    if (ret)
      abort ();

    /* Since we don't currently have an interface for specifying a
       "local name" on the server distinct from the Kerberos principal
       name, just use the principal.  */
    me = k.cred.pname;

    if (debug)
      fprintf (stderr, "%s: writing `AUTH krb4-simple'\n", program_name);
    fprintf (serv_write, "AUTH krb4-simple\r\n");
    fflush (serv_write);
    r = server_reply ();
    if (!r || r->state == CODE_AUTH_TYPE_UNSUP)
      SKIP ("not supported by server");
    else if (r->state == CODE_ERROR)
      {
	fprintf (stderr, "%s: server authentication error: %s\n",
		 program_name, r->text);
	goto krb_exit;
      }
    else if (r->state != CODE_OK)
      abort ();
    else
      free_reply (r);

    if (getsockname (sockfd, (struct sockaddr *) &laddr, &laddrlen) < 0)
      {
	fprintf (stderr, "%s: getsockname: %s", program_name,
		 strerror (errno));
	exit (1);
      }

    k.buf[0] = 'x';
    /* This would be better with a base-64 encoding, but I'm too lazy
       to write it write now.  */
    tohex (k.packet.dat, k.buf + 1, k.packet.length);

    fprintf (serv_write, "%s\r\n%s\r\n", me, k.buf);
    fflush (serv_write);
 
    i = fgetc (serv_read);
    if (i == 'x')
      /* ok */
      k.buf[0] = i;
    else
      {
	/* error return */
	ungetc (i, serv_read);
	r = server_reply ();
	fprintf (stderr, "%s: %s", program_name, r->text);
	goto krb_exit;
      }
    p = fgets (k.buf + 1, sizeof (k.buf) - 1, serv_read);
    if (!p)
      /* eof */
      goto krb_exit;
    p = strchr (k.buf + 1, '\r');
    if (p)
      *p = 0;
    k.packet.length = (strlen (k.buf) - 1) / 2;
    fromhex (k.packet.dat, k.buf + 1, k.packet.length);
    ret = krb_check_auth (&k.packet, CKSUM, &k.msg_data,
			  k.cred.session, k.sched, &laddr, &server);
    if (ret)
      fprintf (stderr, "%s: authentication failed: %s\n",
	       program_name, krb_get_err_text (ret));

    /* Now that authentication has succeeded (presumably), get an "ok"
       response if authorization succeeds, or an error if it fails.
       (Not everyone listed in the Kerberos database is necessarily
       permitted to retrieve information from GNATS.)  */
    get_reply ();

  krb_exit:
    /* Zero out the Kerberos scratch area, since it includes
       authentication information we don't want going into core files
       etc.  */
    memset ((char *) &k, '?', sizeof (k));
  }
#endif
}

void
client_exit ()
{
  /* That's it, say bye-bye.  */
  if (debug)
    fprintf (stderr, "%s: writing `QUIT'\n", program_name);
  fflush (serv_read);
  fprintf (serv_write, "QUIT\r\n");

  fclose (serv_write);
  fclose (serv_read);
}


/* change database command to recognize databse aliases for network
   client commands like nquery-pr. */
void
client_chdb ()
{
  if (! gnats_root)
    return;
  
  /* send the change db command and get server's reply */
  if (debug)
    fprintf (stderr, "%s: writing `CHDB %s'\n", program_name,
	     (gnats_root) ? gnats_root : "");
  fprintf (serv_write, "CHDB %s\r\n", (gnats_root) ? gnats_root : "");
  get_reply();			/* this will exit on error */
}

void
client_user (user, passwd)
     char *user;
     char *passwd;
{
  /* send the user command and get server's reply */
  if (debug)
    fprintf (stderr, "%s: writing `USER %s %s'\n", program_name,
	     user, passwd);
  fprintf (serv_write, "USER %s %s\r\n", user, passwd);
  get_reply();			/* this will exit on error */
}

void
client_init_gnats (user, passwd)
      char *user;
      char *passwd;
{
  
  init_space ();
  client_init ();
  client_chdb ();
  client_user (user, passwd);
}
