#!/usr/local/bin/perl
#
# This code developed based upon work done by Paul Traina, Bill Fenner,
# and others for the FreeBSD project.  Extensions done by Paul Traina
# for Juniper Networks.
#
# Copyright (c) 1995-1998, Paul Traina
# All rights reserved.
#
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# $Id: query-pr-summary.cgi,v 1.11 1997/02/02 22:07:01 pst Exp $
#

$html_mode     = 1 if $ENV{'DOCUMENT_ROOT'};
$self_ref      = $ENV{'SCRIPT_NAME'};
($query_pr_ref = $self_ref) =~ s/-summary//;

$ENV{'PATH'}   = '/bin:/usr/bin:/usr/sbin:/sbin:/usr/local/bin';
$libdir	       = '/volume/httpd/cgi-bin';
$query_pr      = '/usr/local/bin/nquery-pr';

$project       = "Juniper";		# FreeBSD
$mail_prefix   = "bug-";		# freebsd-
$mail_unass    = "bugs";		# freebsd-bugs

push(@INC, $libdir);

use Getopt::Long;

require "cgi-lib.pl";
require "cgi-style.pl";

sub create_field {
	local *table = shift;
	my $name  = shift;
	my $short = shift;
	my $sql   = shift;
	my %rec   = ( name => $name, sql => $sql, short => $short);

	push(@table, \%rec);
};

sub field_from_sql {
	local *table  = shift;
	my $sqlval = shift;
	my $entry;
	foreach $entry (@table) {
	    return $entry if $entry->{sql} eq $sqlval;
	}
	return undef;
}
sub field_from_name {
	local *table  = shift;
	my $name = shift;
	my $entry;
	foreach $entry (@table) {
	    return $entry if $entry->{name} eq $name;
	}
	return undef;
}

sub field_names {
    my $field_type;
    my @result;

    foreach $field_type (@_) {
	$_ = $field_type->{name};
	next if $_ eq '?error?';
	push(@result, $_);
    }
    return @result;
}

&create_field(\@f_classes,	"?error?",		"?",	0);
&create_field(\@f_classes,	"sw-bug",		"s",	1);
&create_field(\@f_classes,	"doc-bug",		"d",	2);
&create_field(\@f_classes,	"support",		"S",	3);
&create_field(\@f_classes,	"change-request",	"c",	4);
&create_field(\@f_classes,	"mistaken",		"m",	5);
&create_field(\@f_classes,	"duplicate",		"x",	6);

# Juniper additions...
&create_field(\@f_classes,	"unreproducable",	"u",	7);
&create_field(\@f_classes,	"hw-bug",		"h",	8);

&create_field(\@f_severities,	"?error?",		"?",	0);
&create_field(\@f_severities,	"critical",		"c",	1);
&create_field(\@f_severities,	"serious",		"s",	2);
&create_field(\@f_severities,	"non-critical",		"n",	3);

&create_field(\@f_priorities,	"?error?",		"?",	0);
&create_field(\@f_priorities,	"high",			"h",	1);
&create_field(\@f_priorities,	"medium",		"m",	2);
&create_field(\@f_priorities,	"low",			"l",	3);

&create_field(\@f_sort,		"none",			"",	0);
&create_field(\@f_sort,		"category",		"",	1);
&create_field(\@f_sort,		"priority",		"",	2);
&create_field(\@f_sort,		"responsible",		"",	3);
&create_field(\@f_sort,		"severity",		"",	4);

&get_states;
&get_categories;

if ($ENV{'QUERY_STRING'} eq 'query') {
	print &html_header("Query $project problem reports");
	&displayform;
	print &html_footer;
	exit(0);
}

if ($html_mode) {
	$query_args    = '';
	$wide = 1;
	$input{"closedtoo"} = "--skip-closed";
	&ReadParse(*input);

} else {

	%optctl = (
		"closed",	=> \$closed,
		"quiet"		=> \$input{quiet},
		"restricted"	=> \$restricted,
		"wide"		=> \$wide,

		"sort"		=> \$input{sort},
		"category"	=> \$input{category},
		"class"		=> \$input{class},
		"priority"	=> \$input{priority},
		"responsible"	=> \$input{responsible},
		"severity",	=> \$input{severity},
		"state"		=> \$input{state},
		"arrived-before",=>\$input{'arrived-before'},
		"arrived-after",=> \$input{'arrived-after'},
	);

	GetOptions(\%optctl,
		   "closed", "quiet", "restricted", "wide", "sort=s",
		   "class=s", "category=s", "priority=s", "responsible=s",
		   "severity=s", "state=s", "arrived-before=s",
		   "arrived-after=s") ||
	    die "usage:\n" .
"--closed		Include closed reports\n",
"--quiet			Skip headers\n",
"--restricted		Omit confidential reports\n",
"--wide			Unlimited line length\n\n",
"--arrived-after=<date>	PR arrived after <date>\n",
"--arrived-before=<date>	PR arrived before <date>\n",
"--category=<>		summary:<category>\n",
"--class=<>		summary:",
	join(':', &field_names(@f_classes)), "\n",
"--priority=<>		summary:",
	join(':', &field_names(@f_priorities)), "\n",
"--responsible=<>	summary:<individual>\n",
"--severity=<>		summary:",
	join(':', &field_names(@f_severities)), "\n",
"--sort=<>		",
	join(':', &field_names(@f_sort)), "\n",
"--state=<>		summary:",
	join(':', &field_names(@f_states)), "\n";

	$input{closedtoo}  = '--skip-closed' unless $closed;
	$query_args	  .= '--restricted ' if $restricted;
}

#------------------------------------------------------------------------

if ($html_mode) {
    $h1 = "<h1>"; $h1_e = "</h1>";
    $h2 = "<h2>"; $h2_e = "</h2>";
    $h3 = "<h3>"; $h3_e = "</h3>";
    $h4 = "<h4>"; $h4_e = "</h4>";
    $p  = "<p>" ; $p_e  = "</p>";
    $br = "<br>";

    $st = "<strong>"; $st_e = "</strong>";
    $pr = "<pre>";    $pr_e = "</pre>";
    $dl = "<dl>";     $dl_e = "</dl>";
    $dt = "<dt>";
    $dd = "<dd>";     $dd_x = "";
    $hr = "<hr>";

#    print "Content-type: text/html\n";

} else {

    $h1 = ""; $h1_e = "";
    $h2 = ""; $h2_e = "";
    $h3 = ""; $h3_e = "";
    $h4 = ""; $h4_e = "";
    $p  = ""; $p_e  = "";
    $br = "";
    $st = ""; $st_e = "";
    $pr = ""; $pr_e = "";
    $dl = ""; $dl_e = "";
    $dt = "";
    $dd = "     "; $dd_x = "     ";
    $hr = "\n----------------------------------------" .
	  "---------------------------------------\n";
}

sub header_info {

    if ($html_mode) {
	print &html_header("Current $project problem reports");
    }
    else {
	print "Current $project problem reports\n\n";
    }

print <<EOM unless $input{"quiet"};

The following is a listing of current problems submitted by $project users.
These represent problem reports covering all versions including
experimental development code and obsolete releases.
${p}
Bugs can be in one of several states:
${dl}
${dt}${st}o - open${st_e}
${dd}A problem report has been submitted, no sanity checking performed.

${dt}${st}a - analyzed${st_e}
${dd}The report has been examined by a team member and evaluated.

${dt}${st}f - feedback${st_e}
${dd}The problem has been solved, and the originator has been given a
${dd_x}patch or a fix has been committed.  The PR remains in this state
${dd_x}pending a response from the originator.

${dt}${st}s - suspended${st_e}
${dd}Work on the problem has been postponed.  This happens if a
${dd_x}timely solution is not possible or is not cost-effective at
${dd_x}the present time.  The PR continues to exist, though a solution
${dd_x}is not being actively sought.  If the problem cannot be solved at all,
${dd_x}it will be closed, rather than suspended.

${dt}${st}c - closed${st_e}
${dd}A problem report is closed when any changes have been integrated,
${dd_x}documented, and tested.
${dl_e}
EOM

	if ($html_mode) {

# These self references are attempts to only change a single variable at a time.
# If someone does a multiple-variable query they will probably do weird things.

$self_ref1 = $self_ref . '?';
$self_ref1 .= 'sort=' . $input{sort} if $input{sort};
print '<P>You may view summaries by <A HREF="', $self_ref1, '">Severity</A>, ';
$self_ref1 .= '&' if ($self_ref1 !~/\?$/);
print '<A HREF="', $self_ref1, 'state=summary">State</A>, ';
print '<A HREF="', $self_ref1, 'category=summary">Category</A>, ';
print '<A HREF="', $self_ref1, 'priority=summary">Category</A>, ';
print '<A HREF="', $self_ref1, 'class=summary">Class</A>, or ';
print '<A HREF="', $self_ref1, 'responsible=summary">Responsible Party</A>.';

$self_ref2 = $self_ref . '?';
foreach ("category", "originator", "priority", "class", "responsible",
	 "severity", "state", "submitter", "text", "multitext", "closedtoo") {
	if ($input{$_}) {
		$self_ref2 .= '&' if ($self_ref2 !~/\?$/);
		$self_ref2 .= $_ . '=' . $input{$_};
	}
}

print '<BR>You may also sort by ';
print '<A HREF="', $self_ref2, '&sort=category">Category</A>, ';
print '<A HREF="', $self_ref2, '&sort=priority">Priority</A>, ';
print '<A HREF="', $self_ref2, '&sort=severity">Severity</A>, or ';
print '<A HREF="', $self_ref2, '&sort=responsible">Responsible Party</A>.', "\n";
print 'Or <A HREF="', $self_ref, '?query">formulate a specific query</A>.';
	}
}

sub trailer_info {
    print &html_footer if $html_mode;
}

&header_info;

#Usage: query-pr [-FhiPqVx] [-C confidential] [-c category] [-d directory]
#       [-e severity] [-m mtext] [-O originator] [-o outfile] [-p priority]
#       [-L class] [-r responsible] [-S submitter] [-s state] [-t text]
#       [--full] [--help] [--sql] [--print-path] [--summary] [--version]
#       [--skip-closed] [--category=category] [--confidential=yes|no]
#       [--directory=directory] [--output=outfile] [--originator=name]
#       [--priority=level] [--class=class] [--responsible=person]
#       [--severity=severity] [--state=state] [--submitter=submitter]
#       [--list-categories] [--list-responsible] [--list-submitters]
#       [--text=text] [--multitext=mtext] [PR] [PR]...

# Only read the appropriate PR's.
foreach ("category", "originator", "priority", "class", "responsible",
	 "severity", "state", "submitter", "text", "multitext",
	 "arrived-before", "arrived-after") {
	if ($input{$_} && $input{$_} ne "summary") {
		$query_args .= " --${_}=\'$input{$_}\'";
	}
}

&read_gnats($query_args);

if ($input{sort} eq 'category') {
	@prs = sort { $a->{'category'} eq $b->{'category'} ?
			$a->{'number'} <=> $b->{'number'}  :
			$a->{'category'} cmp $b->{'category'}
		    } @prs;

} elsif ($input{sort} eq 'responsible') {
	@prs = sort { $a->{responsible} eq $b->{responsible} ?
			$a->{number} <=> $b->{number}  :
			$a->{responsible} cmp $b->{responsible}
		    } @prs;

} elsif ($input{sort} eq 'priority') {
	@prs = sort { ($a->{priority})->{sql} eq ($b->{priority})->{sql} ?
			$a->{number} <=> $b->{number}  :
			($a->{priority})->{sql} cmp ($b->{priority})->{sql}
		    } @prs;

} elsif ($input{sort} eq 'severity') {
	@prs = sort { ($a->{severity})->{sql} eq ($b->{severity})->{sql} ?
			$a->{number} <=> $b->{number}  :
			($a->{severity})->{sql} cmp ($b->{severity})->{sql}
		    } @prs;

} else {
	$input{sort} = 'none';
}

if ($#prs < $[) {
	print "${h1}No matches to your query${h1_e}\n";

} elsif ($input{'responsible'} eq 'summary') {
	&get_responsibles;
	&norm_summary('responsible', @f_responsibles);

	&printcnt(&gnats_summary("Unassigned problems",
	          '$entry->{responsible} eq ""'));
		  
} elsif ($input{'state'} eq 'summary') {
	&enum_summary('state', @f_states);

} elsif ($input{'category'} eq 'summary') {
	&norm_summary('category', @f_categories);

} elsif ($input{'priority'} eq 'summary') {
	&enum_summary('priority', @f_priorities);

} elsif ($input{'severity'} eq '' || $input{'severity'} eq 'summary') {
	&enum_summary('severity', @f_severities);

} else {
	&printcnt(&gnats_summary("Requested problem reports", 1));
}

&trailer_info;
exit(0);

#------------------------------------------------------------------------

sub getline {
    local($_) = @_;
    ($tag,$remainder) = split(/[ \t]+/, $_, 2);
    return $remainder;
}

sub html_fixline {
    local($line) = @_[0];

    $line =~ s/&/&amp;/g;
    $line =~ s/</&lt;/g;
    $line =~ s/>/&gt;/g;

    $line;
}

sub printcnt {
    local($cnt) = $_[0];

    if ($cnt) {
	printf("%d problem%s total.\n\n", $cnt, $cnt == 1 ? "" : "s");
    }
}

sub norm_query {
    my $category = shift;
    my $field	 = shift;

    &printcnt(&gnats_summary("$field $category problems",
	    '$entry->{\'' . $category . '\'} eq "' .  $field . '"'));
}

sub norm_summary {
    my $category = shift;
    my %field_type;

    foreach $field_type (@_) {
	&norm_query($category, $field_type->{name});
    }
}

sub enum_query {
    my $category = shift;
    my $field    = shift;

    &printcnt(&gnats_summary("$field $category problems",
	    '($entry->{\'' . $category . '\'})->{\'name\'} eq "' .
	    $field . '"'));
}

sub enum_summary {
    my $category = shift;
    my %field_type;

    foreach $field_type (@_) {
	&enum_query($category, $field_type->{name});
    }
}

sub get_categories {
    open(Q, "$query_pr --list-categories |") ||
	die "Cannot get categories\n";

    my $sql = 0;
    &create_field(\@f_categories, "?error?", "?", $sql++);

    while(<Q>) {
	chomp;
	my ($cat, $desc, $responsible, $notify) = split(/:/);
	&create_field(\@f_categories, $cat, $cat, $sql++);
    }
    close(Q);
}

sub get_states {
    open(Q, "$query_pr --list-states |") ||
	die "Cannot get states\n";

    my $sql = 0;
    &create_field(\@f_states, "?error?", "?", $sql++);

    while(<Q>) {
	chomp;
	my ($state, $desc) = split(/:/);
	&create_field(\@f_states, $state, substr($state, 0, 1), $sql++);
    }
    close (Q);
}

sub get_responsibles {
    open(Q, "$query_pr --list-responsible |") ||
	die "Cannot get resposible parties\n";

    my $sql = 0;
    &create_field(\@f_responsibles, "?error?", "?", $sql++);

    while(<Q>) {
	chomp;
	my ($resp, $desc) = split(/:/);
	&create_field(\@f_responsibles, $resp, $desc, $sql++);
    }
    close (Q);
}

sub read_gnats {
    my $report = shift;

    open(Q, "$query_pr --sql $report " . $input{"closedtoo"} . " |") 
	|| die "Cannot query the PR's\n";

    while(<Q>) {
	chomp;

	my ($s_number, $s_category, $s_synopsis, $s_confidential,
	    $s_severity, $s_priority, $s_responsible, $s_state,
	    $s_class, $s_submitter, $s_arrival_date, $s_originator,
	    $s_release) = split(/\s*\|/);

	$s_arrival_date =~ s/\s+.*//;
	$s_arrival_date =~ s/^19//;

	$s_responsible  =~ s/@.*//;
	$s_responsible  =~ tr/A-Z/a-z/;
	$s_responsible  =  "" if ($s_responsible =~ /^$mail_unass/);
	$s_responsible  =  "" if ($s_responsible =~ /^$mail_prefix/);
	$s_responsible  =~ s/^$mail_prefix//;

	my %rec = (
		'number'	=> $s_number,
		'category'	=> $s_category,
		'synopsis'	=> $s_synopsis,
		'confidential'	=> $s_confidential,
		'severity'	=> &field_from_sql(\@f_severities, $s_severity),
		'priority'	=> &field_from_sql(\@f_priorities, $s_priority),
		'responsible'	=> $s_responsible,
		'state'		=> &field_from_sql(\@f_states, $s_state),
		'class'		=> &field_from_sql(\@f_classes, $s_class),
		'submitter'	=> $s_submitter,
		'arrival'	=> $s_arrival_date,
		'originator'	=> $s_originator,
		'release'	=> $s_release,
	);

	push(@prs, \%rec);
    }
    close(Q);
}

sub gnats_summary {
    my $header		= ucfirst(shift);
    my $report		= shift;
    my $counter		= 0;

    foreach $entry (@prs) {
	next if (($report ne '') && (eval($report) == 0));

	if ($counter++ == 0) {
	    print "${h3}${header}:${h3_e}\n";
	    print "${pr}\nVPSC Arrived  ID                 " .
			 "Respons. Description${hr}"
	}

	my $synopsis = $entry->{synopsis};
	my $tracker  = $entry->{category} . '/' . $entry->{number};
	my $title    = $tracker;
	my $severity = ($entry->{severity})->{short};
	my $priority = ($entry->{priority})->{short};
	my $state    = ($entry->{state})->{short};
	my $class    = ($entry->{class})->{short};
	my $respons  = substr($entry->{responsible}, 0, 8);

	if ($html_mode) {
	    $synopsis = &html_fixline($synopsis);
	    $title    = "<a href=\"${query_pr_ref}?pr=" . $entry->{number} .
			"\">${tracker}</a>";
	}

        printf "%s%s%s%s %s %s%s %-8.8s %s\n",
		$severity,
		$priority,
		$state,
		$class,
		$entry->{arrival},
		$title, ' ' x (18 - length($tracker)),
		$entry->{responsible},
		($wide ? $synopsis : substr($synopsis, 0, 37));
    }

    print "${pr_e}\n" if $counter;
    return $counter;
}

sub enumform {
    my $result = "";

    foreach (&field_names(@_)) {
	$result .= "<OPTION>$_</OPTION>\n";
    }

    return $result;
}

sub displayform {
print qq`
Please select the items you wish to search for.  Multiple items are AND'ed
together.
<P>
<FORM METHOD=GET ACTION="`, $self_ref, qq`">

<TABLE>
<TR>
<TD><B>Category</B>:</TD>
<TD><SELECT NAME="category">
<OPTION SELECTED VALUE="">Any</OPTION>`, &enumform(@f_categories),

qq`</SELECT></TD>
<TD><B>Severity</B>:</TD>
<TD><SELECT NAME="severity">
<OPTION SELECTED VALUE="">Any</OPTION>`, &enumform(@f_severities),

qq`</SELECT></TD>
</TR><TR>
<TD><B>Priority</B>:</TD>
<TD><SELECT NAME="priority">
<OPTION SELECTED VALUE="">Any</OPTION>`, &enumform(@f_priorities),

qq`</SELECT></TD>
<TD><B>Class</B>:</TD>
<TD><SELECT NAME="class">
<OPTION SELECTED VALUE="">Any</OPTION>`, &enumform(@f_classes),

qq`</SELECT></TD>
</TR><TR>
<TD><B>State</B>:</TD>
<TD><SELECT NAME="state">
<OPTION SELECTED VALUE="">Any</OPTION>`, &enumform(@f_states),

qq`</SELECT></TD>
<TD><B>Sort by</B>:</TD>
<TD><SELECT NAME="sort">
<OPTION SELECTED VALUE="none">No Sort</OPTION>`, &enumform(@f_sort),

qq`</SELECT></TD>
</TR><TR>
<!-- We don't use submitter Submitter: -->
<TD><B>Text in single-line fields</B>:</TD>
<TD><INPUT TYPE=TEXT NAME="text"></TD>
<TD><B>Responsible</B>:</TD>
<TD><INPUT TYPE=TEXT NAME="responsible"></TD>
</TR><TR>
<TD><B>Text in multi-line fields</B>:</TD>
<TD><INPUT TYPE=TEXT NAME="multitext"></TD>
<TD><B>Originator</B>:</TD>
<TD><INPUT TYPE=TEXT NAME="originator"></TD>
</TR><TR>
<TD><B>Closed reports too</B>:</TD>
<TD><INPUT NAME="closedtoo" TYPE=CHECKBOX VALUE="--skip-closed"></TD>
</TR>
</TABLE>
<INPUT TYPE="SUBMIT" VALUE=" Query PR's ">
<INPUT TYPE="RESET" VALUE=" Reset Form ">
</FORM>
`;
}
