/* Commands for the GNATS server.
   Copyright (C) 1994, 95, 95, 1997 Free Software Foundation, Inc.
   Contributed by Mike Sutton (mike_sutton@dayton.saic.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more detailmails.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

/* COMMENTARY:
 *
 * This file contains the functions that operate on the database
 * configuration entry data structure. It contains an initialization
 * routine (init_db_conf) that reads the config file and a free
 * routine (free_db_conf) to release the memory.
 *
 * Once the initialization is performed the following fuctions can be
 * used to access the data: lookup_conf_alias, get_conf_aliases,
 * get_conf_databases, local_chdb.
 */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"
#include "query.h"
#include "db_conf.h"

/* This stucture holds data from the DB_CONF_FILE. The same structure
 will be used to hold a lookup table of aliases and a look up table of
 available databases. */

#define DB_CONF_FILE "/etc/gnats-db.conf"


/* declarations for private functions */
static int get_conf_key_list_and_count  PARAMS((Conf_entry *list, char ***keys,
					int *count));
static int get_conf_entry_count         PARAMS(( Conf_entry *head));
static int get_conf_keys                PARAMS((Conf_entry *ptr, char ***keys,
						int count));
static Conf_entry *lookup_conf_entry    PARAMS((Conf_entry *head, char *key));
static Conf_entry *add_conf_entry       PARAMS((Conf_entry **head, char *key,
						char *value));
static void free_conf_list              PARAMS((Conf_entry *ptr));


/* private  variables */
Conf_entry *Conf_aliases   = NULL;
Conf_entry *Conf_databases = NULL;

/* public function.
 *
 * initialize the data found in the etc config files.  The function
 * will return non-zero status for failures.
 */

int
init_db_conf()
{
  FILE *fp;

  char line[STR_MAX];

  if (Conf_aliases && Conf_databases) /* already initialized */
    return 0;

  /* open the configuration file and begin reading. */
  fp = fopen (DB_CONF_FILE, "r");
  if (fp == NULL)
    return 1;

  while (read_string (line, fp) > 0)
    {
      if (line[0] != '#' && line[0] != ' ' && line[0] != '\n')
	{
	  char token[STR_MAX];
	  char *l, *l2, *aliases, *dir_name;

	  /* array[0] - dir name
	   * array[1] - comma separated alias list */
	  aliases = strchr(line, ':');
	  if (aliases)
	    {
	      /* zero out the : and increment forward one to make
                 point reference the aliases */
	      *aliases = '\0';
	      aliases++;
	      dir_name = line;

	      /* parse out the aliases and add them to the list */
	      for (l2 = aliases, l = get_next_field(l2, token, ',');
		   l || l2;
		   (l2 = l) && (l = get_next_field(l, token, ',')))
		add_conf_entry(&Conf_aliases, token, dir_name);

	      /* store a list of databases */
	      add_conf_entry(&Conf_databases, dir_name, dir_name);
	    } /* end if aliases */

	} /* end if line ! comment */
    } /* end while */

  fclose (fp);
  return 0;
}

/* public function */
void
free_db_conf()
{
  free_conf_list(Conf_aliases);
  free_conf_list(Conf_databases);
}

/* public function.
 *
 * This returns the fully qualified path name for the specified
 * alias. NULL is returned if the alias is not found.
 */
char *
lookup_conf_alias(key)
     char *key;
{
  Conf_entry *alias = lookup_conf_entry(Conf_aliases, key);
  return (alias != NULL) ? alias->value : NULL;
}


/* public function.
 *
 * this lists the aliases. Aliases will be allocated and
 * populated. You must free the memory allocated to each alias and the
 * alias list itself.
 */
int
get_conf_aliases(aliases, count)
     char ***aliases;
     int  *count;
{
  return get_conf_key_list_and_count(Conf_aliases, aliases, count);
}


/* public function:
 *
 * get the list of known databases. databases will be allocated and
 * populated. You must free the memory allocated to each database and
 * databases itself.
 */
int
get_conf_databases(databases, count)
     char ***databases;
     int *count;
{

  return get_conf_key_list_and_count(Conf_databases, databases, count);
}




/* public function:
 * 
 *  change database command to recognize database aliases for local
 *  commands like query-pr.
 */
int
local_chdb(new_root, program_name)
     char **new_root;		/* RETURNED */
     char *program_name;
{
  char  *new_db = *new_root;
  char  *alias_db = NULL;

  FILE *output_fp = stderr;

  struct stat buf;

  /* Any errors changing the database must be fatal to stop following operations
     from being performed on the wrong database (such as gen-index, for example) */
  
  /* If there is no slash assume an alias was specified. */
  if (strchr (new_db, '/') == NULL)
    {
      alias_db = new_db;
      new_db = lookup_conf_alias(alias_db);
      if (!new_db)
	{
	fprintf (output_fp, "%s: Could not resolve database alias (%s).\n",
		program_name, alias_db);
	exit (1);
	}
    }

  if (stat(new_db, &buf) != 0)
    {
      if (errno == ENOENT)
	fprintf (output_fp, "%s: Could not change to database (%s): does not exist.\r\n",
		program_name, new_db);
      else
	fprintf (output_fp, "%s: Error while getting status of (%s): errno = %d.\r\n",
		program_name, new_db, errno);
      exit (1);
    }
  else if ((buf.st_mode & S_IFMT) != S_IFDIR)
    {
      fprintf (output_fp, "%s: Specified database (%s) is not a directory.\r\n",
	      program_name, new_db);
      exit (1);
    }

  /* everything checked out OK so now replace the incoming string */
  *new_root = xstrdup(new_db);

  return 0;
}

/* ---------- Begin Private Functions ---------- */

/* private functions */
static int
get_conf_key_list_and_count(list, keys, count)
     Conf_entry *list;
     char       ***keys;
     int        *count;
{
  *count = get_conf_entry_count(list);

  if (*count > 0)
    get_conf_keys(list, keys, *count);

  return *count;
}

/* private function */
static int
get_conf_entry_count(head)
     Conf_entry *head;
{
  Conf_entry *ptr = head;
  int count = 0;

  while(ptr != NULL)
    {
      ptr = ptr->next;
      count++;
    }
  return count;
}

/* private function to load keys into a dynamic array */
static int
get_conf_keys(ptr, keys, count)
     Conf_entry *ptr;
     char       ***keys;
     int        count;
{
  if (count > 0)
    {
      int size = count * sizeof(char*);
      int i;

      *keys = (char **) xmalloc(size);
      memset(*keys, 0, size);

      i = 0;
      while (ptr != NULL)
	{
	  *(*keys + i) = xstrdup(ptr->key);
	  i++;
	  ptr = ptr->next;
	}
    }
}


/* private function */

static Conf_entry *
lookup_conf_entry(head, key)
     Conf_entry *head;
     char       *key;
{
  Conf_entry *ptr;

  if (key == NULL)
    return NULL;

  for(ptr = head; ptr != NULL; ptr = ptr->next)
    {
      if (!strcmp(ptr->key, key))
	{
	  return ptr;
	}
    }
  return NULL;
}

/* this add routine takes the address of a pointer to the list, the
 * key to the list, and the value for that key.  There can only be
 * one entry for a given key.
 * The routine returns a pointer to the new or replaced entry and
 * NULL on error.
 */

static Conf_entry *
add_conf_entry(head, key, value)
     Conf_entry **head;
     char *key;
     char *value;
{
  Conf_entry *ptr = *head;
  Conf_entry *repl = NULL;

  if (key == NULL)
    return NULL;

  if ( (repl = lookup_conf_entry(*head, key)) == NULL)
    {
      if (*head == NULL)
	{
	  *head = (Conf_entry *) xmalloc (sizeof(Conf_entry));
	  ptr = *head;
	}
      else
	{
      	  /* go to end of list */
	  while (ptr->next != NULL)
	    ptr = ptr->next;

	  /* add new entry */
	  ptr->next = (Conf_entry *) xmalloc (sizeof(Conf_entry));
	  ptr = ptr->next;
	}
      memset( (char *)ptr, 0, sizeof(Conf_entry));
      ptr->key = xstrdup(key);
      ptr->value = (value == NULL) ? value : xstrdup(value);
      ptr->next = NULL;
    }
  else
    {
      /* replace existing entry */
      free(repl->value);
      repl->value = (value == NULL) ? value : xstrdup(value);
    }
  return (repl == NULL) ? ptr : repl ;
}


/* private function */
static void
free_conf_list(ptr)
     Conf_entry *ptr;
{
  Conf_entry *next;

  while(ptr != NULL)
    {
      next = ptr->next;
      free(ptr->key);
      free(ptr->value);
      free(ptr);

      ptr = NULL;
      ptr = next;
    }
}
