;; $Id: dbtable.dsl,v 1.1 1998/02/18 13:14:22 rosalia Exp $
;;
;; This file is part of the Modular DocBook Stylesheet distribution.
;; See ../README or http://www.berkshire.net/~norm/dsssl/
;;
;; Table support completely reimplemented by norm 15/16 Nov 1997
;;
;; ======================================================================
;;
;; This code is intended to implement the SGML Open Exchange Table Model
;; (http://www.sgmlopen.org/sgml/docs/techpubs.htm) as far as is possible
;; in RTF.  There are a few areas where this code probably fails to 
;; perfectly implement the model:
;;
;; - Mixed column width units (4*+2pi) are not supported.
;; - The behavior that results from mixing relative units with 
;;   absolute units has not been carefully considered.
;; - TFOOT appears at the bottom of the table, but is not repeated
;;   across the bottom of pages (RTF limitation).
;; - ENTRYTBL is not supported.
;; - Rotated tables (e.g. landscape tables in a portrait document)
;;   cannot be supported in a simple-page-sequence
;;
;; ======================================================================
;; 
;; My goal in reimplementing the table model was to provide correct
;; formatting in tables that use MOREROWS. The difficulty is that
;; correct formatting depends on calculating the column into which
;; an ENTRY will fall.
;;
;; This is a non-trivial problem because MOREROWS can hang down from
;; preceding rows and ENTRYs may specify starting columns (skipping
;; preceding ones).
;;
;; A simple, elegant recursive algorithm exists. Unfortunately it 
;; requires calculating the column number of every preceding cell 
;; in the entire table. Without memoization, performance is unacceptable
;; even in relatively small tables (5x5, for example).
;;
;; In order to avoid recursion, the algorithm used below is one that
;; works forward from the beginning of the table and "passes along"
;; the relevant information (column number of the preceding cell and
;; overhang from the MOREROWS in preceding rows).
;;
;; Unfortunately, this means that element construction rules can't be
;; used to fire the appropriate rule.  Instead, each TGROUP has to
;; process each THEAD/BODY/FOOT explicitly.  And each of those must
;; process each ROW explicitly, then each ENTRY/ENTRYTBL explicitly.
;; Finally, the contents of each ENTRY must be processed explicitly.
;; The code supplied here handles #PCDATA in ENTRYs and PARAs in
;; ENTRYs.  If you have other block elements in ENTRYs that require
;; special formatting, see %cell-block-element-list% below.
;;
;; ----------------------------------------------------------------------
;;
;; I attempted to simplify this code by relying on inheritence from
;; table-column flow objects, but that wasn't entirely successful.
;; Horizontally spanning cells didn't seem to inherit from table-column
;; flow objects that didn't specify equal spanning.  There seemed to
;; be other problems as well, but they could have been caused by coding
;; errors on my part.
;; 
;; Anyway, by the time I understood how I could use table-column
;; flow objects for inheritence, I'd already implemented all the
;; machinery below to "work it out by hand".  
;;
;; ======================================================================
;; NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE 
;; ----------------------------------------------------------------------
;; A fairly large chunk of this code is in dbcommon.dsl!
;; ======================================================================
;;

;; Default value for FRAME= on tables
(define %cals-frame-default% "ALL")

;; Default for COLSEP/ROWSEP if unspecified.  If FRAME=NONE, the default
;; is no-rules; otherwise the default is to have rules.
(define ($cals-rule-default$  #!optional (node (current-node)))
  (let* ((table (ancestor-member node %table-element-list%))
	 (frame (if (attribute-string "FRAME" table)
		    (attribute-string "FRAME" table)
		    %cals-frame-default%)))
    (not (equal? frame "NONE"))))

;; Default for VALIGN if unspecified
(define %cals-valign-default% "TOP")

;; Margins around cell contents
(define %cals-cell-before-row-margin% 3pt)
(define %cals-cell-after-row-margin% 3pt)
(define %cals-cell-before-column-margin% 3pt)
(define %cals-cell-after-column-margin% 3pt)

;; Inheritable start and end indent for cell contents
(define %cals-cell-content-start-indent% 2pt)
(define %cals-cell-content-end-indent% 2pt)

;; How to indent pgwide tables?  (Non-pgwide tables get inherited-start-indent
(define %cals-pgwide-start-indent% %body-start-indent%)

;; ----------------------------------------------------------------------
;; Convert colwidth units into table-unit measurements

(define (colwidth-unit lenstr)
  (if (string? lenstr)
      (let ((number (length-string-number-part lenstr))
	    (units  (length-string-unit-part lenstr)))
	(if (string=? units "*")
	    (if (string=? number "")
		(table-unit 1)
		(table-unit (string->number number)))
	    (if (string=? units "")
		;; no units, default to points
		(* (string->number number) 1pt)
		(let* ((unum  (string->number number))
		       (uname (case-fold-down units)))
		  (case uname
		    (("mm") (* unum 1mm))
		    (("cm") (* unum 1cm))
		    (("in") (* unum 1in))
		    (("pi") (* unum 1pi))
		    (("pt") (* unum 1pt))
		    (("px") (* unum 1px))
		    ;; unrecognized units; use points
		    (else   (* unum 1pt)))))))
      ;; lenstr is not a string...probably #f
      (table-unit 1)))


(define (cell-align cell colnum)
  (let* ((entry     (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (spanname  (attribute-string "SPANNAME" entry))
	 (calsalign (if (attribute-string "ALIGN" entry)
			(attribute-string "ALIGN" entry)
			(if (and spanname 
				 (spanspec-align (find-spanspec spanname)))
			    (spanspec-align (find-spanspec spanname))
			    (if (colspec-align (find-colspec-by-number colnum))
				(colspec-align (find-colspec-by-number colnum))
				(if (tgroup-align (ancestor "TGROUP" entry))
				    (tgroup-align (ancestor "TGROUP" entry))
				    "LEFT"))))))
    (case calsalign
      (("LEFT") 'start)
      (("CENTER") 'center)
      (("RIGHT") 'end)
      (else 'start))))
    
(define (cell-valign cell colnum)
  (let* ((entry      (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (row        (ancestor "ROW" entry))
	 (tbody      (ancestor-member cell '("TBODY" "THEAD" "TFOOT")))
	 (tgroup     (ancestor "TGROUP" entry))
	 (calsvalign (if (attribute-string "VALIGN" entry)
			 (attribute-string "VALIGN" entry)
			 (if (attribute-string "VALIGN" row)
			     (attribute-string "VALIGN" row)
			     (if (attribute-string "VALIGN" tbody)
				 (attribute-string "VALIGN" tbody)
				 %cals-valign-default%)))))
    (case calsvalign
      (("TOP") 'start)
      (("MIDDLE") 'center)
      (("BOTTOM") 'end)
      (else 'start))))

;; ======================================================================
;; Element rules

(element TGROUP
  (let ((frame-attribute (if (inherited-attribute-string "FRAME")
			     (inherited-attribute-string "FRAME")
			     %cals-frame-default%)))
    (make table
      before-row-border:  (case frame-attribute
			    (("ALL") #t)
			    (("SIDES") #f)
			    (("TOP") #t)
			    (("BOTTOM") #f)
			    (("TOPBOT") #t)
			    (("NONE") #f)
			    (else #f))
      after-row-border:   (case frame-attribute
			    (("ALL") #t)
			    (("SIDES") #f)
			    (("TOP") #f)
			    (("BOTTOM") #t)
			    (("TOPBOT") #t)
			    (("NONE") #f)
			    (else #f))
      before-column-border: (case frame-attribute
			      (("ALL") #t)
			      (("SIDES") #t)
			      (("TOP") #f)
			      (("BOTTOM") #f)
			      (("TOPBOT") #f)
			      (("NONE") #f)
			      (else #f))
      after-column-border:  (case frame-attribute
			      (("ALL") #t)
			      (("SIDES") #t)
			      (("TOP") #f)
			      (("BOTTOM") #f)
			      (("TOPBOT") #f)
			      (("NONE") #f)
			      (else #f))
      (make table-part
	content-map: '((thead header)
		       (tbody #f)
		       (tfoot footer))
	(process-children)
	(make-table-endnotes)))))

(element COLSPEC
  (make table-column
    width: (colwidth-unit (attribute-string "COLWIDTH"))))

(element SPANSPEC
  (empty-sosofo))

(element THEAD
  ($process-table-body$ (current-node)))

(element TFOOT
  ($process-table-body$ (current-node)))

(element TBODY
  ($process-table-body$ (current-node)))

(element ROW
  (empty-sosofo)) ;; this should never happen, they're processed explicitly

(element ENTRY
  (empty-sosofo)) ;; this should never happen, they're processed explicitly

;; ======================================================================
;; Functions that handle processing of table bodies, rows, and cells

;; ----------------------------------------------------------------------
;; Note: In the previous implementation of support for CALS tables,
;; additional construction rules would have been necessary to support
;; arbitrary block elements in table cells.  For example, to support
;; special formatting of an ItemizedList in an Entry, you might have
;; needed to add
;;
;; (element (ROW ENTRY ITEMIZEDLIST) ...)
;;
;; and possibly
;;
;; (element (THEAD ROW ENTRY ITEMIZEDLIST) ...)
;;
;; etc.
;;
;; In this version, you might need to extend
;; $process-cell-block-content$.  I don't think that my current
;; solution is significantly worse than the old one in this
;; regard, but I'm uniformly happy with it either.  If you can
;; think of a better way, please tell norm.
;;

(define ($process-table-body$ body)
  (let* ((tgroup (ancestor "TGROUP" body))
	 (cols   (string->number (attribute-string "COLS" tgroup)))
	 (blabel (case (gi body)
		   (("THEAD") 'thead)
		   (("TBODY") 'tbody)
		   (("TFOOT") 'tfoot))))
    (make sequence
      label: blabel
      (let loop ((rows (node-list-filter-out-pis (children body)))
		 (overhang (constant-list 0 cols)))
	(if (node-list-empty? rows)
	    (empty-sosofo)
	    (make sequence
	      ($process-row$ (node-list-first rows) overhang)
	      (loop (node-list-rest rows)
		    (update-overhang (node-list-first rows) overhang))))))))

(define ($process-row$ row overhang)
  (let* ((tgroup (ancestor "TRGROUP" row))
	 (table  (parent tgroup))
	 (rowsep (if (attribute-string "ROWSEP" row)
		     (attribute-string "ROWSEP" row)
		     (if (attribute-string "ROWSEP" tgroup)
			 (attribute-string "ROWSEP" tgroup)
			 (if (attribute-string "ROWSEP" table)
			     (attribute-string "ROWSEP" table)
			     (if ($cals-rule-default$ row)
				 "1"
				 "0")))))
	 (after-row-border (> (string->number rowsep) 0)))
    (make table-row
      cell-after-row-border: after-row-border
      (let loop ((cells (node-list-filter-out-pis (children row))))
	(if (node-list-empty? cells)
	    (empty-sosofo)
	    (make sequence
	      ($process-cell$ (node-list-first cells) overhang)
	      (loop (node-list-rest cells))))))))

(define ($process-cell$ entry overhang)
  (let* (
;;	 (celldebug (debug (string-append 
;;			  (number->string (cell-column-number entry overhang))
;;			  " "
;;			  (data entry))))
	 (colnum (cell-column-number entry overhang)))
    (make table-cell 
      column-number: colnum
      n-columns-spanned: (hspan entry)
      n-rows-spanned: (vspan entry)

      cell-row-alignment: (cell-valign entry colnum)
      cell-after-column-border: (cell-colsep entry colnum)
      cell-after-row-border: (cell-rowsep entry colnum)
      cell-before-row-margin: %cals-cell-before-row-margin%
      cell-after-row-margin: %cals-cell-after-row-margin%
      cell-before-column-margin: %cals-cell-before-column-margin%
      cell-after-column-margin: %cals-cell-after-column-margin%
      start-indent: %cals-cell-content-start-indent%
      end-indent: %cals-cell-content-end-indent%
      (if (equal? (gi entry) "ENTRYTBL")
	  (make paragraph 
	    (literal "ENTRYTBL not supported."))
	  ($process-cell-contents$ entry colnum)))))

(define ($process-cell-contents$ entry colnum)
  (let ((contents (node-list-filter-out-pis (children entry))))
    (if (node-list-empty? contents)
	(empty-sosofo)
	(if (not (gi (node-list-first contents)))
	    ($process-cell-pcdata-contents$ entry colnum)
	    ($process-cell-block-contents$ entry colnum)))))

(define ($process-cell-pcdata-contents$ entry colnum)
  (let ((font-name (if (have-ancestor? "THEAD" entry)
		       %title-font-family%
		       %body-font-family%))
	(weight    (if (have-ancestor? "THEAD" entry)
		       'bold
		       'medium))
	(align     (cell-align entry colnum)))
    (make paragraph
      font-family-name: font-name
      font-weight: weight
      quadding: align
      (process-node-list (children entry)))))

(define ($process-cell-block-contents$ entry colnum)
  ;; NOTE: this can't be done with element construction rules because
  ;; it is necessary to know the colnum in order to get alignment
  ;; correct and colnum is calculated as part of a forward pass through
  ;; the table.  A different algorithm, that allowed each entry to
  ;; calculate its own colnum, was tried, but the result was unusably
  ;; slow in Jade because every cell was required to calculate the 
  ;; colnum of the previous entry in the row and all of the elements
  ;; in the previous row (to get vspans right).  So every entry 
  ;; recalculated every preceding entry.  Recursively.
  ;; 
  ;; I can envision a solution that optimizes the case where there
  ;; are not many vspans, but it would be some effort to implement, so
  ;; I'm content to leave the current solution unless it can be demonstrated
  ;; that it's flawed in some way.
  ;;
  (let* ((row    (ancestor "ROW" entry))
	 (body   (parent row))
	 (align  (cell-align entry colnum)))
    (let loop ((nl (node-list-filter-out-pis (children entry))))
      (if (node-list-empty? nl)
	  (empty-sosofo)
	  (make sequence
	    (let* ((procpair (assoc (gi (node-list-first nl)) 
				    %cell-block-element-list%))
		   (procfunc (if procpair
				 (cdr procpair)
				 #f)))
	      (if procfunc
		  (procfunc (node-list-first nl) colnum align (gi body))
		  ($process-cell-para-contents$ 
		   (node-list-first nl) colnum align (gi body))))
	      (loop (node-list-rest nl)))))))

;; ----------------------------------------------------------------------

;; This list provides the mapping that would ordinarily be supplied
;; by the element construction rules.  If the block element "FOO"
;; occurs inside a table cell, use the $process-foo$ function to
;; process it's _contents_:

(define %cell-block-element-list%
  `(;; ("FOO" . ,$process-foo$)
    ("PARA" . ,$process-cell-para-contents$)
    ))

(define ($process-cell-para-contents$ para colnum align body)
  (let ((font-name (case body
		     (("THEAD") %title-font-family%)
		     (("TFOOT") %title-font-family%)
		     (else  %body-font-family%)))
	(weight    (case body
		     (("THEAD") 'bold)
		     (("TFOOT") 'bold)
		     (else  'medium))))
    (make paragraph
      font-family-name: font-name
      font-weight: weight
      quadding: align
      (process-node-list (children para)))))

;; EOF
