;; $Id: dbtable.dsl,v 1.1 1998/02/18 13:13:50 rosalia Exp $
;;
;; This file is part of the Modular DocBook Stylesheet distribution.
;; See ../README or http://www.berkshire.net/~norm/dsssl/
;;
;; Table support completely reimplemented by norm 15/16 Nov 1997.
;; Adapted from print support.
;;
;; ======================================================================
;;
;; This code is intended to implement the SGML Open Exchange Table Model
;; (http://www.sgmlopen.org/sgml/docs/techpubs.htm) as far as is possible
;; in RTF.  There are a few areas where this code probably fails to 
;; perfectly implement the model:
;;
;; - Mixed column width units (4*+2pi) are not supported.
;; - The behavior that results from mixing relative units with 
;;   absolute units has not been carefully considered.
;; - TFOOT appears at the bottom of the table, but is not repeated
;;   across the bottom of pages (RTF limitation).
;; - ENTRYTBL is not supported.
;; - Rotated tables (e.g. landscape tables in a portrait document)
;;   cannot be supported in a simple-page-sequence
;;
;; ======================================================================
;; 
;; My goal in reimplementing the table model was to provide correct
;; formatting in tables that use MOREROWS. The difficulty is that
;; correct formatting depends on calculating the column into which
;; an ENTRY will fall.
;;
;; This is a non-trivial problem because MOREROWS can hang down from
;; preceding rows and ENTRYs may specify starting columns (skipping
;; preceding ones).
;;
;; A simple, elegant recursive algorithm exists. Unfortunately it 
;; requires calculating the column number of every preceding cell 
;; in the entire table. Without memoization, performance is unacceptable
;; even in relatively small tables (5x5, for example).
;;
;; In order to avoid recursion, the algorithm used below is one that
;; works forward from the beginning of the table and "passes along"
;; the relevant information (column number of the preceding cell and
;; overhang from the MOREROWS in preceding rows).
;;
;; Unfortunately, this means that element construction rules can't be
;; used to fire the appropriate rule.  Instead, each TGROUP has to
;; process each THEAD/BODY/FOOT explicitly.  And each of those must
;; process each ROW explicitly, then each ENTRY/ENTRYTBL explicitly.
;; Finally, the contents of each ENTRY must be processed explicitly.
;; The code supplied here handles #PCDATA in ENTRYs and PARAs in
;; ENTRYs.  If you have other block elements in ENTRYs that require
;; special formatting, see %cell-block-element-list% below.
;;
;; ----------------------------------------------------------------------
;;
;; I attempted to simplify this code by relying on inheritence from
;; table-column flow objects, but that wasn't entirely successful.
;; Horizontally spanning cells didn't seem to inherit from table-column
;; flow objects that didn't specify equal spanning.  There seemed to
;; be other problems as well, but they could have been caused by coding
;; errors on my part.
;; 
;; Anyway, by the time I understood how I could use table-column
;; flow objects for inheritence, I'd already implemented all the
;; machinery below to "work it out by hand".  
;;
;; ======================================================================
;; NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE 
;; ----------------------------------------------------------------------
;; A fairly large chunk of this code is in dbcommon.dsl!
;; ======================================================================

;; Default for COLSEP/ROWSEP if unspecified
(define %cals-rule-default% "0")

;; Default for VALIGN if unspecified
(define %cals-valign-default% "TOP")

;; ======================================================================
;; Convert colwidth units into table-unit measurements

(define (colwidth-length lenstr)
  (if (string? lenstr)
      (let ((number (length-string-number-part lenstr))
	    (units  (length-string-unit-part lenstr)))
	(if (or (string=? units "*") (string=? number ""))
	    ;; relative units or no number, give up
	    0pt
	    (if (string=? units "")
		;; no units, default to pixels
		(* (string->number number) 1px)
		(let* ((unum  (string->number number))
		       (uname (case-fold-down units)))
		  (case uname
		    (("mm") (* unum 1mm))
		    (("cm") (* unum 1cm))
		    (("in") (* unum 1in))
		    (("pi") (* unum 1pi))
		    (("pt") (* unum 1pt))
		    (("px") (* unum 1px))
		    ;; unrecognized units; use pixels
		    (else   (* unum 1px)))))))
      ;; lenstr is not a string...probably #f
      0pt))

(define (cals-relative-colwidth? colwidth)
  (if (string? colwidth)
      (let ((strlen (string-length colwidth)))
	(if (string=? colwidth "*")
	    #t
	    (string=? (substring colwidth (- strlen 1) strlen) "*")))
      #f))

(define (cals-relative-colwidth colwidth)
  (let ((number (length-string-number-part colwidth))
	(units  (length-string-unit-part colwidth)))
    (if (string=? units "*")
	(if (string=? number "")
	    1
	    (string->number number))
	0)))

(define (cell-relative-colwidth cell relative)
  (let* ((tgroup (ancestor "TGROUP" cell)))
    (let loop ((colspecs (select-elements (children tgroup) "COLSPEC"))
	       (reltotal 0))
      (if (node-list-empty? colspecs)
	  (string-append (number->string (round (* (/ relative reltotal) 100))) "%")
	  (loop (node-list-rest colspecs) 
		(+ reltotal (cals-relative-colwidth 
			     (colspec-colwidth 
			      (node-list-first colspecs)))))))))

(define (cell-colwidth cell colnum)
  (let* ((entry     (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (colspec   (find-colspec-by-number colnum))
	 (colwidth  (colspec-colwidth colspec))
	 (width     (round (/ (colwidth-length colwidth) 1px))))
    (if (node-list-empty? colspec)
	""
	(if (and (equal? (hspan entry) 1) colwidth)
	    (if (cals-relative-colwidth? colwidth)
		(cell-relative-colwidth cell (cals-relative-colwidth colwidth))
		(number->string width))
	    ""))))

;; ======================================================================

(define (cell-align cell colnum)
  (let* ((entry     (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (spanname  (attribute-string "SPANNAME" entry))
	 (calsalign (if (attribute-string "ALIGN" entry)
			(attribute-string "ALIGN" entry)
			(if (and spanname 
				 (spanspec-align (find-spanspec spanname)))
			    (spanspec-align (find-spanspec spanname))
			    (if (colspec-align (find-colspec-by-number colnum))
				(colspec-align (find-colspec-by-number colnum))
				(if (tgroup-align (ancestor "TGROUP" entry))
				    (tgroup-align (ancestor "TGROUP" entry))
				    "LEFT"))))))
    (case calsalign
      (("LEFT") "LEFT")
      (("CENTER") "CENTER")
      (("RIGHT") "RIGHT")
      (else 'start))))
    
(define (cell-valign cell colnum)
  (let* ((entry      (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (row        (ancestor "ROW" entry))
	 (tbody      (ancestor-member cell '("TBODY" "THEAD" "TFOOT")))
	 (tgroup     (ancestor "TGROUP" entry))
	 (calsvalign (if (attribute-string "VALIGN" entry)
			 (attribute-string "VALIGN" entry)
			 (if (attribute-string "VALIGN" row)
			     (attribute-string "VALIGN" row)
			     (if (attribute-string "VALIGN" tbody)
				 (attribute-string "VALIGN" tbody)
				 %cals-valign-default%)))))
    (case calsvalign
      (("TOP") "TOP")
      (("MIDDLE") "CENTER")
      (("BOTTOM") "BOTTOM")
      (else 'start))))

;; ======================================================================
;; Element rules

(element TGROUP
  (let* ((wrapper (parent (current-node)))
	 (frameattr (attribute-string "FRAME" wrapper))
	 (footnotes (select-elements (descendants (current-node)) "FOOTNOTE"))
	 (border (if (equal? frameattr "NONE")
		     '(("BORDER" "0"))
		     '(("BORDER" "1"))))
	 (width (if (and (attribute-string "PGWIDE" wrapper)
			 (equal? (attribute-string "PGWIDE" wrapper) "1"))
		    '(("WIDTH" "100%"))
		    '()))
	 (head (select-elements (children (current-node)) "THEAD"))
	 (body (select-elements (children (current-node)) "TBODY"))
	 (feet (select-elements (children (current-node)) "TFOOT")))
    (make element gi: "TABLE"
	  attributes: (append border width)
	  (process-node-list head)
	  (process-node-list body)
	  (process-node-list feet)
	  (make-table-endnotes))))

(element COLSPEC
  (empty-sosofo))

(element SPANSPEC
  (empty-sosofo))

(element THEAD
  ($process-table-body$ (current-node)))

(element TFOOT
  ($process-table-body$ (current-node)))

(element TBODY
  ($process-table-body$ (current-node)))

(element ROW
  (empty-sosofo)) ;; this should never happen, they're processed explicitly

(element ENTRY
  (empty-sosofo)) ;; this should never happen, they're processed explicitly

;; ======================================================================
;; Functions that handle processing of table bodies, rows, and cells

;; ----------------------------------------------------------------------
;; Note: In the previous implementation of support for CALS tables,
;; additional construction rules would have been necessary to support
;; arbitrary block elements in table cells.  For example, to support
;; special formatting of an ItemizedList in an Entry, you might have
;; needed to add
;;
;; (element (ROW ENTRY ITEMIZEDLIST) ...)
;;
;; and possibly
;;
;; (element (THEAD ROW ENTRY ITEMIZEDLIST) ...)
;;
;; etc.
;;
;; In this version, you might need to extend
;; $process-cell-block-content$.  I don't think that my current
;; solution is significantly worse than the old one in this
;; regard, but I'm uniformly happy with it either.  If you can
;; think of a better way, please tell norm.
;;

(define ($process-table-body$ body)
  (let* ((tgroup (ancestor "TGROUP" body))
	 (cols   (string->number (attribute-string "COLS" tgroup))))
    (let loop ((rows (node-list-filter-out-pis (children body)))
	       (overhang (constant-list 0 cols)))
      (if (node-list-empty? rows)
	  (empty-sosofo)
	  (make sequence
	    ($process-row$ (node-list-first rows) overhang)
	    (loop (node-list-rest rows)
		  (update-overhang (node-list-first rows) overhang)))))))

(define ($process-row$ row overhang)
  (let* ((tgroup (ancestor "TRGROUP" row))
	 (table  (parent tgroup))
	 (rowsep (if (attribute-string "ROWSEP" row)
		     (attribute-string "ROWSEP" row)
		     (if (attribute-string "ROWSEP" tgroup)
			 (attribute-string "ROWSEP" tgroup)
			 (if (attribute-string "ROWSEP" table)
			     (attribute-string "ROWSEP" table)
			     %cals-rule-default%))))
	 (after-row-border (if rowsep
			       (> (string->number rowsep) 0)
			       #f)))
    (make element gi: "TR"
	  (let loop ((cells (node-list-filter-out-pis (children row)))
		     (prevcell (empty-node-list)))
	    (if (node-list-empty? cells)
		(empty-sosofo)
		(make sequence
		  ($process-cell$ (node-list-first cells) 
				  prevcell overhang)
		  (loop (node-list-rest cells) 
			(node-list-first cells))))))))

;; (define ($debug-pr-overhang$ overhang)
;;   (make sequence
;;     (literal "(")
;;     (let loop ((nl overhang))
;;       (if (null? nl)
;; 	  (empty-sosofo)
;; 	  (make sequence
;; 	    (literal (number->string (car nl)) " ")
;; 	    (loop (cdr nl)))))
;;     (literal ")")))

(define ($process-cell$ entry preventry overhang)
  (let* (
;;	 (celldebug (debug (string-append 
;;			  (number->string (cell-column-number entry overhang))
;;			  " "
;;			  (data entry))))
	 (colnum (cell-column-number entry overhang))
	 (lastcellcolumn (if (node-list-empty? preventry)
			     0
			     (- (+ (cell-column-number preventry overhang)
				   (hspan preventry))
				1)))
	 (lastcolnum (if (> lastcellcolumn 0)
			 (overhang-skip overhang lastcellcolumn)
			 0))
	 (htmlgi (if (have-ancestor? "TBODY" entry)
		     "TD"
		     "TH")))
    (make sequence
      ;; This is a little bit complicated.  We want to output empty cells
      ;; to skip over missing data.  We start count at the column number
      ;; arrived at by adding 1 to the column number of the previous entry
      ;; and skipping over any MOREROWS overhanging entrys.  Then for each
      ;; iteration, we add 1 and skip over any overhanging entrys.
      (let loop ((count (overhang-skip overhang (+ lastcolnum 1))))
	(if (>= count colnum)
	    (empty-sosofo)
	    (make element gi: htmlgi
		  (make entity-ref name: "nbsp")
;;		  (literal (number->string lastcellcolumn) ", ")
;;		  (literal (number->string lastcolnum) ", ")
;;		  (literal (number->string (hspan preventry)) ", ")
;;		  (literal (number->string colnum ", "))
;;		  ($debug-pr-overhang$ overhang)
		  (loop (overhang-skip overhang (+ count 1))))))

      (make element gi: htmlgi
	    attributes: (append
			 (if (> (hspan entry) 1)
			     (list (list "COLSPAN" (number->string (hspan entry))))
			     '())
			 (if (> (vspan entry) 1)
			     (list (list "ROWSPAN" (number->string (vspan entry))))
			     '())
			 (if (equal? (cell-colwidth entry colnum) "")
			     '()
			     (list (list "WIDTH" (cell-colwidth entry colnum))))
			 (list (list "ALIGN" (cell-align entry colnum)))
			 (list (list "VALIGN" (cell-valign entry colnum))))
	    ($process-cell-contents$ entry colnum)))))

(define ($process-cell-contents$ entry colnum)
  (let ((contents (node-list-filter-out-pis (children entry))))
    (if (node-list-empty? contents)
	(empty-sosofo)
	(if (not (gi (node-list-first contents)))
	    ($process-cell-pcdata-contents$ entry colnum)
	    ($process-cell-block-contents$ entry colnum)))))

(define ($process-cell-pcdata-contents$ entry colnum)
  (process-node-list (children entry)))

(define ($process-cell-block-contents$ entry colnum)
  ;; NOTE: this can't be done with element construction rules because
  ;; it is necessary to know the colnum in order to get alignment
  ;; correct and colnum is calculated as part of a forward pass through
  ;; the table.  A different algorithm, that allowed each entry to
  ;; calculate its own colnum, was tried, but the result was unusably
  ;; slow in Jade because every cell was required to calculate the 
  ;; colnum of the previous entry in the row and all of the elements
  ;; in the previous row (to get vspans right).  So every entry 
  ;; recalculated every preceding entry.  Recursively.
  ;; 
  ;; I can envision a solution that optimizes the case where there
  ;; are not many vspans, but it would be some effort to implement, so
  ;; I'm content to leave the current solution unless it can be demonstrated
  ;; that it's flawed in some way.
  ;;
  (let* ((row    (ancestor "ROW" entry))
	 (body   (parent row))
	 (align     (cell-align entry colnum)))
    (let loop ((nl (node-list-filter-out-pis (children entry))))
      (if (node-list-empty? nl)
	  (empty-sosofo)
	  (make sequence
	    (let* ((procpair (assoc (gi (node-list-first nl)) 
				    %cell-block-element-list%))
		   (procfunc (if procpair
				 (cdr procpair)
				 #f)))
	      (if procfunc
		  (procfunc (node-list-first nl) colnum align (gi body))
		  ($process-cell-para-contents$ 
		   (node-list-first nl) colnum align (gi body))))
	      (loop (node-list-rest nl)))))))

;; ----------------------------------------------------------------------

;; This list provides the mapping that would ordinarily be supplied
;; by the element construction rules.  If the block element "FOO"
;; occurs inside a table cell, use the $process-foo$ function to
;; process it's _contents_

(define %cell-block-element-list%
  `(;; ("FOO" . ,$process-foo$)
    ("PARA" . ,$process-cell-para-contents$)
    ))

(define ($process-cell-para-contents$ para colnum align body)
  (make element gi: "P"
      (process-node-list (children para))))

;; EOF